package cn.com.duiba.goods.center.biz.service.item.impl;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsCouponDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.tool.ItemKeyUtils;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.dao.item.CouponBatchDao;
import cn.com.duiba.goods.center.biz.dao.item.CouponDao;
import cn.com.duiba.goods.center.biz.entity.CouponBatchEntity;
import cn.com.duiba.goods.center.biz.entity.CouponEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.item.CouponFlowService;
import cn.com.duiba.goods.center.biz.service.item.ItemCouponGoodsService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * 商品GTYPE 为DUIBA和APP类的优惠券相关服务
 */
@Service
public class ItemCouponGoodsServiceImpl implements ItemCouponGoodsService {

	@Autowired
	private GoodsCouponBO goodsCouponBO;
	@Autowired
	private GoodsBatchService goodsBatchService;
	@Autowired
	private CouponFlowService couponFlowService;
	@Autowired
	private CouponDao couponDao;
	@Autowired
	private CouponBatchDao couponBatchDao;

	/**
	 * 发一个券
	 */
	@Override
	public GoodsCouponDto takeCoupon(ItemKeyDto itemKeyDto, Long consumerId, String bizNum) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			// 兼容老的券码发放逻辑
			return takeOldCoupon(itemKeyDto);
		} else {
			// 新版商品中心优惠券
			GoodsTypeEnum gtype = null;
			Long gid = null;
			if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
				gtype = GoodsTypeEnum.DUIBA;
				gid = itemKeyDto.getItemDto().getId();
			} else if (itemKeyDto.isSelfAppItemMode()) {
				gtype = GoodsTypeEnum.APP;
				gid = itemKeyDto.getAppItemDto().getId();
			}
			return takeGoodsCoupon(gtype, gid, consumerId, bizNum);
		}
	}

	/**
	 * 使用一个券
	 */
	@Override
	public Boolean completeCoupon(ItemKeyDto itemKeyDto, Long couponId, long orderId) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			// 兼容老的券码逻辑
			couponFlowService.completeCoupon(couponId, orderId);
		} else {
			// 新版商品中心优惠券
			goodsCouponBO.completeCoupon(couponId, orderId);
		}
		return true;
	}

	/**
	 * 回滚一个券
	 */
	@Override
	public Boolean rollbackCoupon(ItemKeyDto itemKeyDto, Long couponId, String bizNum) {
		boolean ret = true;
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			// 兼容老的券码逻辑
			couponFlowService.returnOneCoupon(couponId);
		} else {
			// 发新版商品中心优惠券
			ret = goodsCouponBO.rollbackCoupon(couponId, bizNum);
		}
		return ret;
	}

	/**
	 * 查询一个券
	 */
	@Override
	public GoodsCouponDto findCoupon(ItemKeyDto itemKeyDto, Long couponId) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			// 兼容老的券码逻辑
			CouponEntity ce = couponDao.find(couponId);
			return convert(itemKeyDto, ce);
		} else {
			// 新版商品中心优惠券
			GoodsCouponEntity entity = goodsCouponBO.findCoupon(couponId);
			return BeanUtils.copy(entity, GoodsCouponDto.class);
		}
	}
	
	/**
	 * 查询当前使用中的批次
	 */
	@Override
	public GoodsBatchDto findCurrentBatch(ItemKeyDto itemKeyDto) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			CouponBatchEntity entity = couponBatchDao.find(getOldBatchId(itemKeyDto));
			return convertBatch(itemKeyDto, entity);
		} else {
			GoodsBatchEntity entity = goodsCouponBO.findUsingBatch(getGtype(itemKeyDto), getGid(itemKeyDto));
			return BeanUtils.copy(entity, GoodsBatchDto.class);
		}
	}

	/**
	 * 查询批次
	 */
	@Override
	public GoodsBatchDto findCouponBatch(ItemKeyDto itemKeyDto, Long batchId) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			CouponBatchEntity entity = couponBatchDao.find(batchId);
			return convertBatch(itemKeyDto, entity);
		} else {
			GoodsBatchEntity entity = goodsCouponBO.findBatch(batchId);
			return BeanUtils.copy(entity, GoodsBatchDto.class);
		}
	}
	
	/**
	 * 获取老的商品的批次ID
	 * @param itemKey
	 * @return
	 */
	private Long getOldBatchId(ItemKeyDto itemKey) {
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			return itemKey.getItemDto().getBatchId();
		}
		if (itemKey.isSelfAppItemMode()) {
			return itemKey.getAppItemDto().getBatchId();
		}
		throw new RuntimeGoodsException(ErrorCode.E0202008);
	}
	
	/**
	 * ItemKey获取商品Gtype
	 * @param itemKey
	 * @return
	 */
	private GoodsTypeEnum getGtype(ItemKeyDto itemKey) {
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			return GoodsTypeEnum.DUIBA;
		}
		if (itemKey.isSelfAppItemMode()) {
			return GoodsTypeEnum.APP;
		}
		throw new RuntimeGoodsException(ErrorCode.E0202008);
	}
	
	/**
	 * ItemKey获取商品Gtype
	 * @param itemKey
	 * @return
	 */
	private Long getGid(ItemKeyDto itemKey) {
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			return itemKey.getItemDto().getId();
		}
		if (itemKey.isSelfAppItemMode()) {
			return itemKey.getAppItemDto().getId();
		}
		throw new RuntimeGoodsException(ErrorCode.E0202008);
	}

	/**
	 * 老的单表发券逻辑
	 */
	private GoodsCouponDto takeOldCoupon(ItemKeyDto itemKeyDto) {
		CouponEntity ce = couponFlowService.getOneCoupon(itemKeyDto);
		return convert(itemKeyDto, ce);
	}

	/**
	 * 新版发券逻辑
	 */
	private GoodsCouponDto takeGoodsCoupon(GoodsTypeEnum gtype, Long gid, Long consumerId, String bizNum) {
		GoodsCouponEntity entity = goodsCouponBO.takeCoupon(gtype, gid, consumerId, bizNum);
		if (entity == null) {
			throw new RuntimeGoodsException(ErrorCode.E0202006);
		}
		GoodsCouponDto dto = BeanUtils.copy(entity, GoodsCouponDto.class);
		GoodsBatchEntity batch = goodsBatchService.find(entity.getGoodsBatchId());
		dto.setOverDue(batch.getEndDay());
		return dto;
	}
	
	/**
	 * 是否是链接券
	 * 
	 * @param itemKeyDto
	 * @return
	 */
	private Integer getCouponType(ItemKeyDto itemKeyDto) {
		Integer linkType = null;
		if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
			linkType = itemKeyDto.getItemDto().getSubType();
		} else if (itemKeyDto.isSelfAppItemMode()) {
			linkType = itemKeyDto.getAppItemDto().getSubType();
		}
		return linkType;
	}
	
	/**
	 * 老的优惠券封装到新的优惠券
	 * 
	 * @param ce
	 * @return
	 */
	private GoodsCouponDto convert(ItemKeyDto itemKey, CouponEntity ce) {
		if (ce == null) {
			return null;
		}
		GoodsCouponDto dto = new GoodsCouponDto();
		Long gid = ce.getItemId() != null ? ce.getItemId() : ce.getAppItemId();
		Integer gtype = ce.getItemId() != null ? GoodsTypeEnum.DUIBA.getGtype() : GoodsTypeEnum.APP.getGtype();
		dto.setGid(gid);
		dto.setGtype(gtype);
		dto.setGmtCreate(ce.getGmtCreate());
		dto.setGmtModified(ce.getGmtModified());
		dto.setBatchId(ce.getBatchId());
		dto.setGoodsCouponId(ce.getId());
		dto.setGoodsBatchId(ce.getBatchId());
		if (ItemDO.SubTypeLink == getCouponType(itemKey)) {
			dto.setLink(ce.getCode());
			dto.setPassword(ce.getCode());
		} else {
			dto.setCode(ce.getCode());
			dto.setPassword(ce.getPassword());
		}
		return dto;
	}
	
	private GoodsBatchDto convertBatch(ItemKeyDto itemKey, CouponBatchEntity cbe) {
		if (cbe == null) {
			return null;
		}
		GoodsBatchDto dto = new GoodsBatchDto();
		dto.setId(cbe.getId());
		dto.setGid(getGid(itemKey));
		dto.setGtype(getGtype(itemKey).getGtype());
		dto.setBatchType(getCouponType(itemKey));
		dto.setStartDay(cbe.getValidStartDate());
		dto.setEndDay(cbe.getValidStartDate());
		dto.setGmtCreate(cbe.getGmtCreate());
		dto.setGmtModified(cbe.getGmtModified());
		return dto;
	}

}
