package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.dao.item.ItemAppSpecifyDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemStockConsumeDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemStockSpecifyConsumeDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockConsumeDetailDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockPointDao;
import cn.com.duiba.goods.center.biz.entity.ItemAppSpecifyEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.entity.ItemStockConsumeEntity;
import cn.com.duiba.goods.center.biz.entity.ItemStockSpecifyConsumeEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockConsumeDetailEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockPointEntity;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.EverydayLimitService;
import cn.com.duiba.goods.center.biz.service.item.ItemKeyStockService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.biz.service.stock.MemStockService;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.service.domain.dataobject.PreStockConsumeDetailDO;
import cn.com.duiba.service.item.domain.dataobject.ItemStockConsumeDO;
import cn.com.duiba.service.item.domain.dataobject.ItemStockSpecifyConsumeDO;

@Service
public class ItemKeyStockServiceImpl implements ItemKeyStockService {

	private static Logger LOGGER = LoggerFactory.getLogger(ItemKeyStockServiceImpl.class);

	@Autowired
	private AppItemService appItemService;
	@Autowired
	private ItemService itemService;
	@Autowired
	private MemStockService memStockService;
	@Autowired
	private PreStockService preStockService;
	@Autowired
	private ItemStockConsumeDao itemStockConsumeDao;
	@Autowired
	private ItemAppSpecifyDao itemAppSpecifyDao;
	@Autowired
	private ItemStockSpecifyConsumeDao itemStockSpecifyConsumeDao;
	@Autowired
	private PreStockDao preStockDao;
	@Autowired
	private PreStockPointDao preStockPointDao;
	@Autowired
	private PreStockConsumeDetailDao preStockConsumeDetailDao;
	@Autowired
	private EverydayLimitService everydayLimitService;

	@Override
	public Boolean decrStock(ItemKeyDto itemKeyDto, String bizId, String bizSource) throws GoodsException {
		if (!itemKeyDto.getItemDtoType().equals(ItemDO.TypeObject) && !itemKeyDto.getItemDtoType().equals(ItemDO.TypeVirtual)) {
			LOGGER.info("decrStock: type=" + itemKeyDto.getItemDtoType() + " 不支持该商品类型扣库存");
			throw new GoodsException("0", "不支持该类型：" + itemKeyDto.getItemDtoType());
		}
		Long relationId = null;
		String relationType = null;
		if (itemKeyDto.isSelfAppItemMode()) {
			AppItemDto appItemDto = itemKeyDto.getAppItemDto();
			if (appItemDto.getRemaining() == null || appItemDto.getRemaining() <= 0) {
				LOGGER.info("decrStock: type=" + itemKeyDto.getItemDtoType() + " appItemId=" + appItemDto.getId() + " 库存不足");
				throw new GoodsException("0", "库存不足");
			}
			boolean ret = appItemService.decrStock(appItemDto.getAppId(), appItemDto.getId());
			if (!ret) {
				LOGGER.info("decrStock: type=" + itemKeyDto.getItemDtoType() + " appItemId=" + appItemDto.getId() + " 扣库存失败:库存不足");
				throw new GoodsException("0", "库存不足");
			}
			relationId = appItemDto.getId();
			relationType = ItemStockConsumeDO.TYPE_DEV_ITEM;
		} else if (itemKeyDto.isItemMode() || itemKeyDto.isDuibaAppItemMode()) {
			ItemDto itemDto = itemKeyDto.getItemDto();
			if (itemDto.getRemaining() == null || itemDto.getRemaining() <= 0) {
				LOGGER.info("decrStock: type=" + itemKeyDto.getItemDtoType() + " itemId=" + itemDto.getId() + " 库存不足");
				throw new GoodsException("0", "库存不足");
			}
			boolean ret = itemService.decrStock(itemDto.getId());
			if (!ret) {
				LOGGER.info("decrStock: type=" + itemKeyDto.getItemDtoType() + " itemId=" + itemDto.getId() + " 扣库存失败:库存不足");
				throw new GoodsException("0", "库存不足");
			}
			relationId = itemDto.getId();
			relationType = ItemStockConsumeDO.TYPE_DUIBA;
		}

		Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
		ItemStockConsumeEntity consume = new ItemStockConsumeEntity();
		consume.setBizId(bizId);
		consume.setBizSource(bizSource);
		consume.setRelationId(relationId);
		consume.setRelationType(relationType);
		consume.setAppId(itemKeyDto.getAppId());
		consume.setAction(ItemStockConsumeDO.ACTION_PAY);
		consume.setQuantity(1L);
		consume.setGmtCreate(now);
		consume.setGmtModified(now);
		itemStockConsumeDao.insert(consume);
		return true;
	}

	@Override
	public Boolean decrAppSpecifyStock(Long appId, ItemEntity itemEntity, String bizId, String bizSource) throws GoodsException {
		if (!ItemDO.TypeObject.equals(itemEntity.getType()) && !ItemDO.TypeCoupon.equals(itemEntity.getType())) {
			LOGGER.info("decrAppSpecifyStock: type=" + itemEntity.getType() + " 不支持该商品类型扣APP定向库存");
			throw new GoodsException("0", "不支持该类型：" + itemEntity.getType());
		}
		ItemAppSpecifyEntity specify = itemAppSpecifyDao.findByItemIdAndAppId(itemEntity.getId(), appId);
		if (null == specify) {
			LOGGER.info("decrAppSpecifyStock: itemId=" + itemEntity.getId() + " 商品无定向库存");
			return false;
		}
		if (specify.getRemaining() <= 0) {
			LOGGER.info("decrAppSpecifyStock: type=" + itemEntity.getType() + " itemId=" + itemEntity.getId() + " 定向库存不足");
			throw new GoodsException("0", "定向库存不足");
		}
		int row = itemAppSpecifyDao.reduceRemaining(itemEntity.getId(), appId);
		if (row < 1) {
			LOGGER.info("decrAppSpecifyStock: type=" + itemEntity.getType() + " itemId=" + itemEntity.getId() + " 扣库存失败:定向库存不足");
			throw new GoodsException("0", "定向库存不足");
		}

		Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
		ItemStockSpecifyConsumeEntity consume = new ItemStockSpecifyConsumeEntity();
		consume.setBizId(bizId);
		consume.setBizSource(bizSource);
		consume.setRelationId(itemEntity.getId());
		consume.setRelationType(ItemStockSpecifyConsumeDO.TYPE_DUIBA);
		consume.setAppId(appId);
		consume.setAction(ItemStockConsumeDO.ACTION_PAY);
		consume.setQuantity(1L);
		consume.setGmtCreate(now);
		consume.setGmtModified(now);
		itemStockSpecifyConsumeDao.insert(consume);
		return true;
	}

	@Override
	public Boolean decrPreStock(Long appId, Long itemId, PreStockPointEntity pointStock, String bizId, String bizSource) throws GoodsException {
		PreStockEntity preStock = preStockDao.findByPointId(pointStock.getId());
		if (null == preStock) {
			LOGGER.error("decrPreStock: itemId=" + itemId + " 商品无预分配库存");
			throw new GoodsException("0", "商品无预分配库存");
		}
		if (preStock.getStock() < 1L) {
			LOGGER.info("decrPreStock: itemId=" + itemId + " 预分配库存不足");
			throw new GoodsException("0", "预分配库存不足");
		}
		Long preStockId = preStock.getId();

		if (null != pointStock && pointStock.getLimitEverydayQuantity() != null) {
			boolean limit = memStockService.everydayStock(pointStock.getId().toString(), pointStock.getLimitEverydayQuantity());
			if (!limit) {
				throw new GoodsException("0", "预分配每日限量库存不足");
			}
			// Integer sales =
			// preStockConsumeDetailDao.countSalesTimeByLock(pointId,DateUtils.getDayDate(new
			// Date()));
			// if(sales >= pointStock.getLimitEverydayQuantity()){
			// throw new GoodsException("0", "预分配每日限量库存不足");
			// }
		}

		boolean success = preStockDao.reduceQuantity(preStock.getId(), 1L);
		if (!success) {
			LOGGER.info("decrPreStock: itemId=" + itemId + " 预分配库存不足");
			throw new GoodsException("0", "预分配库存不足");
		}

		PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
		detail.setBizId(bizId);
		detail.setAction(PreStockConsumeDetailEntity.ActionPay);
		detail.setPointId(pointStock.getId());
		detail.setStockId(preStockId);
		detail.setQuantity(1L);
		detail.setAppId(appId);
		Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
		detail.setGmtCreate(now);
		detail.setGmtModified(now);
		preStockConsumeDetailDao.insert(detail);
		preStockService.deletePreStockCounter(pointStock.getId());
		return true;
	}

	@Override
	public Boolean rollbackStock(String bizId, String bizSource) {
		try {
			ItemStockConsumeEntity consume = itemStockConsumeDao.findByBizIdAndSource(bizId, bizSource);
			if (null == consume) {
				LOGGER.info("rollbackStock=" + bizId + " 返还库存无消费记录");
				return false;
			}

			if (consume.getRelationType().equals(ItemStockConsumeDO.TYPE_DEV_ITEM)) {
				boolean back = appItemService.incrStock(consume.getRelationId());
				if (!back) {
					LOGGER.info("rollbackStock:appItemId= " + consume.getRelationId() + " 返还库存失败");
					return false;
				}
			} else if (consume.getRelationType().equals(ItemStockConsumeDO.TYPE_DUIBA)) {
				boolean back = itemService.incrStock(consume.getRelationId());
				if (!back) {
					LOGGER.info("rollbackStock:itemId= " + consume.getRelationId() + " 返还库存失败");
					return false;
				}
			}
			Date now = new Date(System.currentTimeMillis() / 1000 * 1000);

			ItemStockConsumeEntity detail = new ItemStockConsumeEntity();
			detail.setBizId(consume.getBizId());
			detail.setBizSource(consume.getBizSource());
			detail.setRelationId(consume.getRelationId());
			detail.setRelationType(consume.getRelationType());
			detail.setAppId(consume.getAppId());
			detail.setAction(ItemStockConsumeDO.ACTION_BACK);
			detail.setQuantity(1L);
			detail.setGmtCreate(now);
			detail.setGmtModified(now);
			itemStockConsumeDao.insert(detail);
			return true;
		} catch (Exception e) {
			LOGGER.error("rollbackStock:bizId=" + bizId + " 返还库存异常", e);
		}
		return false;
	}

	@Override
	public Boolean rollbackAppSpecifyStock(String bizId, String bizSource) {
		try {
			ItemStockSpecifyConsumeEntity consume = itemStockSpecifyConsumeDao.findByBizIdAndSource(bizId, bizSource);
			if (null == consume) {
				LOGGER.info("rollbackAppSpecifyStock:bizId=" + bizId + " 返还定向库存无消费记录");
				return false;
			}

			ItemAppSpecifyEntity specify = itemAppSpecifyDao.findByItemIdAndAppId(consume.getRelationId(), consume.getAppId());
			if (null == specify) {
				LOGGER.info("rollbackAppSpecifyStock:bizId=" + bizId + " 返还定向库存无定向");
				return false;
			}

			int row = itemAppSpecifyDao.increaseRemaining(specify.getItemId(), specify.getAppId());
			if (row < 1) {
				LOGGER.info("rollbackAppSpecifyStock:bizId=" + bizId + " 返还定向库存失败");
				return false;
			}

			Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
			ItemStockSpecifyConsumeEntity detail = new ItemStockSpecifyConsumeEntity();
			detail.setBizId(bizId);
			detail.setBizSource(bizSource);
			detail.setRelationId(consume.getRelationId());
			detail.setRelationType(consume.getRelationType());
			detail.setAppId(consume.getAppId());
			detail.setAction(ItemStockConsumeDO.ACTION_BACK);
			detail.setQuantity(consume.getQuantity());
			detail.setGmtCreate(now);
			detail.setGmtModified(now);
			itemStockSpecifyConsumeDao.insert(detail);
			return true;
		} catch (Exception e) {
			LOGGER.error("rollbackAppSpecifyStock:bizId=" + bizId + " 返还定向库存异常", e);
		}
		return false;
	}

	@Override
	public Boolean rollbackPreStock(String bizId) {
		try {
			PreStockConsumeDetailEntity consume = preStockConsumeDetailDao.findByBizIdAndPayAction(bizId);
			if (null == consume) {
				LOGGER.info("rollbackPreStock:bizId=" + bizId + " 返还预分配库存无消费记录");
				return false;
			}

			boolean success = preStockDao.addQuantity(consume.getStockId(), 1L);
			if (!success) {
				LOGGER.info("rollbackPreStock:bizId=" + bizId + " 返还预分配库存失败");
				return false;
			}

			Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
			PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
			detail.setBizId(bizId);
			detail.setAction(PreStockConsumeDetailDO.ActionBack);
			detail.setPointId(consume.getPointId());
			detail.setStockId(consume.getStockId());
			detail.setQuantity(consume.getQuantity());
			detail.setAppId(consume.getAppId());
			detail.setGmtCreate(now);
			detail.setGmtModified(now);

			preStockConsumeDetailDao.insert(detail);
			preStockService.deletePreStockCounter(consume.getPointId());
			return true;
		} catch (Exception e) {
			LOGGER.error("rollbackPreStock:bizId=" + bizId + " 返还预分配库存异常", e);
		}
		return false;
	}

	@Override
	public Boolean decrEverydayStock(ItemKeyDto itemKeyDto) {
		if (!everydayLimitService.isEverydayLimit(itemKeyDto)) {
			return false;
		}
		return everydayLimitService.decrEeverydayStock(itemKeyDto);
	}

	@Override
	public Boolean rollbackEverydayStock(ItemKeyDto itemKeyDto) {
		try {
			return everydayLimitService.incrEverydayStock(itemKeyDto);
		} catch (Exception e) {
			LOGGER.error("rollbackEverydayStock:返还每日限量库存异常", e);
		}
		return false;
	}
}
