package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.dao.item.PreStockConsumeDetailDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockManualChangeDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockPointDao;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockConsumeDetailEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockManualChangeEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockPointEntity;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.biz.service.stock.MemStockService;
import cn.com.duiba.goods.center.biz.util.CacheConstants;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.wolf.cache.CacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;

@Service
public class PreStockServiceImpl implements PreStockService {

	private static final Logger log = LoggerFactory.getLogger(PreStockServiceImpl.class);
	@Autowired
	private PreStockPointDao preStockPointDao;
	@Autowired
	private PreStockDao preStockDao;
	@Autowired
	private CacheClient cacheClient;
	@Autowired
	private PreStockConsumeDetailDao preStockConsumeDetailDao;
	@Autowired
	private PreStockManualChangeDao preStockManualChangeDao;
	@Autowired
	private MemStockService memStockService;

	@Override
	public void consumeStock(String bizId, Long appId, Long pointId) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException("0", "预分配库存不存在");
		}
		if (stock.getStock() < 1L) {
			throw new GoodsException("0", "预分配库存不足");
		}
		Long preStockId = stock.getId();

		PreStockPointEntity pointStock = preStockPointDao.find(pointId);

		if (null != pointStock && pointStock.getLimitEverydayQuantity() != null) {
			boolean limit = memStockService.everydayStock(pointStock.getId().toString(), pointStock.getLimitEverydayQuantity());
			if (!limit) {
				throw new GoodsException("0", "预分配每日限量库存不足");
			}
			// Integer sales =
			// preStockConsumeDetailDao.countSalesTimeByLock(pointId,DateUtils.getDayDate(new
			// Date()));
			// if(sales >= pointStock.getLimitEverydayQuantity()){
			// throw new GoodsException("0", "预分配每日限量库存不足");
			// }
		}

		boolean success = preStockDao.reduceQuantity(stock.getId(), 1L);
		if (!success) {
			throw new GoodsException("0", "扣预分配库存失败");
		}

		PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
		detail.setBizId(bizId);
		detail.setAction(PreStockConsumeDetailEntity.ActionPay);
		detail.setPointId(pointId);
		detail.setStockId(preStockId);
		detail.setQuantity(1L);
		detail.setAppId(appId);
		Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
		detail.setGmtCreate(now);
		detail.setGmtModified(now);
		preStockConsumeDetailDao.insert(detail);

		try {
			deletePreStockCounter(pointId);
		} catch (Exception e) {
			log.error("className: " + this.getClass().getName() + " method: deletePreStockCounter error", e);
		}
	}

	@Override
	public void paybackStock(String bizId) throws GoodsException {
		PreStockConsumeDetailEntity consume = preStockConsumeDetailDao.findByBizIdAndPayAction(bizId);
		if (consume == null) {
			return;
		}

		boolean success = preStockDao.addQuantity(consume.getStockId(), 1L);
		if (!success) {
			throw new GoodsException("0", "返还预分配库存失败");
		}

		PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
		detail.setBizId(bizId);
		detail.setAction(PreStockConsumeDetailEntity.ActionBack);
		detail.setPointId(consume.getPointId());
		detail.setStockId(consume.getStockId());
		detail.setQuantity(consume.getQuantity());
		detail.setAppId(consume.getAppId());
		Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
		detail.setGmtCreate(now);
		detail.setGmtModified(now);
		preStockConsumeDetailDao.insert(detail);

		try {
			deletePreStockCounter(consume.getPointId());
		} catch (Exception e) {
			log.error("className: " + this.getClass().getName() + " method: deletePreStockCounter error", e);
		}
	}

	/**
	 * 查询库存预分配的库存 为null则最兑换项本身的
	 */
	@Override
	public Long getPreStock(ItemEntity itemEntity, Long appId) {
		PreStockPointEntity point = getPointStock(itemEntity, appId);
		if (null != point) {
			PreStockEntity stock = preStockDao.findByPointId(point.getId());
			if (null != stock) {
				return stock.getStock();
			}
		}
		return null;
	}

	@Override
	public Integer getPreStockQuantitySales(ItemKeyDto itemKey) {
		if (itemKey == null || itemKey.getItemDto() == null) {
			return null;
		}
		ItemEntity itemEntity = BeanUtils.copy(itemKey.getItemDto(), ItemEntity.class);
		PreStockPointEntity point = getPointStock(itemEntity, itemKey.getAppId());
		if (null == point || null == point.getLimitEverydayQuantity()) {
			return null;
		}
		try {
			final String key = getKey(point.getId());
			Integer salesCount = (Integer) cacheClient.get(key);
			if (null == salesCount) {
				Integer countNum = preStockConsumeDetailDao.countSalesTime(point.getId(), DateUtils.getDayDate(new Date()));
				int exp = Math.min(DateUtils.getToTomorrowSeconds(), 3600);
				cacheClient.set(key, countNum, exp, TimeUnit.SECONDS);
				salesCount = countNum;
			}
			Integer limit = point.getLimitEverydayQuantity();
			Integer remaining = 0;
			if (salesCount < limit) {
				remaining = limit - salesCount;
			}
			return remaining;
		} catch (Exception e) {
			log.error("className: " + this.getClass().getName() + " method: getPreStockQuantitySales 获取限量剩余库存异常", e);
			return 0;
		}
	}

	/**
	 * 限量库存失效
	 * 
	 * @param pointId
	 * @return
	 */
	public Boolean deletePreStockCounter(Long pointId) {
		try {
			String key = getKey(pointId);
			cacheClient.remove(key);
			return true;
		} catch (Exception e) {
			log.error("className: " + this.getClass().getName() + " method: deletePreStockCounter", e);
		}
		return false;
	}

	/**
	 * 判断是否是兑吧库存预分配
	 * 
	 * @return
	 */
	public PreStockPointEntity getPointStock(ItemEntity itemEntity, Long appId) {
		if (null != itemEntity && itemEntity.isOpTypeItem(ItemDto.OpTypePreStockSwith)) {
			// 先查是否定向给某个APP
			PreStockPointEntity point = preStockPointDao.findAppIdAndItemId(appId, itemEntity.getId());
			if (null != point) {
				return point;
			} else {
				// 查共享库
				PreStockPointEntity pointShare = preStockPointDao.findItemIdAndNullApp(itemEntity.getId());
				if (null != pointShare) {
					return pointShare;
				}
			}
		}
		return null;
	}

	private String getKey(Long pointId) {
		return CacheConstants.MS_PRE_STOCK_QUANTITY + "-" + pointId;
	}

	@Override
	public long newStock(Long pointId, Long stockNum) throws GoodsException {
		PreStockEntity old = preStockDao.findByPointId(pointId);
		if (old != null) {
			return old.getId();
		}
		PreStockEntity stock = new PreStockEntity();
		stock.setPointId(pointId);
		stock.setStock(stockNum);
		stock = preStockDao.newStock(stock);
		return stock.getId();
	}

	@Override
	public boolean addStockQuantity(Long bizId, Long pointId, Integer quantity) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException("0", "预分配库存不存在");
		}

		preStockDao.addQuantity(stock.getId(), quantity);

		PreStockManualChangeEntity instance = new PreStockManualChangeEntity();
		instance.setBeforeStock(stock.getStock());
		instance.setAfterStock(stock.getStock() + quantity);
		instance.setChangeKind(PreStockManualChangeEntity.ChangeKindAdd);
		instance.setChangeQuantity(quantity);
		instance.setStockId(stock.getId());
		instance.setBizId(bizId);
		preStockManualChangeDao.insert(instance);
		return true;
	}

	@Override
	public boolean reduceStockQuantity(Long bizId, Long pointId, Integer quantity) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException("0", "预分配库存不存在");
		}

		if (stock.getStock() < quantity) {
			throw new GoodsException("0", "预分配库存不足");
		}

		preStockDao.reduceQuantity(stock.getId(), quantity);

		PreStockManualChangeEntity instance = new PreStockManualChangeEntity();
		instance.setBeforeStock(stock.getStock());
		instance.setAfterStock(stock.getStock() - quantity);
		instance.setChangeKind(PreStockManualChangeEntity.ChangekIndSub);
		instance.setChangeQuantity(quantity);
		instance.setStockId(stock.getId());
		instance.setBizId(bizId);
		preStockManualChangeDao.insert(instance);
		return true;
	}

	@Override
	public boolean reduceStockAll(Long bizId, Long pointId) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException("0", "预分配库存不存在");
		}

		preStockDao.reduceQuantity(stock.getId(), stock.getStock());

		PreStockManualChangeEntity instance = new PreStockManualChangeEntity();
		instance.setBeforeStock(stock.getStock());
		instance.setAfterStock(stock.getStock() - stock.getStock());
		instance.setChangeKind(PreStockManualChangeEntity.ChangekIndSub);
		instance.setChangeQuantity(Integer.parseInt(stock.getStock() + ""));
		instance.setStockId(stock.getId());
		instance.setBizId(bizId);
		preStockManualChangeDao.insert(instance);
		return true;
	}

	private void deletePreStockCounter_only(Long pointId) {
		if (null == pointId) {
			return;
		}
		String key = getKey(pointId);
		cacheClient.remove(key);
	}

	@Override
	public void submitPreStock(List<PreStockPointEntity> insertPointList, List<PreStockPointEntity> updatePointList) throws GoodsException {
		if (!insertPointList.isEmpty()) {
			for (PreStockPointEntity stock : insertPointList) {
				if (null == stock.getLimitCount()) {
					stock.setLimitCount(0);
				}
				// 库存中心插入库存
				preStockPointDao.insert(stock);
				newStock(stock.getId(), Long.parseLong(stock.getLimitCount() + ""));
			}
		}
		if (!updatePointList.isEmpty()) {
			for (PreStockPointEntity point : updatePointList) {
				preStockPointDao.udpate(point);
				// 库存中心更新库存
				if (null == point.getLimitCount()) {
					this.reduceStockAll(point.getId(), point.getId());
				} else {
					if (point.getLimitCount() < 0) {
						this.reduceStockQuantity(point.getId(), point.getId(), Math.abs(point.getLimitCount()));
					} else if (point.getLimitCount() > 0) {
						this.addStockQuantity(point.getId(), point.getId(), Math.abs(point.getLimitCount()));
					}
				}
				// 失效MEMCACHE
				this.deletePreStockCounter_only(point.getId());
			}
		}
	}
}
