package cn.com.duiba.goods.center.biz.bo.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.AddrLimitDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.bo.AddrLimitBO;
import cn.com.duiba.goods.center.biz.cache.IPCache;
import cn.com.duiba.goods.center.biz.service.item.AddrLimitService;
import cn.com.duiba.goods.center.biz.util.IpUtil;

/**
 * Created by xiaoxuda on 2016/11/24.
 */
@Service("addrLimitBO")
public class AddrLimitBOImpl implements AddrLimitBO {
    @Autowired
    private AddrLimitService addrLimitService;

    @Override
    public List<ItemKeyDto> getItemCanPassAddrLimit(List<ItemKeyDto> items, String ip) {
        if (null == items || items.isEmpty()) {
            return items;
        }

        String code = getAddrLimitCode(ip);
        Set<String> appItemAddrLimitKeys = forbidByAddrLimit(getAddrLimits(items, GoodsTypeEnum.APP), code);
        Set<String> itemAddrLimitsKeys = forbidByAddrLimit(getAddrLimits(items, GoodsTypeEnum.DUIBA), code);

        if (appItemAddrLimitKeys.isEmpty() && itemAddrLimitsKeys.isEmpty()) {
            return items;
        }

        itemAddrLimitsKeys.addAll(appItemAddrLimitKeys);
        ListIterator<ItemKeyDto> iter = items.listIterator();
        while (iter.hasNext()) {
            ItemKeyDto key = iter.next();
            if (itemAddrLimitsKeys.contains(generateForbidKey(key))) {
                iter.remove();
            }
        }
        return items;
    }
    
	@Override
	public Boolean ipValidation(ItemKeyDto key, String ip) {
		List<AddrLimitDto> addrLimitDOs = new ArrayList<>();
		if (key.isSelfAppItemMode()) {
			if (key.getAppItem().isOpTypeAppItem(ItemDto.OpTypeAddrLimit)) {
				addrLimitDOs = addrLimitService.findAddrLimitByAppItemId(key.getAppItem().getId());
			}
		} else if (key.isDuibaAppItemMode() || key.isItemMode()) {
			if (key.getItem().isOpTypeItem(ItemDto.OpTypeAddrLimit)) {
				addrLimitDOs = addrLimitService.findAddrLimitByItemId(key.getItem().getId());
			}
		}
		if (!addrLimitDOs.isEmpty()) {
			String code = getAddrLimitCode(ip);
			if (StringUtils.isBlank(code)) {
				return false;
			}
			return isValid(addrLimitDOs, code);
		}
		return true;
	}
    

    /**
     * 提取给定集合中符合特定商品类型且地域信息限制开关的地域商品的地域限制信息
     * 区分兑吧商品和开发者自有商品，兑吧商品不允许开发者设置地域限制，AddrLimitDto中appItemId和itemId只有一个不为空
     *
     * @param items
     * @param itemEnum
     * @return
     */
	public List<AddrLimitDto> getAddrLimits(List<ItemKeyDto> items, GoodsTypeEnum itemEnum) {
		List<Long> itemIds = new ArrayList<>();
		if (itemEnum.getGtype() == GoodsTypeEnum.APP.getGtype()) {
			for (ItemKeyDto item : items) {
				if (item.isSelfAppItemMode() && item.getAppItem().isOpTypeAppItem(ItemDto.OpTypeAddrLimit)) {
					itemIds.add(item.getAppItem().getId());
				}
			}
			return addrLimitService.findAddrLimitByAppItemIds(itemIds);
		}
		if (itemEnum.getGtype() == GoodsTypeEnum.DUIBA.getGtype()) {
			for (ItemKeyDto item : items) {
				if (item.getItem() != null && item.getItem().isOpTypeItem(ItemDto.OpTypeAddrLimit)) {
					itemIds.add(item.getItem().getId());
				}
			}
			return addrLimitService.findAddrLimitByItemIds(itemIds);
		}
		return new ArrayList<>();
	}

    /**
     * @param ip
     * @return ip在限制域内则返回地区编码，否则返回null
     */
    private String getAddrLimitCode(final String ip) {
        if (StringUtils.isBlank(ip)) {
            return null;
        }
        Long ipLong = IpUtil.convertIpLong(ip);
        if (null == ipLong) {
            return null;
        }
        //查询IP库
		IPCache library = addrLimitService.findCodeByIpLong(ipLong);
		if (null == library) {
			return null;
		}
        Long endIpNum = library.getEndIpNum();
        if (null != endIpNum && ipLong <= endIpNum) {
            if (StringUtils.isNotBlank(library.getCode() + "")) {
                return library.getCode() + "";
            }
        }
        return null;
    }


    /**
     * 生成key，用于禁止名单的集合
     *
     * @param appItemId
     * @param itemId
     * @return appItemId+_+itemId
     */
    private String generateForbidKey(Long appItemId, Long itemId) {
        String key = "";
        if (appItemId != null) {
            key += appItemId;
        }
        key += "_";
        if (itemId != null) {
            key += itemId;
        }
        return key;
    }

    private String generateForbidKey(ItemKeyDto dto) {
        if(dto.isSelfAppItemMode()){
            return generateForbidKey(dto.getAppItem().getId(),null);
        }else if(dto.isItemMode() || dto.isDuibaAppItemMode()){
            return generateForbidKey(null,dto.getItem().getId());
        }else{
            return generateForbidKey(null,null);
        }
    }

    /**
     * 过滤不能通过地域限制的商品
     *
     * @param addrLimits
     * @param code
     * @return
     */
    private Set<String> forbidByAddrLimit(List<AddrLimitDto> addrLimits, String code) {

        Map<String, List<AddrLimitDto>> map = new HashMap<>();
        for (AddrLimitDto limit : addrLimits) {
            String key = generateForbidKey(limit.getAppItemId(), limit.getItemId());
            List<AddrLimitDto> addrs = map.get(key);
            if (addrs == null) {
                addrs = new ArrayList<>();
            }
            addrs.add(limit);
            map.put(key, addrs);
        }

        Set<String> forbidIdSet = new HashSet<>();
        for (Map.Entry<String, List<AddrLimitDto>> entry : map.entrySet()) {
            if (!isValid(entry.getValue(), code)) {
                forbidIdSet.add(entry.getKey());
            }
        }
        return forbidIdSet;
    }

    /**
     * 判断是否在地域限制内可用
     *
     * @param addrLimitDtos
     * @param code
     * @return
     */
    public boolean isValid(List<AddrLimitDto> addrLimitDtos, String code) {

        if (StringUtils.isBlank(code)) {
            return false;
        }

        String provinceCode = code.substring(0, 2);//用户所在的省编码

        String cityCode = code.substring(2, 4);//用户所在的市编码

        //判断是否为直辖市,若为直辖市,则用最后两位区编码判断
        String zxCode = code.substring(4, 6);
        if (!"00".equals(zxCode)) {
            cityCode = zxCode;
        }

        boolean valid = false;//默认不可通过限制

        for (AddrLimitDto limit : addrLimitDtos) {

            String limitProviceCode = limit.getProCode().substring(0, 2);//当前兑换项限制的省编码

            String limitCityCode = limit.getProCode().substring(2, 4);//当前兑换项限制的市编码

            if (!"all".equals(limit.getCityCode())) {//如果当前兑换项只限制到省,则市的编码为‘all’

                limitCityCode = limit.getCityCode().substring(2, 4);//如果当前兑换项限制到市,则获取市的编码

                //判断是否为直辖市,若为直辖市,则用最后两位区判断
                String zxLimieCode = limit.getCityCode().substring(4, 6);
                if (!"00".equals(zxLimieCode)) {
                    limitCityCode = zxLimieCode;
                }

            }
            if (limit.getType().equals(AddrLimitDto.TYPE_WHITE_LIST)) {//白名单下
                if (provinceCode.equals(limitProviceCode)) {//省匹配
                    if (limitCityCode.equals(AddrLimitDto.ONLY_PROVINCE)) {//省级别白名单,省匹配则通过
                        valid = true;
                        break;
                    } else if (cityCode.equals(limitCityCode)) {//市级别白名单，用户省市信息也匹配时才通过。
                        valid = true;
                        break;
                    }
                }
            } else if (limit.getType().equals(AddrLimitDto.TYPE_BLACK_LIST)) {//黑名单下

                if (!provinceCode.equals(limitProviceCode)) {//省不匹配，可通过
                    valid = true;
                } else {
                    if (limitCityCode.equals(AddrLimitDto.ONLY_PROVINCE)) {//省级别黑名单,省匹配则不能通过，否则能通过
                        valid = false;
                        break;
                    } else {
                        if (cityCode.equals(AddrLimitDto.ONLY_PROVINCE)) {//用户只能获取到省的编码,则不能通过
                            valid = false;
                            break;
                        } else if (provinceCode.equals(limitProviceCode) && cityCode.equals(limitCityCode)) {
                            valid = false;
                            break;
                        } else {
                            valid = true;
                        }
                    }
                }
            }
        }
        return valid;
    }

}
