package cn.com.duiba.goods.center.biz.remoteservice.item;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.ElasticGiftsItemShowDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.item.RemoteDuibaItemGoodsService;
import cn.com.duiba.goods.center.biz.bo.item.ItemAutoRecommendBo;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.service.item.ItemGoodsService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * 兑吧商品接口实现类
 */
@Service
public class RemoteDuibaItemGoodsServiceImpl implements RemoteDuibaItemGoodsService {

	private static final Logger LOGGER = LoggerFactory.getLogger(RemoteDuibaItemGoodsServiceImpl.class);

	@Autowired
	private ItemService itemService;
	
	@Autowired
	private ItemGoodsService itemGoodsService;

	@Autowired
	private ItemAutoRecommendBo itemAutoRecommendBo;

	@Override
	public DubboResult<ItemDto> find(Long id) {
		try {
			if(id==null){
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			ItemEntity entity = itemService.find(id);
			ItemDto dto = BeanUtils.copy(entity, ItemDto.class);
			return DubboResult.successResult(dto);
		} catch (Exception e) {
			LOGGER.error("find:" + id, e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<List<ItemDto>> findByIds(List<Long> ids) {
		try {
			if (ids == null || ids.isEmpty()) {
				List<ItemDto> ret = Collections.emptyList();
				return DubboResult.successResult(ret);
			}
			List<ItemEntity> entitys = itemService.findByIds(ids);
			List<ItemDto> dtos = BeanUtils.copyList(entitys, ItemDto.class);
			return DubboResult.successResult(dtos);
		} catch (Exception e) {
			LOGGER.error("findByIds:", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Long> insert(ItemDto itemDto) {
		try {
			if (itemDto == null || itemDto.getId() != null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			ItemEntity entity = BeanUtils.copy(itemDto, ItemEntity.class);
			Long id = itemService.insert(entity);
			itemDto.setId(id);
			return DubboResult.successResult(id);
		} catch (Exception e) {
			LOGGER.error("insert:", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> update(ItemDto itemDto) {
		try {
			if (itemDto == null || itemDto.getId() == null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			ItemEntity entity = BeanUtils.copy(itemDto, ItemEntity.class);
			Boolean ret = itemService.update(entity);
			return DubboResult.successResult(ret);
		} catch (Exception e) {
			LOGGER.error("update:", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> delete(Long itemId) {
		try {
			if (itemId == null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			Boolean ret = itemService.delete(itemId);
			return DubboResult.successResult(ret);
		} catch (Exception e) {
			LOGGER.error("delete:", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> updateEnable(Long itemId, Boolean enable) {
		try {
			if (itemId == null || enable == null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			Boolean ret = itemService.updateEnable(itemId, enable);
			return DubboResult.successResult(ret);
		} catch (Exception e) {
			LOGGER.error("updateEnable:", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> appendStock(Long itemId, Long number) {
		try {
			if (itemId == null || number == null || number < 0) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			Boolean ret = itemService.appendStock(itemId, number);
			return DubboResult.successResult(ret);
		} catch (Exception e) {
			LOGGER.error("appendStock:", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<Boolean> deductStock(Long itemId, Long number) {
		try {
			if (itemId == null || number == null || number < 0) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			Boolean ret = itemService.deductStock(itemId, number);
			return DubboResult.successResult(ret);
		} catch (Exception e) {
			LOGGER.error("deductStock:", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<List<ItemDto>> findAutoRecommend(Long appId) {
		try {
			if (appId == null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			List<ItemEntity> entitys = itemService.findAutoRecommend(appId);
			List<ItemDto> dtos = BeanUtils.copyList(entitys, ItemDto.class);
			return DubboResult.successResult(dtos);
		} catch (Exception e) {
			LOGGER.error("findAutoRecommend:appId=" + appId, e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<List<ItemKeyDto>> findAutoRecommendItemKey(Long appId) {
		try {
			if (appId == null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			List<ItemKeyDto> dtos = itemAutoRecommendBo.findHomeAutoRecommend(appId, 20);
			return DubboResult.successResult(dtos);
		} catch (Exception e) {
			LOGGER.error("findAutoRecommendItemKey:appId=" + appId, e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<List<ItemKeyDto>> findAutoRecommendItemKey(Long appId, Integer size, Map<String, Object> query) {
		try {
			if (appId == null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			List<ItemKeyDto> dtos = itemAutoRecommendBo.findAutoRecommend(appId, size, query);
			return DubboResult.successResult(dtos);
		} catch (Exception e) {
			LOGGER.error("findAutoRecommendItemKey:appId=" + appId, e);
			return DubboResult.failResult(e.getMessage());
		}

	}

	@Override
	public DubboResult<ItemDto> findByType(String type) {
		try {
			if (type == null) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			ItemEntity entity = itemService.findByType(type);
			ItemDto dto = BeanUtils.copy(entity, ItemDto.class);
			return DubboResult.successResult(dto);
		} catch (Exception e) {
			LOGGER.error("findByType:type=" + type, e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<String> getJsonValue(Long id, String key) {
		try {
			if (id == null || StringUtils.isEmpty(key)) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			String v = itemService.getJsonValue(id, key);
			return DubboResult.successResult(v);
		} catch (Exception e) {
			LOGGER.error("getJsonValue:id=" + id + " key=" + key, e);
			return DubboResult.failResult(e.getMessage());
		}
	}
	
    @Override
    public DubboResult<Integer> setJsonValue(Long id, String key, String value) {
    	try {
			if (id == null || StringUtils.isEmpty(key)) {
				return DubboResult.failResult(ErrorCode.E0404004.getDesc());
			}
			Integer v = itemService.setJsonValue(id, key, value);
			return DubboResult.successResult(v);
		} catch (Exception e) {
			LOGGER.error("setJsonValue:id=" + id + " key=" + key+" value="+value, e);
			return DubboResult.failResult(e.getMessage());
		}
    }
    
	@Override
	public DubboResult<Boolean> checkElasticTakeOrder(Long itemId, Long appId, Long consumerId) {
		try {
			itemGoodsService.checkElasticTakeOrder(itemId, appId, consumerId);
			return DubboResult.successResult(true);
		} catch (Exception e) {
			LOGGER.error("checkElasticTakeOrder ", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<List<ElasticGiftsItemShowDto>> getItemsByElItemIds4Mobile(List<Long> itemIds, Long appId, Long consumerId) {
		try {
			List<ElasticGiftsItemShowDto> result = itemGoodsService.getItemsByElItemIds4Mobile(itemIds, appId, consumerId);
			return DubboResult.successResult(result);
		} catch (Exception e) {
			LOGGER.error("getItemsByElItemIds ", e);
			return DubboResult.failResult(e.getMessage());
		}
	}

	@Override
	public DubboResult<ItemDto> findBySourceRelationIdAndSourceType(Long sourceRelationId, Integer sourceType) {
		return DubboResult.successResult(BeanUtils.copy(itemService.findBySourceRelationIdAndSourceType(sourceRelationId, sourceType), ItemDto.class));
	}

}
