package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemAppSpecifyDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemStockConsumeDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.PreStockDto;
import cn.com.duiba.goods.center.biz.entity.AppItemEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.entity.ItemStockSpecifyConsumeEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockPointEntity;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.EverydayLimitService;
import cn.com.duiba.goods.center.biz.service.item.ItemKeyService;
import cn.com.duiba.goods.center.biz.service.item.ItemKeyStockService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.biz.service.stock.ItemAppSpecifyService;
import cn.com.duiba.goods.center.biz.util.DateUtil;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * 
 * ItemKeyServiceImpl
 *
 */
@Service
public class ItemKeyServiceImpl implements ItemKeyService {

	private static final Logger log = LoggerFactory.getLogger(ItemKeyServiceImpl.class);

	@Autowired
	private AppItemService appItemService;
	@Autowired
	private ItemService itemService;
	@Autowired
	private PreStockService preStockService;
	@Autowired
	private ItemKeyStockService itemKeyStockService;
	@Autowired
	private EverydayLimitService everydayLimitService;
	@Autowired
	private ItemAppSpecifyService itemAppSpecifyService;

	private static final String DECR_STOCK = "decr_stock";
	private static final String DECR_PRE_STOCK = "decr_pre_stock";
	private static final String DECR_APP_SPECIFY_STOCK = "decr_app_specify_stock";
	private static final String DECR_EVERDAY_STOCK = "decr_everday_stock";

	@Override
	public ItemKeyDto findItemKey(Long appItemId, Long itemId, Long appId) {
		return findItemKey(appItemId, itemId, appId, false);
	}

	@Override
	public ItemKeyDto findItemKeyIncludeDeleted(Long appItemId, Long itemId, Long appId) {
		return findItemKey(appItemId, itemId, appId, true);
	}

	private ItemKeyDto findItemKey(Long appItemId, Long itemId, Long appId, boolean includeDeleted) {
		AppItemEntity appItemEntity = null;
		ItemEntity itemEntity = null;
		if (appItemId != null) {
			appItemEntity = appItemService.find(appItemId);
			if (appItemEntity != null && appItemEntity.getItemId() != null) {
				itemEntity = itemService.find(appItemEntity.getItemId());
			}
		} else if (itemId != null) {
			itemEntity = itemService.find(itemId);
			appItemEntity = appItemService.findByAppIdAndItemId(appId, itemId);
		}
		if (!includeDeleted && appItemEntity != null && appItemEntity.getDeleted()) {
			appItemEntity = null;
		}
		AppItemDto appItemDto = null;
		if (appItemEntity != null) {
			appItemDto = BeanUtils.copy(appItemEntity, AppItemDto.class);
		}
		ItemDto itemDto = null;
		if (itemEntity != null) {
			itemDto = BeanUtils.copy(itemEntity, ItemDto.class);
		}
		return new ItemKeyDto(appItemDto, itemDto, appId);
	}

	@Override
	public List<ItemKeyDto> findItemKeyByAppItemIds(List<Long> appItemIds, Long appId) {
		List<AppItemEntity> appItems = appItemService.findByIds(appItemIds);
		List<Long> itemIds = new ArrayList<>();
		for (AppItemEntity appItem : appItems) {
			if (appItem.getItemId() != null) {
				itemIds.add(appItem.getItemId());
			}
		}
		Map<Long, ItemEntity> itemMap = itemService.findByIdsToMap(itemIds);
		List<ItemKeyDto> itemKeys = new ArrayList<>();
		for (AppItemEntity appItem : appItems) {
			AppItemDto appItemDto = BeanUtils.copy(appItem, AppItemDto.class);
			if (appItem.getItemId() == null) {
				itemKeys.add(new ItemKeyDto(appItemDto, null, appId));
			} else {
				ItemDto itemDto = BeanUtils.copy(itemMap.get(appItem.getItemId()), ItemDto.class);
				if (itemDto == null || !itemDto.getEnable() || itemDto.getDeleted()) {
					continue;
				}
				itemKeys.add(new ItemKeyDto(appItemDto, itemDto, appId));
			}
		}
		return itemKeys;
	}

	@Override
	public Long findStock(ItemKeyDto itemKeyDto) {
		Long stock = 0L;
		if (itemKeyDto.isSelfAppItemMode()) {
			AppItemEntity entity = appItemService.find(itemKeyDto.getAppItem().getId());
			// 商品库存
			if (entity.getRemaining() != null) {
				stock = Long.valueOf(entity.getRemaining());
			}
			// 每日限量库存
			Long everydayStock = everydayLimitService.findEverydayStock(itemKeyDto);
			if (everydayStock != null && everydayStock <= stock) {
				stock = everydayStock;
			}
		} else if (itemKeyDto.isItemMode() || itemKeyDto.isDuibaAppItemMode()) {
			ItemEntity entity = itemService.find(itemKeyDto.getItem().getId());
			// 商品库存
			if (entity.getRemaining() != null) {
				stock = Long.valueOf(entity.getRemaining());
			}
			// 直充类不需要库存商品
			if (itemService.isRechargeGoods(entity.getType())) {
				stock = 1L;
			}
			// 取定向库存
			if (stock != null && stock > 0 && entity.isOpTypeItem(ItemDto.OpTypeSpecify)) {
				Long specifyStock = itemService.findSpecifyRemaining(itemKeyDto.getAppId(), entity.getId());
				if (specifyStock != null && specifyStock <= stock) {
					stock = specifyStock;
				}
			}
			// 预分配库存
			if (stock != null && stock > 0 && entity.isOpTypeItem(ItemDto.OpTypePreStockSwith)) {
				Long preStock = preStockService.getPreStock(entity, itemKeyDto.getAppId());
				if (preStock != null && preStock <= stock) {
					stock = preStock;
				}
			}
			// 每日限量库存
			Long everydayStock = everydayLimitService.findEverydayStock(itemKeyDto);
			if (everydayStock != null && everydayStock <= stock) {
				stock = everydayStock;
			}
		}
		return stock;
	}

	@Override
	public Boolean consumeStock(ItemKeyDto itemKeyDto, String bizId, String bizSource) {
		Map<String, Boolean> context = new HashMap<>();
		try {
			// 扣每日限量库存
			if (everydayLimitService.isEverydayLimit(itemKeyDto)) {
				boolean decrEverydayStock = itemKeyStockService.decrEverydayStock(itemKeyDto);
				context.put(DECR_EVERDAY_STOCK, decrEverydayStock);
			}
			if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
				ItemEntity itemEntity = BeanUtils.copy(itemKeyDto.getItem(), ItemEntity.class);
				// 扣预分配库存
				PreStockPointEntity pointStock = preStockService.getPointStock(itemEntity, itemKeyDto.getAppId());
				if (pointStock != null) {
					boolean decrPreStock = itemKeyStockService.decrPreStock(itemKeyDto.getAppId(), itemEntity.getId(), pointStock, bizId, bizSource);
					context.put(DECR_PRE_STOCK, decrPreStock);
				}
				// 扣定向库存
				if (ItemDto.TypeObject.equals(itemKeyDto.getItemDtoType()) || ItemDto.TypeCoupon.equals(itemEntity.getType())) {
					String redirectBizType = redirectType(bizSource);
					boolean decrAppSpecifyStock = itemKeyStockService.decrAppSpecifyStock(itemKeyDto.getAppId(), itemEntity, bizId, redirectBizType);
					context.put(DECR_APP_SPECIFY_STOCK, decrAppSpecifyStock);
				}
			}
			// 扣商品库存
			if (itemKeyDto.getItemDtoType().equals(ItemDto.TypeObject) || itemKeyDto.getItemDtoType().equals(ItemDto.TypeVirtual)) {
				boolean decrStock = itemKeyStockService.decrStock(itemKeyDto, bizId, bizSource);
				context.put(DECR_STOCK, decrStock);
			}
			return true;
		} catch (Exception e) {
			Long itemId = itemKeyDto.getItem() == null ? null : itemKeyDto.getItem().getId();
			Long appItemId = itemKeyDto.getAppItem() == null ? null : itemKeyDto.getAppItem().getId();
			if (e instanceof GoodsException) {
				log.warn("consumeStock:appId=" + itemKeyDto.getAppId() + ":itemId=" + itemId + ":appItemId=" + appItemId + ":bizId=" + bizId + ":bizSource=" + bizSource + ":" + e.getMessage());
			} else {
				log.error("consumeStock:appId=" + itemKeyDto.getAppId() + ":itemId=" + itemId + ":appItemId=" + appItemId + ":bizId=" + bizId + ":bizSource=" + bizSource + ":", e);
			}
			// 返还商品库存
			Boolean stock = context.get(DECR_STOCK);
			if (stock != null && stock) {
				itemKeyStockService.rollbackStock(bizId, bizSource);
			}
			// 返还定向库存
			Boolean appSpecifyStock = context.get(DECR_APP_SPECIFY_STOCK);
			if (appSpecifyStock != null && appSpecifyStock) {
				String redirectBizType = redirectType(bizSource);
				itemKeyStockService.rollbackAppSpecifyStock(bizId, redirectBizType);
			}
			// 返还预分配库存
			Boolean preStock = context.get(DECR_PRE_STOCK);
			if (preStock != null && stock) {
				itemKeyStockService.rollbackPreStock(bizId);
			}
			// 返还每日限量库存
			Boolean everydayStock = context.get(DECR_EVERDAY_STOCK);
			if (everydayStock != null && everydayStock) {
				itemKeyStockService.rollbackEverydayStock(itemKeyDto);
			}
		}
		return false;
	}

	@Override
	public Boolean rollbackStock(ItemKeyDto itemKeyDto, String bizId, String bizSource, Date bizDate) {
		if (itemKeyDto.getItemDtoType().equals(ItemDto.TypeObject) || itemKeyDto.getItemDtoType().equals(ItemDto.TypeVirtual)) {
			// 返还商品库存
			itemKeyStockService.rollbackStock(bizId, bizSource);
		}
		if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
			ItemEntity itemEntity = itemService.find(itemKeyDto.getItem().getId());
			// 返还定向库存
			if (ItemDto.TypeObject.equals(itemKeyDto.getItemDtoType()) || ItemDto.TypeCoupon.equals(itemEntity.getType())) {
				String redirectBizType = redirectType(bizSource);
				itemKeyStockService.rollbackAppSpecifyStock(bizId, redirectBizType);
			}
			// 返还预分配库存
			PreStockPointEntity pointStock = preStockService.getPointStock(itemEntity, itemKeyDto.getAppId());
			if (pointStock != null) {
				itemKeyStockService.rollbackPreStock(bizId);
			}
		}
		if (everydayLimitService.isEverydayLimit(itemKeyDto) && bizDate.after(DateUtil.getTodayZeroDate())) {
			// 返还每日限量库存(只返回当天的)
			itemKeyStockService.rollbackEverydayStock(itemKeyDto);
		}
		return true;
	}

	/**
	 * 定向库存类型
	 * 
	 * @param bizSource
	 * @return
	 */
	private String redirectType(String bizSource) {
		if (ItemStockConsumeDto.BIZ_SOURCE_NORMAL.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_HDTOOL.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_HDTOOL_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_ACTIVITY.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_ACTIVITY_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_NGAME.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_NGAME_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_GAME.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_GAME_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_GUESS.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_GUESS_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_QUESTION.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_QUESTION_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_QUIZZ.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_QUIZZ_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_SINGLE.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_SINGLE_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_TURNTABLE.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_TURNTABLE_SPECIFY;
		}
		return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_SPECIFY;
	}

	@Override
	public List<ItemKeyStockDto> batchFindStock(List<ItemKeyDto> itemKeyDtos, Long appId) {
		List<ItemKeyStockDto> keySotcks = new ArrayList<>();
		List<Long> specifyItemIds = new ArrayList<>();
		List<Long> preItemIds = new ArrayList<>();
		// 1.商品库存
		for (ItemKeyDto itemKeyDto : itemKeyDtos) {
			ItemKeyStockDto stock = new ItemKeyStockDto();
			Long goodsStock = 0L;
			if (itemKeyDto.isSelfAppItemMode()) {
				if (itemKeyDto.getAppItem().getRemaining() != null) {
					goodsStock = Long.valueOf(itemKeyDto.getAppItem().getRemaining());
				}
			} else if (itemKeyDto.isItemMode() || itemKeyDto.isDuibaAppItemMode()) {
				ItemEntity entity = BeanUtils.copy(itemKeyDto.getItem(), ItemEntity.class);
				if (entity.getRemaining() != null) {
					goodsStock = Long.valueOf(entity.getRemaining());
				}
				// 直充类不需要库存商品
				if (itemService.isRechargeGoods(entity.getType())) {
					goodsStock = 1L;
				}
				if (goodsStock != null && goodsStock > 0 && entity.isOpTypeItem(ItemDto.OpTypeSpecify)) {
					specifyItemIds.add(entity.getId());
				}
				if (goodsStock != null && goodsStock > 0 && entity.isOpTypeItem(ItemDto.OpTypePreStockSwith)) {
					preItemIds.add(entity.getId());
				}
			}
			stock.setItemKeyDto(itemKeyDto);
			stock.setGoodsStock(goodsStock);
			keySotcks.add(stock);
		}
		// 2.商品定向库存
		Map<Long, Long> specifyStockMap = new HashMap<>();
		if (!specifyItemIds.isEmpty()) {
			List<ItemAppSpecifyDto> specify = itemAppSpecifyService.findSpecifyByItemIdsAndAppId(specifyItemIds, appId);
			for (ItemAppSpecifyDto dto : specify) {
				if (dto.getRemaining() != null) {
					specifyStockMap.put(dto.getItemId(), dto.getRemaining().longValue());
				}
			}
		}
		// 3.商品预分配库存
		Map<Long, Long> preStockMap = new HashMap<>();
		if (!preItemIds.isEmpty()) {
			List<PreStockDto> pres = preStockService.batchFindPreStock(preItemIds, appId);
			for (PreStockDto pre : pres) {
				preStockMap.put(pre.getItemId(), pre.getStock());
			}
		}
		for (ItemKeyStockDto dto : keySotcks) {
			ItemKeyDto key = dto.getItemKeyDto();
			Long stock = dto.getGoodsStock();
			if (key.isItemMode() || key.isDuibaAppItemMode()) {
				Long specifyStock = specifyStockMap.get(key.getItem().getId());
				dto.setSpecifyStock(specifyStock);
				if (specifyStock != null && specifyStock <= stock) {
					stock = specifyStock;
				}
				Long preStock = preStockMap.get(key.getItem().getId());
				dto.setPreStock(preStock);
				if (preStock != null && preStock <= stock) {
					stock = preStock;
				}
			}
			dto.setStock(stock);
		}
		return keySotcks;
	}

}
