package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.Date;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemStockConsumeDto;
import cn.com.duiba.goods.center.biz.dao.item.ItemAppSpecifyDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemStockConsumeDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemStockSpecifyConsumeDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockConsumeDetailDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockPointDao;
import cn.com.duiba.goods.center.biz.entity.ItemAppSpecifyEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.entity.ItemStockConsumeEntity;
import cn.com.duiba.goods.center.biz.entity.ItemStockSpecifyConsumeEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockConsumeDetailEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockPointEntity;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.EverydayLimitService;
import cn.com.duiba.goods.center.biz.service.item.ItemKeyStockService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.biz.service.stock.MemStockService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

/**
 * 
 * ItemKeyStockServiceImpl
 *
 */
@Service
public class ItemKeyStockServiceImpl implements ItemKeyStockService {

	private static final Logger LOGGER = LoggerFactory.getLogger(ItemKeyStockServiceImpl.class);

	@Autowired
	private AppItemService appItemService;
	@Autowired
	private ItemService itemService;
	@Autowired
	private MemStockService memStockService;
	@Autowired
	private PreStockService preStockService;
	@Autowired
	private ItemStockConsumeDao itemStockConsumeDao;
	@Autowired
	private ItemAppSpecifyDao itemAppSpecifyDao;
	@Autowired
	private ItemStockSpecifyConsumeDao itemStockSpecifyConsumeDao;
	@Autowired
	private PreStockDao preStockDao;
	@Autowired
	private PreStockPointDao preStockPointDao;
	@Autowired
	private PreStockConsumeDetailDao preStockConsumeDetailDao;
	@Autowired
	private EverydayLimitService everydayLimitService;
	
	@Resource(name = "creditsTransactionManager")
	private DataSourceTransactionManager creditsTransactionManager;

	@Override
	public Boolean decrStock(ItemKeyDto itemKeyDto, String bizId, String bizSource) throws GoodsException {
		try {
			DBTimeProfile.enter("decrStock");
			if (!itemKeyDto.getItemDtoType().equals(ItemDto.TypeObject) && !itemKeyDto.getItemDtoType().equals(ItemDto.TypeVirtual)) {
				throw new GoodsException(ErrorCode.E0404004.getErrorCode(), "不支持该类型：" + itemKeyDto.getItemDtoType());
			}
			if (itemKeyDto.isSelfAppItemMode()) {
				AppItemDto appItemDto = itemKeyDto.getAppItem();
				if (appItemDto.getRemaining() == null || appItemDto.getRemaining() <= 0) {
					throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "库存不足");
				}
				decrAppItemStock(appItemDto.getAppId(), appItemDto.getId(), bizId, bizSource);
				return true;
			} else if (itemKeyDto.isItemMode() || itemKeyDto.isDuibaAppItemMode()) {
				ItemDto itemDto = itemKeyDto.getItem();
				if (itemDto.getRemaining() == null || itemDto.getRemaining() <= 0) {
					throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "库存不足");
				}
				decrItemStock(itemKeyDto.getAppId(), itemDto.getId(), bizId, bizSource);
				return true;
			}
			return false;
		} finally {
			DBTimeProfile.release();
		}
	}
	
	private void decrAppItemStock(Long appId, Long appItemId, String bizId, String bizSource) throws GoodsException {
		TransactionStatus status = creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
		try {
			boolean ret = appItemService.decrStock(appId, appItemId);
			if (!ret) {
				throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "库存不足");
			}
			Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
			ItemStockConsumeEntity consume = new ItemStockConsumeEntity();
			consume.setBizId(bizId);
			consume.setBizSource(bizSource);
			consume.setRelationId(appItemId);
			consume.setRelationType(ItemStockConsumeDto.TYPE_DEV_ITEM);
			consume.setAppId(appId);
			consume.setAction(ItemStockConsumeDto.ACTION_PAY);
			consume.setQuantity(1L);
			consume.setGmtCreate(now);
			consume.setGmtModified(now);
			itemStockConsumeDao.insert(consume);
		} catch (DuplicateKeyException e) {
			status.setRollbackOnly();
			throw new GoodsException(ErrorCode.E0404011.getErrorCode(), "重复扣库存");
		} catch (Exception e) {
			status.setRollbackOnly();
			throw e;
		} finally {
			creditsTransactionManager.commit(status);
		}
		appItemService.removeCache(appItemId);
	}
	
	private void decrItemStock(Long appId, Long itemId, String bizId, String bizSource) throws GoodsException {
		TransactionStatus status = creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
		try {
			boolean ret = itemService.decrStock(itemId);
			if (!ret) {
				throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "库存不足");
			}
			Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
			ItemStockConsumeEntity consume = new ItemStockConsumeEntity();
			consume.setBizId(bizId);
			consume.setBizSource(bizSource);
			consume.setRelationId(itemId);
			consume.setRelationType(ItemStockConsumeDto.TYPE_DUIBA);
			consume.setAppId(appId);
			consume.setAction(ItemStockConsumeDto.ACTION_PAY);
			consume.setQuantity(1L);
			consume.setGmtCreate(now);
			consume.setGmtModified(now);
			itemStockConsumeDao.insert(consume);
		} catch (DuplicateKeyException e) {
			status.setRollbackOnly();
			throw new GoodsException(ErrorCode.E0404011.getErrorCode(), "重复扣库存");
		} catch (Exception e) {
			status.setRollbackOnly();
			throw e;
		} finally {
			creditsTransactionManager.commit(status);
		}
		itemService.removeCache(itemId);
	}
	
	@Override
	public Boolean decrAppSpecifyStock(Long appId, ItemEntity itemEntity, String bizId, String bizSource) throws GoodsException {
		try {
			DBTimeProfile.enter("decrAppSpecifyStock");
			if (!ItemDto.TypeObject.equals(itemEntity.getType()) && !ItemDto.TypeCoupon.equals(itemEntity.getType())) {
				return false;
			}
			if (!itemEntity.isOpTypeItem(ItemDto.OpTypeSpecify)){
				return false;
			}
			ItemAppSpecifyEntity specify = itemAppSpecifyDao.findByItemIdAndAppId(itemEntity.getId(), appId);
			if (null == specify) {
				return false;
			}
			if (specify.getRemaining() <= 0) {
				throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "定向库存不足");
			}
	        TransactionStatus status=creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
	        try {
				int row = itemAppSpecifyDao.reduceRemaining(itemEntity.getId(), appId);
				if (row < 1) {
					throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "定向库存不足");
				}
				Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
				ItemStockSpecifyConsumeEntity consume = new ItemStockSpecifyConsumeEntity();
				consume.setBizId(bizId);
				consume.setBizSource(bizSource);
				consume.setRelationId(itemEntity.getId());
				consume.setRelationType(ItemStockSpecifyConsumeEntity.TYPE_DUIBA);
				consume.setAppId(appId);
				consume.setAction(ItemStockConsumeDto.ACTION_PAY);
				consume.setQuantity(1L);
				consume.setGmtCreate(now);
				consume.setGmtModified(now);
				itemStockSpecifyConsumeDao.insert(consume);
			} catch (DuplicateKeyException e) {
	        	status.setRollbackOnly();
				throw new GoodsException(ErrorCode.E0404011.getErrorCode(), "重复扣定向库存");
			} catch (Exception e) {
	        	status.setRollbackOnly();
				throw e;
			} finally {
				creditsTransactionManager.commit(status);
			}
			return true;
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public Boolean decrPreStock(Long appId, Long itemId, PreStockPointEntity pointStock, String bizId, String bizSource) throws GoodsException {
		try {
			DBTimeProfile.enter("decrPreStock");
			PreStockEntity preStock = preStockDao.findByPointId(pointStock.getId());
			if (null == preStock) {
				throw new GoodsException(ErrorCode.E0404004.getErrorCode(), "商品无预分配库存");
			}
			if (preStock.getStock() < 1L) {
				throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "预分配库存不足");
			}
			decrPreEveryday(pointStock);
			TransactionStatus status = creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
			try {
				boolean success = preStockDao.reduceQuantity(preStock.getId(), 1L);
				if (!success) {
					throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "预分配库存不足");
				}
				PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
				detail.setBizId(bizId);
				detail.setAction(PreStockConsumeDetailEntity.ActionPay);
				detail.setPointId(pointStock.getId());
				detail.setStockId(preStock.getId());
				detail.setQuantity(1L);
				detail.setAppId(appId);
				Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
				detail.setGmtCreate(now);
				detail.setGmtModified(now);
				preStockConsumeDetailDao.insert(detail);
			} catch (DuplicateKeyException e) {
				status.setRollbackOnly();
				throw new GoodsException(ErrorCode.E0404011.getErrorCode(), "重复扣预分库存");
			} catch (Exception e) {
				status.setRollbackOnly();
				throw e;
			} finally {
				creditsTransactionManager.commit(status);
			}
			preStockService.deletePreStockCounter(pointStock.getId());
			return true;
		} finally {
			DBTimeProfile.release();
		}
	}
	
	private void decrPreEveryday(PreStockPointEntity pointStock) throws GoodsException {
		if(pointStock == null){
			return;
		}
		if (pointStock.getLimitEverydayQuantity() != null) {
			boolean limit = memStockService.consumeEverydayStock(pointStock.getId().toString(), pointStock.getLimitEverydayQuantity());
			if (!limit) {
				throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "预分配每日限量库存不足");
			}
		}
	}

	@Override
	public Boolean rollbackStock(String bizId, String bizSource) {
		try {
			ItemStockConsumeEntity consume = itemStockConsumeDao.findByBizIdAndSource(bizId, bizSource);
			if (null == consume) {
				return false;
			}
			TransactionStatus status = creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
			try {
				if (consume.getRelationType().equals(ItemStockConsumeDto.TYPE_DEV_ITEM)) {
					boolean back = appItemService.incrStock(consume.getRelationId());
					if (!back) {
						LOGGER.info("rollbackStock:appItemId= " + consume.getRelationId() + " 返还库存失败");
						return false;
					}
				} else if (consume.getRelationType().equals(ItemStockConsumeDto.TYPE_DUIBA)) {
					boolean back = itemService.incrStock(consume.getRelationId());
					if (!back) {
						LOGGER.info("rollbackStock:itemId= " + consume.getRelationId() + " 返还库存失败");
						return false;
					}
				}
				Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
				ItemStockConsumeEntity detail = new ItemStockConsumeEntity();
				detail.setBizId(consume.getBizId());
				detail.setBizSource(consume.getBizSource());
				detail.setRelationId(consume.getRelationId());
				detail.setRelationType(consume.getRelationType());
				detail.setAppId(consume.getAppId());
				detail.setAction(ItemStockConsumeDto.ACTION_BACK);
				detail.setQuantity(1L);
				detail.setGmtCreate(now);
				detail.setGmtModified(now);
				itemStockConsumeDao.insert(detail);
			} catch (Exception e) {
				status.setRollbackOnly();
				throw e;
			} finally {
				creditsTransactionManager.commit(status);
			}
			return true;
		} catch (Exception e) {
			LOGGER.error("rollbackStock:bizId=" + bizId + " 返还库存异常", e);
		}
		return false;
	}

	@Override
	public Boolean rollbackAppSpecifyStock(String bizId, String bizSource) {
		try {
			ItemStockSpecifyConsumeEntity consume = itemStockSpecifyConsumeDao.findByBizIdAndSource(bizId, bizSource);
			if (null == consume) {
				return false;
			}

			ItemAppSpecifyEntity specify = itemAppSpecifyDao.findByItemIdAndAppId(consume.getRelationId(), consume.getAppId());
			if (null == specify) {
				return false;
			}
			
			TransactionStatus status = creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
			try {
				int row = itemAppSpecifyDao.increaseRemaining(specify.getItemId(), specify.getAppId());
				if (row < 1) {
					LOGGER.info("rollbackAppSpecifyStock:bizId=" + bizId + " 返还定向库存失败");
					return false;
				}
				Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
				ItemStockSpecifyConsumeEntity detail = new ItemStockSpecifyConsumeEntity();
				detail.setBizId(bizId);
				detail.setBizSource(bizSource);
				detail.setRelationId(consume.getRelationId());
				detail.setRelationType(consume.getRelationType());
				detail.setAppId(consume.getAppId());
				detail.setAction(ItemStockConsumeDto.ACTION_BACK);
				detail.setQuantity(consume.getQuantity());
				detail.setGmtCreate(now);
				detail.setGmtModified(now);
				itemStockSpecifyConsumeDao.insert(detail);
			} catch (Exception e) {
				status.setRollbackOnly();
				throw e;
			} finally {
				creditsTransactionManager.commit(status);
			}
			
			return true;
		} catch (Exception e) {
			LOGGER.error("rollbackAppSpecifyStock:bizId=" + bizId + " 返还定向库存异常", e);
		}
		return false;
	}

	@Override
	public Boolean rollbackPreStock(String bizId) {
		try {
			PreStockConsumeDetailEntity consume = preStockConsumeDetailDao.findByBizIdAndPayAction(bizId);
			if (null == consume) {
				return false;
			}

			TransactionStatus status = creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
			try {
				boolean success = preStockDao.addQuantity(consume.getStockId(), 1L);
				if (!success) {
					LOGGER.info("rollbackPreStock:bizId=" + bizId + " 返还预分配库存失败");
					return false;
				}
				Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
				PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
				detail.setBizId(bizId);
				detail.setAction(PreStockConsumeDetailEntity.ActionBack);
				detail.setPointId(consume.getPointId());
				detail.setStockId(consume.getStockId());
				detail.setQuantity(consume.getQuantity());
				detail.setAppId(consume.getAppId());
				detail.setGmtCreate(now);
				detail.setGmtModified(now);
				preStockConsumeDetailDao.insert(detail);
			} catch (Exception e) {
				status.setRollbackOnly();
				throw e;
			} finally {
				creditsTransactionManager.commit(status);
			}
			
			preStockService.deletePreStockCounter(consume.getPointId());
			return true;
		} catch (Exception e) {
			LOGGER.error("rollbackPreStock:bizId=" + bizId + " 返还预分配库存异常", e);
		}
		return false;
	}

	@Override
	public Boolean decrEverydayStock(ItemKeyDto itemKeyDto) throws GoodsException {
		try {
			DBTimeProfile.enter("decrEverydayStock");
			if (!everydayLimitService.isEverydayLimit(itemKeyDto)) {
				return false;
			}
			return everydayLimitService.decrEeverydayStock(itemKeyDto);
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public Boolean rollbackEverydayStock(ItemKeyDto itemKeyDto) {
		try {
			return everydayLimitService.incrEverydayStock(itemKeyDto);
		} catch (Exception e) {
			LOGGER.error("rollbackEverydayStock:返还每日限量库存异常", e);
		}
		return false;
	}
}
