package cn.com.duiba.goods.center.biz.dao.item;

import java.util.List;
import java.util.Map;

import cn.com.duiba.goods.center.biz.entity.CouponEntity;

/**
 * 老的优惠券DAO
 */
public interface CouponDao {

	/**
	 * 查询券码
	 * @param id
	 * @return
	 */
	public CouponEntity find(Long id);

	/**
	 * 根据批次查询一个未锁定，未使用的优惠券
	 * @param batchId
	 * @return
	 */
	public CouponEntity findOneCouponByBatchId(Long batchId);

	/**
	 * 锁定一张（未锁定，未使用的优惠券）
	 * @param couponId
	 * @return
	 */
	public int lockCoupon(long couponId);

	/**
	 * 解锁一张（已锁定，未使用的优惠券）
	 * @param couponId
	 * @return
	 */
	public int unlockCoupon(long couponId);

	/**
	 * 通过批次ID删除批次下
	 * @param batchId
	 * @return
	 */
	public int deleteByBatchId(long batchId);

	/**
	 * 查询批次下所有券码
	 * @param batchId
	 * @return
	 */
	public List<String> findAllCodeByBatchId(long batchId);

	/**
	 * 释放优惠券
	 * @param couponId
	 */
	public void releaseCouponById(Long couponId);

	/**
	 * 批量插入优惠券
	 * @param ls
	 */
	public int batchInsertCoupon(List<CouponEntity> ls);

	/**
	 * 根据ID批量删除优惠券
	 * @param ls
	 * @return
	 */
	public Integer batchDeleteCoupon(List<Long> ls);

	/**
	 * 删除一张未使用的优惠券
	 * @param id
	 * @return
	 */
	public Integer deleteOneCoupon(Long id);

	/**
	 * 删除一个批次里所有未使用优惠券,每次删除1000条，所以此方法需要循环调用
	 * @param batchId
	 * @return
	 */
	public Integer deleteCouponByBatch(Long batchId);

	/**
	 * 是否存在优惠券
	 * @param batchId
	 * @return
	 */
	public Integer ishavingCoupon(Long batchId);

	/**
	 * 获取一张未使用的优惠券 for update,会对查出来的行加悲观锁
	 * @param batchId
	 * @return
	 */
	public CouponEntity getOneCouponByBatchForUpdate(Long batchId);

	/**
	 * 找一张优惠券
	 * @param batchId
	 * @return
	 */
	public CouponEntity findOneByBatch(Long batchId);

	/**
	 * 根据批次ID和code查询
	 * @param batchId
	 * @param code
	 * @return
	 */
	public CouponEntity findByBatchAndCode(Long batchId, String code);

	/**
	 * 根据批次号获取一批优惠券
	 * @param batchId
	 * @param max
	 * @return
	 */
	public List<CouponEntity> findNoUsedByBatch(Long batchId, Integer max);

	/**
	 * 优惠券搜索
	 * @param appItemId
	 * @param itemId
	 * @param code
	 * @param batchId
	 * @param used
	 * @param start
	 * @param pageSize
	 * @return
	 */
	public List<CouponEntity> searchCoupon(Long appItemId, Long itemId, String code, Long batchId, String used, Integer start, Integer pageSize);

	/**
	 * 根据商品ID批次ID统计券码数
	 * @param appItemId
	 * @param itemId
	 * @param batchId
	 * @return
	 */
	public Integer countCoupon(Long appItemId, Long itemId, Long batchId);

	/**
	 * 根据商品ID批次ID统计使用的券码
	 * @param appItemId
	 * @param itemId
	 * @param batchId
	 * @return
	 */
	public Integer countUsedCoupon(Long appItemId, Long itemId, Long batchId);

	/**
	 * 根据itemId计算coupon数量
	 * @param itemId
	 * @return
	 */
	public Integer countByItemId(Long itemId);

	/**
	 * 根据Id列表查询优惠券列表
	 * @param list
	 * @return
	 */
	public List<CouponEntity> findCouponList(List<Long> list);

	/**
	 * 统计某批次优惠券剩余数量
	 * @param batchId
	 * @return
	 */
	public Integer countByCouponBatch(Long batchId);

	/**
	 * 没有锁定也没有使用的量
	 * @param batchId
	 * @return
	 */
	public Integer noLockCountByBatchId(Long batchId);

	/**
	 * 优惠券翻页
	 * @param params
	 * @return
	 */
	public List<CouponEntity> findCouponByLimit(Map<String, Object> params);

	/**
	 * 优惠券翻页Count
	 * @param params
	 * @return
	 */
	public Long findCouponByCount(Map<String, Object> params);

	/**
	 * 按批次ID，和导出总数查询
	 * @param map
	 * @return
	 */
	public List<CouponEntity> findCouponByBatchAndTotal(Map<String, Object> map);

	/**
	 * 查询一批未使用的券
	 * @param batchId
	 * @param limit
	 * @return
	 */
	public List<CouponEntity> findCouponsForUpdate(long batchId, int limit);
	
	/**
	 * 使用一张券
	 * @param couponId
	 * @param orderId
	 * @return
	 */
	public int useCoupon(Long couponId, Long orderId);

	/**
	 * 异步锁定优惠券
	 * @param ids
	 * @return
	 */
	public Integer asyncLockedCoupon(List<Long> ids);
	
	/**
	 * 添加一张券
	 * @param couponEntity
	 */
	public void insert(CouponEntity couponEntity);
}
