/**
 * Project Name:goods-center-biz File Name:GoodsBatchService.java Package Name:cn.com.duiba.goods.center.biz.service
 * Date:2016年5月25日下午12:21:02 Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.goods.center.biz.service;

import java.util.Date;
import java.util.List;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.ACGStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchDto;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.common.GoodsException;

/**
 * ClassName:GoodsBatchService <br/>
 * Date: 2016年5月25日 下午12:21:02 <br/>
 *
 * @author xuhengfei
 * @see
 * @since JDK 1.6
 */
public interface GoodsBatchService {
	
	/**
	 * 创建一个普通批次
     * @param gtype 商品类型
     * @param gid   商品id
     * @param start 开始时间
     * @param end   结束时间
	 * @param stockId
	 * @return
	 */
    Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId);

    /**
     * 创建链接批次
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param start   开始时间
     * @param end     结束时间
     * @param stockId 库存id
     * @return 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId);

    /**
     * 创建重复券批次
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param start   开始时间
     * @param end     结束时间
     * @param stockId 库存id
     * @return 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId);

    /**
     * 不包含已删除和已过期批次
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @return 结果集
     * @author xuhengfei
     * @since JDK 1.6
     */
    public List<GoodsBatchEntity> findNormalBatchs(GoodsTypeEnum gtype, long gid);

    /**
     * 不包含已删除的批次
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @return 结果集
     * @author xuhengfei
     * @since JDK 1.6
     */
    List<GoodsBatchEntity> findNotDeletedBatchs(GoodsTypeEnum gtype, long gid);

    /**
     * 逻辑删除批次
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param batchId 批次id
     * @return 是否成功
     * @author xuhengfei
     * @since JDK 1.6
     */
    Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId);

    /**
     * 因为唯一索引限制，物理删除批次数据<br/>
     * 此方法只被广告类型商品调用，其他地方不允许调用
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param batchId 批次id
     * @return 是否成功
     * @author zufengze
     * @since JDK 1.6
     */
    Boolean physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId);

    /**
     * 查询批次
     *
     * @param goodsBatchId 批次id
     * @return 批次信息
     * @author xuhengfei
     * @since JDK 1.6
     */
    GoodsBatchEntity find(Long goodsBatchId);

    /**
     * 获取商品的所有批次的剩余库存
     *
     * @param gtype        商品类型
     * @param gid          商品id
     * @param goodsBatchId 批次id
     * @return 库存数
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long getSumBatchStock(GoodsTypeEnum gtype, long gid);

    /**
     * 获取商品的所有批次的库存,相对高效
     * @param gtype
     * @param gid
     * @return
     */
    Long getSumBatchStockBatch(GoodsTypeEnum gtype, long gid);

    /**
     * 获取商品的总库存(包含已经消耗的库存)
     *
     * @param gtype 商品类型
     * @param gid   商品id
     * @return 库存数
     * @author xuhengfei
     * @since JDK 1.6
     */
    Long getTotalAllBatchStock(GoodsTypeEnum gtype, long gid);

    /**
     * 将批次标记为已经售罄<br/>
     * 由于批次只记录信息，不记录库存。当库存没有时，需要外部来设置售罄状态<br/>
     *
     * @param batchId 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    void markBatchStatusUsed(long batchId);
    
    /**
     * 将批次标记为已经售罄<br/>
     * 
     * @param gtype
     * @param gid
     * @param batchId
     */
    void markBatchStatusUsed(GoodsTypeEnum gtype, long gid,long batchId);

    /**
     * 将批次标记为正在使用中<br/>
     * 由于批次只记录信息，不记录库存。当需要切换批次时，需要外部来设置售罄状态<br/>
     *
     * @param batchId 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    void markBatchStatusUsing(long batchId);
    
    /**
     * 将批次标记为正在使用中
     * 
     * @param gtype
     * @param gid
     * @param batchId
     */
    void markBatchStatusUsing(GoodsTypeEnum gtype, long gid, long batchId);

    /**
     * 将批次标记为未使用<br/>
     * 由于批次只记录信息，不记录库存。当批次被再次导入券码时，需要外部设置为未使用，以便后续使用<br/>
     *
     * @param batchId 批次id
     * @author xuhengfei
     * @since JDK 1.6
     */
    void markBatchStatusNotUsed(long batchId);

    /**
     * 更新批次有效时间.
     *
     * @param gtype    商品类型
     * @param gid      商品ID
     * @param batchId  批次ID
     * @param startDay 生效期
     * @param endDay   失效期
     * @return true, if update valid date
     */
    Boolean updateValidDate(GoodsTypeEnum gtype, long gid, long batchId, Date startDay, Date endDay);

    /**
     * 更新批次类型,广告券使用 <br/>
     *
     * @param gtype   商品类型
     * @param gid     商品id
     * @param dstType
     * @return
     * @author sunjiangrong
     * @since JDK 1.6
     */
    boolean updateGoodsType(GoodsTypeEnum gtype, long gid, int dstType);

    /**
     * 查询多个商品批次
     *
     * @param gtype 商品类型
     * @param gids  商品id列表
     * @return 结果集
     */
	List<GoodsBatchEntity> findGoodsBatchs(GoodsTypeEnum gtype, List<Long> gids);
    
    /**
     * 回收批次券码
     * 
     * @param gtype
     * @param gid
     * @param batchId
     * @param dstQuantity
     * @return
     * @throws GoodsException
     */
    Long retrieveGoodsBatch(GoodsTypeEnum gtype, Long gid, Long batchId, Long dstQuantity) throws GoodsException;

    
    /**
     * 根据批次日期查询批次
     * @param gtype
     * @param gid
     * @param start
     * @param end
     * @return
     */
    public GoodsBatchEntity findByStartDayAndDay(GoodsTypeEnum gtype, Long gid, Date start, Date end);
    
    /**
     * 批量查询库存
     * @param gtype
     * @param gids
     * @return
     */
    public List<ACGStockDto> findStockByGids(GoodsTypeEnum gtype, List<Long> gids);
    
    /**
     * 查询商品库存
     * @param gtype
     * @param gid
     * @return
     */
    public ACGStockDto findStockByGid(GoodsTypeEnum gtype, Long gid);
    
    /**
     * 查询批次信息，不封装批次对应的重复券码密码，和链接
     * @param gtype
     * @param gids
     * @return
     */
    public List<GoodsBatchDto> findBatchsNoCode(GoodsTypeEnum gtype, List<Long> gids);
    
	/**
	 * 获取一个可用批次<br/>
	 * 刷新批次状态切换下一个可用批次<br/>
	 * 如果没有返回null<br/>
	 * 
	 * @param goodsBatchs
	 * @param gtype
	 * @param gid
	 * @return
	 */
	public GoodsBatchEntity refreshGoodsBatchs(GoodsTypeEnum gtype, long gid);
	
	/**
	 * 刷新批次忽略库存用于创建批次后
	 * @param gtype
	 * @param gid
	 * @return
	 */
	public GoodsBatchEntity refreshGoodsBatchsNow(GoodsTypeEnum gtype, long gid);
    
    
}
