package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyStockDto;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.bo.item.ItemKeyBo;
import cn.com.duiba.goods.center.biz.cache.EventHomeCache;
import cn.com.duiba.goods.center.biz.dao.item.AppItemDao;
import cn.com.duiba.goods.center.biz.entity.AppItemEntity;
import cn.com.duiba.goods.center.biz.entity.AppItemExtraEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.service.AppItemClassifyService;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.item.AddrLimitService;
import cn.com.duiba.goods.center.biz.service.item.AppBannerService;
import cn.com.duiba.goods.center.biz.service.item.AppItemExtraService;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.ItemKeyService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.service.item.ItemStockManualChangeService;
import cn.com.duiba.goods.center.biz.util.CacheConstants;
import cn.com.duiba.goods.center.biz.util.DataSource;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.service.domain.dataobject.AppItemDO;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * 
 * 开发者商品
 *
 */
@Service("appItemService")
public class AppItemServiceImpl implements AppItemService {

	private static final Logger LOGGER = LoggerFactory.getLogger(AppItemServiceImpl.class);
	
	@Value("${goods.test.couponId}")
	private Long testCouponId;

	@Autowired
	private AppItemDao appItemDao;

	@Autowired
	private ItemService itemService;
	
	@Autowired
	private ItemKeyService itemKeyService;

	@Autowired
	private GoodsCouponBO goodsCouponBO;

	@Autowired
	private GoodsBatchService goodsBatchService;

	@Resource
	private EventHomeCache eventHomeCache;

	@Resource(name = "stringRedisTemplate03")
	private AdvancedCacheClient advancedCacheClient;

	@Autowired
	private ItemKeyBo itemKeyBo;
	
	@Autowired
	private AppItemExtraService appItemExtraService;
	
	@Autowired
	private AddrLimitService addrLimitService;
	
	@Autowired
	private AppBannerService appBannerService;
	
	@Autowired
	private AppItemClassifyService appItemClassifyService;
	
	@Autowired
	private ItemStockManualChangeService itemStockManualChangeService;
	
	/**
	 * 获取AppItem缓存Key
	 * 
	 * @param id
	 * @return
	 */
	private String getKeyById(Long id) {
		return CacheConstants.KEY_APP_ITEM_BY_ID + id;
	}
	
	/**
	 * 清理老的duiba-service中设置的缓存,待duiba-service下线删除掉
	 * 
	 * @param key
	 */
	private void cacheClear(String key) {
		try {
			advancedCacheClient.remove(key);
		} catch (Exception e) {
			LOGGER.error("cacheClear:", e);
		}
	}
	
	private void asyncClearCache(Long id){
		eventHomeCache.clearAppItemCache(getKeyById(id));
	}

	@Override
	public AppItemEntity find(Long id) {
		AppItemEntity entity = appItemDao.find(id);
		if (entity == null) {
			return null;
		}
		if (entity.isOpTypeAppItem(ItemDto.OpTypeGoodsCoupon)) {
			Long stock = goodsBatchService.getSumBatchStockBatch(GoodsTypeEnum.APP, entity.getId());
			if (stock != null) {
				entity.setRemaining(Integer.valueOf(stock.toString()));
			}
		}
		return entity;
	}

	@Override
	public List<AppItemEntity> findByIds(List<Long> ids) {
		if (ids == null || ids.isEmpty()) {
			return Collections.emptyList();
		}
		List<AppItemEntity> entitys = appItemDao.findByIds(ids);
		List<GoodsStockDto> stocks = new ArrayList<>();
		for (AppItemEntity entity : entitys) {
			if (entity.isOpTypeAppItem(ItemDto.OpTypeGoodsCoupon)) {
				GoodsStockDto stock = new GoodsStockDto();
				stock.setGid(entity.getId());
				stock.setGtype(GoodsTypeEnum.APP);
				stocks.add(stock);
			}
		}
		if (!stocks.isEmpty()) {
			List<GoodsStockDto> gst = goodsCouponBO.findGoodsStockByBatch(stocks);
			Map<Long, Integer> map = new HashMap<>();
			for (GoodsStockDto dto : gst) {
				map.put(dto.getGid(), Integer.valueOf(dto.getStock().toString()));
			}
			for (AppItemEntity entity : entitys) {
				if (entity.isOpTypeAppItem(ItemDto.OpTypeGoodsCoupon)) {
					Integer stock = map.get(entity.getId());
					entity.setRemaining(stock);
				}
			}
		}
		return entitys;
	}

	@Override
	public Long insert(AppItemEntity appItemEntity) {
		appItemDao.insert(appItemEntity);
		eventHomeCache.invalidApp(appItemEntity.getAppId());
		return appItemEntity.getId();
	}

	@Override
	public Boolean update(AppItemEntity appItemEntity) {
		int ret = appItemDao.update(appItemEntity);
		if (ret > 0) {
			eventHomeCache.invalidApp(appItemEntity.getAppId());
			cacheClear(getKeyById(appItemEntity.getId()));
			return true;
		}
		return false;
	}

	@Override
	public Boolean delete(Long appId, Long appItemId) {
		int ret = appItemDao.deleteOrRecoverAppItem(appId, appItemId, true, AppItemDto.StatusOff);
		if (ret > 0) {
			appItemExtraService.deleteByAppItemId(appItemId);
			appBannerService.disableByAppIdAndAppItemId(appId, appItemId);
			appItemClassifyService.deleteRelationAppItem(appItemId);
			return true;
		}
		return false;
	}

	@Override
	public Integer deleteByIds(Long appId, List<Long> appItemIds) {
		int ret = appItemDao.deleteByIds(appId, appItemIds);
		eventHomeCache.invalidAppItem(appItemIds);
		for (Long id : appItemIds) {
			asyncClearCache(id);
		}
		return ret;
	}

	@Override
	@Transactional(value = DataSource.CREDITS)
	public Boolean updateStatus(Long appId, Long appItemId, String status) {
		AppItemEntity appItem = find(appItemId);
		appItem.setStatus(status);
		if (status.equals(AppItemDO.StatusOff)) {
			appBannerService.disableByAppIdAndAppItemId(appId, appItemId);
			if (appItem.getType().equals(ItemDto.TypeTurntable) || appItem.getType().equals(ItemDto.TypeFake)) {
				appItem.setDeleted(true);
			}
			appItem.setPayload(0);
		} else if (status.equals(AppItemDO.StatusOn)) {
			int maxPayload = appItemDao.findAppMaxPayload(appId);
			appItem.setPayload(maxPayload + 1);
			appItem.setHomeDisplay(true);
			if (appItem.getItemId() != null && (ItemDto.TypeCoupon.equals(appItem.getType()) || ItemDto.TypeObject.equals(appItem.getType()))) {
				ItemEntity item = itemService.find(appItem.getItemId());
				if (item.getAutoOffDate() != null && item.getAutoOffDate().before(new Date())) {
					throw new RuntimeGoodsException(ErrorCode.E0404020);
				} else if (appItem.getAutoOffDate() != null && appItem.getAutoOffDate().before(new Date())) {
					throw new RuntimeGoodsException(ErrorCode.E0404020);
				}
			} else if (appItem.getAutoOffDate() != null && appItem.getAutoOffDate().before(new Date())) {
				throw new RuntimeGoodsException(ErrorCode.E0404020);
			}
		} else {
			throw new RuntimeGoodsException(ErrorCode.E0404004);
		}
		int ret = standUpOrDownAppItem(appItemId, status, appItem.getDeleted(), appItem.getPayload(), appItem.getHomeDisplay());
		if (ret > 0) {
			eventHomeCache.invalidAppItem(appItemId);
			cacheClear(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public Integer updateStatusByIds(Long appId, List<Long> appItemIds, String status) {
		int ret = appItemDao.updateStatusByIds(appId, appItemIds, status);
		eventHomeCache.invalidAppItem(appItemIds);
		for (Long id : appItemIds) {
			asyncClearCache(id);
		}
		return ret;
	}

	@Override
	public Boolean decrStock(Long appId, Long appItemId) {
		int ret = appItemDao.decrStock(appItemId);
		if (ret > 0) {
			return true;
		}
		return false;
	}

	@Override
	public void removeCache(Long appItemId) {
		try {
			eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClear(getKeyById(appItemId));
		} catch (Exception e) {
			LOGGER.error("removeCache:appItemId=" + appItemId, e);
		}
	}

	@Override
	public Boolean incrStock(Long appItemId) {
		int ret = appItemDao.incrStock(appItemId);
		if (ret > 0) {
			eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClear(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean appendStock(Long appId, Long appItemId, Long number) {
		int ret = appItemDao.appendStock(appItemId, number);
		if (ret > 0) {
			eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClear(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean deductStock(Long appId, Long appItemId, Long number) {
		int ret = appItemDao.deductStock(appItemId, number);
		if (ret > 0) {
			return true;
		}
		return false;
	}

	@Override
	public AppItemEntity findByAppIdAndItemId(Long appId, Long itemId) {
		return appItemDao.findByAppIdAndItemId(appId, itemId);
	}
	
	@Override
	public Integer findCountByItemId(Long itemId) {
		return appItemDao.findCountByItemId(itemId);
	}

	@Override
	public List<AppItemEntity> findByAppIdAndItemIds(Long appId, List<Long> itemIds) {
		return appItemDao.findByAppIdAndItemIds(appId, itemIds);
	}

	@Override
	public int updateRemainingAndvalidEndDate(Long appItemId, Long totalStock, Date validEndDate) {
		int ret = appItemDao.updateRemainingAndvalidEndDate(appItemId, totalStock, validEndDate);
		eventHomeCache.invalidAppItem(appItemId);
		cacheClear(getKeyById(appItemId));
		return ret;
	}

	@Override
	public int updateExpiedAppItem(Long itemId) {
		int ret = appItemDao.updateExpiedAppItem(itemId);
		List<Long> ids = appItemDao.findIdByItemId(itemId);
		for (Long id : ids) {
			asyncClearCache(id);
		}
		return ret;
	}

	@Override
	public int updateExpiedAppItemById(Long appItemId) {
		int ret = appItemDao.updateExpiedAppItemById(appItemId);
		eventHomeCache.invalidAppItem(appItemId);
		cacheClear(getKeyById(appItemId));
		return ret;
	}

	@Override
	public List<Long> findAppIdsByInActivityId(Long activityId, Integer type) {
		return appItemDao.findAppIdsByInActivityId(activityId, type);
	}

	@Override
	public List<Long> findAppIdsByActivityId(Long relationId) {
		return appItemDao.findAppIdsByActivityId(relationId);
	}

	@Override
	public List<Long> findAppIdByItemIds(List<Long> itemIds) {
		return appItemDao.findAppIdByItemIds(itemIds);
	}

	@Override
	public List<Long> findAppIdsByIds(List<Long> ids) {
		return appItemDao.findAppIdsByIds(ids);
	}
	
	@Override
	public List<Long> findIDBySourceIdsAndSourceTypes(List<Long> sourceIds, List<Integer> sourceTypes, Long appId) {
		return appItemDao.findIDBySourceIdsAndSourceTypes(sourceIds, sourceTypes, appId);
	}

	@Override
	public List<AppItemEntity> findBySourceIdsAndSourceTypes(List<Long> sourceIds, List<Integer> sourceTypes, Long appId) {
		return appItemDao.findBySourceIdsAndSourceTypes(sourceIds, sourceTypes, appId);
	}

	@Override
	public List<Long> findTopAppItemIdsDescByPayload(Long appId) {
		return appItemDao.findTopAppItemIdsDescByPayload(appId);
	}

	@Override
	public int updateAppItemPayload(Long appId, Long appItemId, Integer payload) {
		return appItemDao.updateAppItemPayload(appId, appItemId, payload);
	}

	@Override
	public int updateOfflineAppItem4Disable(Long appId) {
		List<Long> ids = appItemDao.findOfflineAppItemIds4Disable(appId);
		int ret = appItemDao.updateOfflineAppItem4Disable(appId);
		eventHomeCache.invalidApp(appId);
		for (Long id : ids) {
			asyncClearCache(id);
		}
		return ret;
	}

	@Override
	public int updateStatusByActivityIds(String status, Boolean delete, List<Long> oids) {
		int ret = appItemDao.updateStatusByActivityIds(status, delete, oids);
		List<Long> appItemIds = appItemDao.findIdsByActivityIds(oids);
		eventHomeCache.invalidAppItem(appItemIds);
		for (Long id : appItemIds) {
			asyncClearCache(id);
		}
		return ret;
	}

	@Override
	public Integer updateStatusByActivityId(String status, Boolean delete, Long activityId, Integer type) {
		int ret = appItemDao.updateStatusByActivityId(status, delete, activityId, type);
		eventHomeCache.invalidRelationActivity(activityId, type);
		List<Long> ids = appItemDao.findIdsByInActivityId(activityId, type);
		for (Long id : ids) {
			asyncClearCache(id);
		}
		return ret;
	}

	@Override
	public int findAppMaxPayload(Long appId) {
		return appItemDao.findAppMaxPayload(appId);
	}
	
	@Override
	public List<AppItemDto> findOnShelfByItemId(Long itemId) {
		return appItemDao.findOnShelfByItemId(itemId);
	}
	
	@Override
	public Integer standUpOrDownAppItem(Long id, String status, Boolean deleted, Integer payload, Boolean homeDisplay) {
        int ret = appItemDao.standUpOrDownAppItem(id, status, deleted, payload, homeDisplay);
        eventHomeCache.invalidAppItem(id);
		cacheClear(getKeyById(id));
		return ret;
	}

	@Override
	public Integer updateHdtoolAppItem(Long id, String title, String smallImage, String whiteImage, Long credits, String customPrice, String logo, Integer operationsType) {
		int ret = appItemDao.updateHdtoolAppItem(id, title, smallImage, whiteImage, credits, customPrice, logo, operationsType);
		eventHomeCache.invalidAppItem(id);
		cacheClear(getKeyById(id));
		return ret;
	}

	@Override
	public int updateMainPushAppItem(Long id, String title, String logo, String smallImage, String whiteImage, String bannerImage, Long credits, String subtitle) {
		int ret = appItemDao.updateMainPushAppItem(id, title, logo, smallImage, whiteImage, bannerImage, credits, subtitle);
		eventHomeCache.invalidAppItem(id);
		cacheClear(getKeyById(id));
		return ret;
	}

	@Override
	public List<AppItemDto> findAllBySourceTypeAndRelationIdAndDeleted(Integer sourceType, Long sourceRelationId, Boolean deleted) {
		return appItemDao.findAllBySourceTypeAndRelationIdAndDeleted(sourceType, sourceRelationId, deleted);
	}

	@Override
	public Page<AppItemDto> findPublishItemLimit(Map<String, Object> queryPublish) {
		Page<AppItemDto> page = new Page<>(0, 0);
		page.setList(appItemDao.findPublishItemLimit(queryPublish));
		page.setTotalCount(appItemDao.countPublishItemRow(queryPublish));
		return page;
	}

	@Override
	public Page<AppItemDto> findByLimit(Map<String, Object> params) {
		Page<AppItemDto> page = new Page<>(0, 0);
		page.setList(appItemDao.findByLimit(params));
		page.setTotalCount(appItemDao.findByLimitCount(params));
		return page;
	}

	@Override
	public int updateSubTypeById(Long appItemId, Integer subType) {
		return appItemDao.updateSubTypeById(appItemId, subType);
	}

	@Override
	public int updateValidEndDateById(Long appItemId, Date validEndDate) {
		return appItemDao.updateValidEndDateById(appItemId, validEndDate);
	}
	
    @Override
    public List<Long> findSelfAppItemIdsByAppId(Long appId,List<String> types,Integer warnLine) {
        return appItemDao.findSelfAppItemIdsByAppId(appId,types,warnLine);
    }
	

	@Override
	public List<AppItemDto> findAllByAppIdAndInType(Long appId, String[] types) {
		return appItemDao.findAllByAppIdAndInType(appId, types);
	}

	@Override
	public Integer getRemaining0Count(Long appId) {
		List<AppItemDto> ais = appItemDao.findOnlineAppItems(appId);
		List<ItemKeyDto> itemKeys = itemKeyBo.getBatchItemKeyByAppItem(ais, appId);
		List<ItemKeyStockDto> stocks = itemKeyService.batchFindStock(itemKeys, appId);
		int num = 0;
		for (ItemKeyStockDto stock : stocks) {
			if (stock.getStock() != null && stock.getStock() < 1) {
				num = num + 1;
			}
		}
		return num;
	}

	@Override
	public Integer findOnlineCount(Long appId) {
		return appItemDao.findOnlineCount(appId);
	}

	@Override
	public Long getOrAddAppItemId(String type, Long appId, Long appCreditsRate) {
		if(!itemService.isRechargeGoods(type)){
			throw new RuntimeGoodsException(ErrorCode.E0404004);
		}
		ItemEntity item = itemService.findByType(type);
		AppItemEntity appItem = addItemToAppItem(item, appId, Boolean.TRUE, appCreditsRate);
        return appItem.getId();
	}
	
    /**
     * 添加item到appItem或者编辑appItem
     */
	private AppItemEntity addItemToAppItem(ItemEntity item, Long appId, Boolean deleted, Long appCreditsRate) {
		Date now = new Date();
		AppItemEntity entity = appItemDao.findByAppIdAndItemId(appId, item.getId());
		if (entity == null) {
			AppItemDto ai = new AppItemDto(true);
			ai.setAppId(appId);
			ai.setItemId(item.getId());
			ai.setDeleted(deleted);
			ai.setStatus(AppItemDO.StatusOff);
			ai.setAddTime(now);
			ai.setTitle(item.getName());
			ai.setSubtitle(item.getSubtitle());
			ai.setCustomPrice(item.getCustomPrice());
			int minCredits = (int) (item.getFacePrice() * appCreditsRate / 100.0);
			if (minCredits < 1) {
				minCredits = 1;
			}
			ai.setMinFacePrice(minCredits);
			ai.setIsOwner(false);
			ai.setExchangeTipTemplate(item.getExchangeTipTemplate());
			ai.setType(item.getType());
			if (item.getValidEndDate() != null && item.getValidEndDate().getTime() <= now.getTime()) {
				ai.setSubStatus(AppItemDO.SubStatusExpired);
			}
			ai.setCredits(null);// 设置为null 表示从Item来计算此credits
			ai.setPayload(0);
			AppItemEntity inst = BeanUtils.copy(ai, AppItemEntity.class);
			appItemDao.insert(inst);
			entity = inst;
		}
		return entity;
	}

	@Override
	public List<AppItemDto> findAllUpVirtaulAppItems(Long appId) {
		return appItemDao.findAllUpVirtaulAppItems(appId);
	}

	@Transactional(value = DataSource.CREDITS)
    public void doUpdateCoupon(AppItemEntity appItem, String exchangeLimit, String arealimitSwitch, String arealimitType, String arealimitAreas, String usePrompt, String btnText){
        if (appItem.getId() == null) {
            insert(appItem);
        } else {
            if (appItem.getLimitCount() == null || appItem.getLimitTimeBetween() == null || appItem.getLimitEverydayQuantity() == null) {
                appItemDao.setLimitNull(appItem.getId(), appItem.getLimitCount(), appItem.getLimitTimeBetween(),	appItem.getLimitEverydayQuantity());
            }
            update(appItem);
        }
        // 额外限制开关打开
        if (StringUtils.isNotBlank(exchangeLimit) && exchangeLimit.equals("on")) {
            // 是否开启地域限制
            if (StringUtils.isNotBlank(arealimitSwitch) && arealimitSwitch.equals("on")) {
                // 增加地域限制
            	addrLimitService.devAppItemAddAddrLimit(appItem, arealimitType, arealimitAreas);
            }
        }
        // 保存新增字段
        AppItemExtraEntity appItemExtraDO = new AppItemExtraEntity();
        appItemExtraDO.setAppItemId(appItem.getId());
        appItemExtraDO.setUsePrompt(usePrompt);
        if(StringUtils.isNotBlank(btnText)) {
            appItemExtraDO.setBtnText(btnText);
        } else {
            appItemExtraDO.setBtnText("马上使用");
        }
        appItemExtraService.insertOrUpdateByAppItemId(appItemExtraDO);
    }

	@Override
	public void updateOperationsTypeById(Long id, Integer operationsType) {
		appItemDao.updateOperationsTypeById(id, operationsType);
        eventHomeCache.invalidAppItem(id);
		cacheClear(getKeyById(id));
	}

	@Transactional(value = DataSource.CREDITS)
	public boolean doBatchDel(Long appId, List<Long> appItemIds) {
        if(appItemIds.isEmpty()){
        	return false;
        }
        for(Long appItemId : appItemIds){
        	appItemDao.deleteOrRecoverAppItem(appId, appItemId, true, AppItemDO.StatusOff);
            appItemExtraService.deleteByAppItemId(appItemId);
            appBannerService.disableByAppIdAndAppItemId(appId, appItemId);
            appItemClassifyService.deleteRelationAppItem(appItemId);
        }
        appBannerService.removeCacheByAppId(appId);
        return true;
	}
	
	public int bathcStandUp(Long appId, String virtualExchangeUrl, List<Long> appItemIds) {
		if (appItemIds.isEmpty()) {
			return 0;
		}
		List<AppItemEntity> ais = findByIds(appItemIds);
		// 虚拟商品批量上架排除
		for (int i = ais.size() - 1; i >= 0; i--) {
			AppItemEntity appItem = ais.get(i);
			if (!appItem.getAppId().equals(appId)) {// 在遍历时，进行权限验证
				ais.remove(i);
			} else if (appItem.getType().equals(ItemDO.TypeVirtual)) {
				if (StringUtils.isBlank(virtualExchangeUrl)) {
					ais.remove(i);
				}
			}
		}
		int temp = 0;
		for (AppItemEntity ai : ais) {
			if (ai.getStatus().equals(AppItemDO.StatusOff)) {
				int maxPayload = findAppMaxPayload(appId);
				standUpOrDownAppItem(ai.getId(), AppItemDO.StatusOn, false, maxPayload + 1, ai.getHomeDisplay());
				temp = temp + 1;
			}
		}
		return temp;
	}
	
	@Override
	public Long addTestCoupon(Long appId) {
        ItemEntity item = itemService.find(testCouponId);
        Date today = new Date();
        AppItemDto appItem = new AppItemDto(true);
        appItem.setAppId(appId);
        appItem.setItemId(item.getId());
        appItem.setStatus(AppItemDO.StatusOn);
        appItem.setAddTime(today);
        appItem.setTitle(item.getName());
        appItem.setSubtitle(item.getSubtitle());
        appItem.setCustomPrice(item.getCustomPrice());
        appItem.setCredits(null);
        appItem.setPayload(findAppMaxPayload(appId) + 1);
        appItem.setMinFacePrice(item.getMinFacePrice());
        appItem.setLogo(item.getLogo());
        appItem.setImage(item.getImage());
        appItem.setSmallImage(item.getSmallImage());
        appItem.setMultiImage(item.getMultiImage());
        appItem.setDeleted(false);
        appItem.setType(item.getType());
        appItem.setOperationsType(0);
        appItem.setExchangeTipTemplate(item.getExchangeTipTemplate());
        appItem.setIsOwner(false);
        if (item.getValidEndDate() != null) {
            if (item.getValidEndDate().getTime() > today.getTime()) {
                appItem.setSubStatus(null);
            } else {
                appItem.setSubStatus(AppItemDO.SubStatusExpired);
            }
        }
        return insert(BeanUtils.copy(appItem, AppItemEntity.class));
	}
	
	@Override
	public Boolean doUpdateOwnerObject(AppItemDto appItem, Integer preRemaining, Integer remaining, String exchangeLimit, String arealimitSwitch, String arealimitType, String arealimitAreas) {
		AppItemEntity appItemEntity = BeanUtils.copy(appItem, AppItemEntity.class);
        if (appItem.getId() == null) {
            insert(appItemEntity);
        } else {
            itemStockManualChangeService.updateRemainingByEdit(appItem.getId(), preRemaining, remaining);
            if (appItem.getLimitCount() == null || appItem.getLimitTimeBetween() == null || appItem.getLimitEverydayQuantity() == null) {
                appItemDao.setLimitNull(appItem.getId(), appItem.getLimitCount(), appItem.getLimitTimeBetween(), appItem.getLimitEverydayQuantity());
            }
            update(appItemEntity);
        }
        //额外限制开关打开
        if(StringUtils.isNotBlank(exchangeLimit) && exchangeLimit.equals("on")){
            //是否开启地域限制
            if(StringUtils.isNotBlank(arealimitSwitch) && arealimitSwitch.equals("on")){
                //增加地域限制
            	addrLimitService.devAppItemAddAddrLimit(appItemEntity,arealimitType,arealimitAreas);
            }
        }
        return true;
    }

}
