package cn.com.duiba.goods.center.biz.service.item.impl;

import java.text.SimpleDateFormat;
import java.util.Date;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.dao.item.EverydayLimitDAO;
import cn.com.duiba.goods.center.biz.entity.EverydayLimitEntity;
import cn.com.duiba.goods.center.biz.service.item.EverydayLimitService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.wolf.cache.CacheClient;

/**
 * EverydayLimitServiceImpl
 *
 */
@Service
public class EverydayLimitServiceImpl implements EverydayLimitService {

	private static Logger log = LoggerFactory.getLogger(EverydayLimitServiceImpl.class);

	private static final String ITEM_NAMESPACE = "itemcache";
	private static final String APPITEM_NAMESPACE = "appitemcache";

	@Autowired
	private EverydayLimitDAO everydayLimitDAO;

	@Resource(name = "stringRedisTemplate03")
	private StringRedisTemplate stringRedisTemplate;

	@Override
	public Boolean isEverydayLimit(ItemKeyDto itemKeyDto) {
		if (itemKeyDto.isItemMode()) {
			return itemKeyDto.getItem().isOpTypeItem(ItemDto.OpTypeQuantityLimit);
		} else if (itemKeyDto.isDuibaAppItemMode()) {
			if (itemKeyDto.getAppItem().isOpTypeAppItem(ItemDto.OpTypeQuantityLimit)) {
				return true;
			} 
			if (itemKeyDto.getItem().isOpTypeItem(ItemDto.OpTypeQuantityLimit)) {
				return true;
			}
		} else if (itemKeyDto.isSelfAppItemMode()) {
			return itemKeyDto.getAppItem().isOpTypeAppItem(ItemDto.OpTypeQuantityLimit);
		}
		return false;
	}

	@Override
	public Long findEverydayStock(ItemKeyDto itemKeyDto) {
		if (!isEverydayLimit(itemKeyDto)) {
			return null;
		}
		Integer limit = getLimit(itemKeyDto);
		if (limit == null) {
			return null;
		}
		Integer sales = findEverydayLimitSales(itemKeyDto);
		Integer stock = 0;
		if (sales < limit) {
			stock = limit - sales;
		}
		return Long.valueOf(stock);
	}

	@Override
	public Boolean decrEeverydayStock(ItemKeyDto itemKeyDto) throws GoodsException {
		if (!isEverydayLimit(itemKeyDto)) {
			return null;
		}
		Integer limit = getLimit(itemKeyDto);
		if (limit == null) {
			return null;
		}
		Long appItemId = itemKeyDto.getAppItem() == null ? 0 : itemKeyDto.getAppItem().getId();
		Long itemId = itemKeyDto.getItem() == null ? 0 : itemKeyDto.getItem().getId();
		boolean ret = updateEverydaySales(appItemId, itemId, limit);
		// DELETE 兼容写法之前每日限量是在MEMCACHED里面计数数据容易出现不一致现在以everydat_limit表为准
		deleteEverydayCache(itemKeyDto);
		return ret;
	}

	@Override
	public Boolean incrEverydayStock(ItemKeyDto itemKeyDto) {
		if (!isEverydayLimit(itemKeyDto)) {
			return null;
		}
		Long appItemId = itemKeyDto.getAppItem() == null ? 0 : itemKeyDto.getAppItem().getId();
		Long itemId = itemKeyDto.getItem() == null ? 0 : itemKeyDto.getItem().getId();
		rollbackEverydayLimit(appItemId, itemId);
		// DELETE 兼容写法之前每日限量是在MEMCACHED里面计数数据容易出现不一致现在以everydat_limit表为准
		deleteEverydayCache(itemKeyDto);
		return true;
	}

	/**
	 * 获取限量商品限制数
	 * 
	 * @param itemKey
	 * @return
	 */
	private Integer getLimit(ItemKeyDto itemKey) {
		Integer limit = null;
		if (itemKey.isItemMode()) {
			limit = itemKey.getItem().getLimitEverydayQuantity();
		} else if (itemKey.isDuibaAppItemMode()) {
			if (itemKey.getItem().isOpTypeItem(ItemDto.OpTypeQuantityLimit)) {
				limit = itemKey.getItem().getLimitEverydayQuantity();
			} else if (itemKey.getAppItem().isOpTypeAppItem(ItemDto.OpTypeQuantityLimit)) {
				limit = itemKey.getAppItem().getLimitEverydayQuantity();
			}
		} else if (itemKey.isSelfAppItemMode()) {
			limit = itemKey.getAppItem().getLimitEverydayQuantity();
		}
		return limit;
	}

	@Override
	public boolean updateEverydaySales(Long appItemId, Long itemId, int limit) throws GoodsException{
		Date day = new Date();
		Long appItem = appItemId == null ? 0 : appItemId;
		Long item = itemId == null ? 0 : itemId;
		EverydayLimitEntity everyday = everydayLimitDAO.find(appItem, item, day);
		if (everyday == null) {
			everyday = new EverydayLimitEntity();
			everyday.setAppItemId(appItem);
			everyday.setItemId(item);
			everyday.setSales(0);
			everyday.setDay(day);
			try {
				everydayLimitDAO.insert(everyday);
			} catch (DuplicateKeyException e) {
				log.warn("everydayLimitCheck unique:appItemId:" + appItemId + " itemId:" + itemId + " day:" + new SimpleDateFormat("yyyy-MM-dd").format(day), e);
			}
		}
		int ret = everydayLimitDAO.updateSales(everyday.getId(), limit);
		if (ret < 1) {
			throw new GoodsException(ErrorCode.E0404011.getErrorCode(), "每日限量库存不足");
		}
		return true;
	}

	@Override
	public void rollbackEverydayLimit(Long appItemId, Long itemId) {
		Date day = new Date();
		Long appItem = appItemId == null ? 0 : appItemId;
		Long item = itemId == null ? 0 : itemId;
		EverydayLimitEntity everyday = everydayLimitDAO.find(appItem, item, day);
		if (everyday == null) {
			return;
		}
		everydayLimitDAO.rollbackSales(everyday.getId());
	}

	@Override
	public Integer findEverydayLimitSales(Long appItemId, Long itemId) {
		Date day = new Date();
		Long appItem = appItemId == null ? 0 : appItemId;
		Long item = itemId == null ? 0 : itemId;
		EverydayLimitEntity everyday = everydayLimitDAO.find(appItem, item, day);
		if (everyday == null) {
			return 0;
		}
		return everyday.getSales();
	}

	@Override
	public Integer findEverydayLimitSales(ItemKeyDto itemKeyDto) {
		Long appItemId = itemKeyDto.getAppItem() == null ? 0 : itemKeyDto.getAppItem().getId();
		Long itemId = itemKeyDto.getItem() == null ? 0 : itemKeyDto.getItem().getId();
		return findEverydayLimitSales(appItemId, itemId);
	}

	private void deleteEverydayCache(ItemKeyDto itemKey) {
		try {
			String key = getKey(itemKey);
			stringRedisTemplate.delete(key);
		} catch (Exception e) {
			log.error("deleteEverydayCache", e);
		}
	}

	private String getKey(ItemKeyDto itemKey) {
		String key = null;
		if (itemKey.isItemMode()) {
			key = ITEM_NAMESPACE + itemKey.getItem().getId();
		} else if (itemKey.isDuibaAppItemMode()) {
			if (itemKey.getItem().isOpTypeItem(ItemDto.OpTypeQuantityLimit)) {
				key = ITEM_NAMESPACE + itemKey.getItem().getId();
			} else if (itemKey.getAppItem().isOpTypeAppItem(ItemDto.OpTypeQuantityLimit)) {
				key = APPITEM_NAMESPACE + itemKey.getAppItem().getId();
			}
		} else if (itemKey.isSelfAppItemMode()) {
			key = APPITEM_NAMESPACE + itemKey.getAppItem().getId();
		}
		return key;
	}

}
