package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.ElasticGiftsItemShowDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.biz.dao.item.ItemDao;
import cn.com.duiba.goods.center.biz.entity.GoodsLimitRecordEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.service.GoodsLimitRecordService;
import cn.com.duiba.goods.center.biz.service.item.ItemGoodsService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;

import com.google.common.base.Preconditions;

/**
 * Created by gyf .
 * 16/10/17 .
 */
@Service("itemGoodsService")
public class ItemGoodsServiceImpl implements ItemGoodsService {

    @Autowired
    private ItemDao itemDao;

    @Autowired
    private GoodsLimitRecordService goodsLimitRecordService;

    @Autowired
    private PreStockService preStockService;

    @Autowired
    private ItemService itemService;

    @Override
    public List<ElasticGiftsItemShowDto> getItemsByElItemIds4Mobile(List<Long> itemIds, Long appId, Long consumerId) throws GoodsException {
        List<ItemEntity> items = getItemWithoutConsumerWithCache(itemIds, appId);
        itemLimitFilter(items, consumerId);
        List<ElasticGiftsItemShowDto> result = new ArrayList<>();
        for (ItemEntity it : items) {
            result.add(new ElasticGiftsItemShowDto(it.getId(), it.getWhiteImage()));
        }
        return result;
    }

    @Override
    public void checkElasticTakeOrder(Long itemId, Long appId, Long consumerId) throws GoodsException {
        Preconditions.checkNotNull(itemId, "兑换项参数不全");
        Preconditions.checkNotNull(appId, "app参数不全");
        Preconditions.checkNotNull(consumerId, "用户参数不全");
        ItemEntity item = itemService.findCorrectStockItem(itemId);
        if (item == null || item.getDeleted()) {
            throw new GoodsException(ErrorCode.E9999999.getErrorCode(),"兑换项不存在");
        }
        if (!item.getEnable()) {
        	throw new GoodsException(ErrorCode.E9999999.getErrorCode(),"兑换项未启用");
        }
        Long stock = preStockService.getPreStock(item, appId);
        if (stock == null) {
            stock = Long.valueOf(item.getRemaining());
        }
        if (stock <= 0) {
        	throw new GoodsException(ErrorCode.E0202006.getErrorCode(),"库存不足");
        }
        int countRecord = goodsLimitRecordService.findCount(GoodsTypeEnum.DUIBA.getGtype(), itemId, consumerId);
        if (countRecord == 0) {  // 表示没兑换过
            return;
        }
        throw new GoodsException(ErrorCode.E9999999.getErrorCode(),"用户兑换超过限制");
    }

    /**
     * 过滤兑换项兑换项限制
     * @param items
     * @param consumerId
     */
    private void itemLimitFilter(List<ItemEntity> items, Long consumerId) {
        if (CollectionUtils.isEmpty(items)) {
            return;
        }
        List<Long> itemIds = new ArrayList<>(items.size());
        for (ItemEntity it : items) {
            itemIds.add(it.getId());
        }
        List<GoodsLimitRecordEntity> limitRecords = goodsLimitRecordService.findCounts(GoodsTypeEnum.DUIBA.getGtype(), itemIds, consumerId);
        Map<Long, GoodsLimitRecordEntity> limitMap = new HashMap<>();
        for (GoodsLimitRecordEntity it : limitRecords) {
            limitMap.put(it.getGid(), it);
        }
        Iterator<ItemEntity> iterator = items.iterator();
        while (iterator.hasNext()) {
            ItemEntity entity = iterator.next();
            GoodsLimitRecordEntity limit = limitMap.get(entity.getId());
            if (limit == null) {    // 没兑换过
                continue;
            }
            if (limit.getCount() == 0) {    // 没兑换过
                continue;
            }
            iterator.remove();
        }
    }

    /**
     * 缓存获取 items
     * @param ids
     * @param appId
     * @return
     * @throws ExecutionException
     */
    private List<ItemEntity> getItemWithoutConsumerWithCache(final List<Long> ids, Long appId) throws GoodsException {
        return getItemWithoutConsumer(ids);
    }

    /**
     * 获取 items 不通过 consumerid 筛选
     * @param ids
     * @param appId
     * @return
     */
    private List<ItemEntity> getItemWithoutConsumer(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.emptyList();
        }
        List<ItemEntity> items = itemDao.findAllByIds(ids);
        if (CollectionUtils.isEmpty(items)) {
            return Collections.emptyList();
        }
        List<ItemEntity> result = new ArrayList<>();
        for (ItemEntity it : items) {
            if (!it.getEnable()) {  // 未启用
                continue;
            }
            if (!it.isOpTypeItem(ItemDto.OpTypeActivity)) {   // 非活动专用
                continue;
            }
            if (!ItemDto.TypeCoupon.equals(it.getType())) {  // 非优惠券
                continue;
            }
            if (ItemDto.SubTypeLink != it.getSubType()) {    // 非链接券
                continue;
            }
            result.add(it);
        }
        return result;
    }

	@Override
	public ItemEntity find(Long itemId) {
		return itemDao.find(itemId);
	}

	@Override
	public Boolean updateEnable(Long itemId, Boolean enable) {
		int ret = itemDao.updateEnable(itemId, enable);
		if(ret > 0){
			return true;
		}
		return false;
	}
}
