package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.ListUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Preconditions;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ExchangeStockWarnDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.StockWarnEmailDto;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.cache.EventHomeCache;
import cn.com.duiba.goods.center.biz.dao.item.AppBannerDao;
import cn.com.duiba.goods.center.biz.dao.item.AppItemMaskDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemAppSpecifyDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemAutoRecommendDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemDao;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.RedisCacheService;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.util.CacheConstants;
import cn.com.duiba.goods.center.biz.util.RedisKeyFactory;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;

/**
 * itemservice Created by gyf . 16/10/25 .
 */
@Service("itemService")
public class ItemServiceImpl implements ItemService {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(ItemServiceImpl.class);

	@Autowired
	private RedisCacheService redisCacheService;
	@Autowired
	private ItemDao itemDao;
	@Autowired
	private ItemAppSpecifyDao itemAppSpecifyDao;
	@Autowired
	private GoodsCouponBO goodsCouponBO;
	@Autowired
	private GoodsBatchService goodsBatchService;
	@Autowired
	private ItemAutoRecommendDao itemAutoRecommendDao;
	@Autowired
	private AppItemMaskDao appItemMaskDao;
	@Resource(name = "creditsTransactionManager")
	private DataSourceTransactionManager creditsTransactionManager;
    @Resource
    private EventHomeCache eventHomeCache;
    @Autowired
    private AppItemService appItemService;
    @Autowired
    private AppBannerDao appBannerDao;
	//暂时需要保留，等待duiba-service缓存下线后再删除
    @Resource(name = "stringRedisTemplate03")
    private AdvancedCacheClient advancedCacheClient;
    
    /**
     * 获取item缓存ID
     * @param id
     * @return
     */
    private String getCacheKeyById(Long id){
        return CacheConstants.KEY_ITEM_BY_ID + id;
    }

	@Override
	public ItemEntity findCorrectStockItem(Long id) {
		Preconditions.checkNotNull(id, "id 不能为 null");
		ItemEntity item = itemDao.find(id);
		if (item == null) {
			return null;
		}
		if (item.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
			Long stock = goodsBatchService.getSumBatchStockBatch(GoodsTypeEnum.DUIBA, item.getId());
			item.setRemaining(Integer.valueOf(stock.toString()));
		}
		return item;
	}

	@Override
	public ItemEntity find(Long id) {
		ItemEntity entity = itemDao.find(id);
		if (entity == null) {
			return null;
		}
		if (entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
			Long stock = goodsBatchService.getSumBatchStockBatch(GoodsTypeEnum.DUIBA, entity.getId());
			entity.setRemaining(Integer.valueOf(stock.toString()));
		}
		return entity;
	}
	
	@Override
	public ItemEntity findByType(String type) {
		// 1.只用于查询兑吧固定的直充类商品
		ItemEntity entity = itemDao.findByType(type);
		if (entity == null) {
			return null;
		}
		return entity;
	}

	@Override
	public List<ItemEntity> findByIds(List<Long> ids) {
		List<ItemEntity> entitys = itemDao.findAllByIds(ids);
		List<GoodsStockDto> stocks = new ArrayList<>();
		for (ItemEntity entity : entitys) {
			if (entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
				GoodsStockDto stock = new GoodsStockDto();
				stock.setGid(entity.getId());
				stock.setGtype(GoodsTypeEnum.DUIBA);
				stocks.add(stock);
			}
		}
		if (!stocks.isEmpty()) {
			List<GoodsStockDto> gst = goodsCouponBO.findGoodsStockByBatch(stocks);
			Map<Long, Long> map = new HashMap<>();
			for (GoodsStockDto dto : gst) {
				map.put(dto.getGid(), dto.getStock());
			}
			for (ItemEntity entity : entitys) {
				Long stock = map.get(entity.getId());
				if (stock != null && entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
					entity.setRemaining(Integer.valueOf(stock.toString()));
				}
			}
		}
		return entitys;
	}
	
	@Override
	public Map<Long, ItemEntity> findByIdsToMap(List<Long> ids) {
		List<ItemEntity> entitys = itemDao.findAllByIds(ids);
		List<GoodsStockDto> stocks = new ArrayList<>();
		Map<Long, ItemEntity> maps = new HashMap<>();
		for (ItemEntity entity : entitys) {
			maps.put(entity.getId(), entity);
			if (entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
				GoodsStockDto stock = new GoodsStockDto();
				stock.setGid(entity.getId());
				stock.setGtype(GoodsTypeEnum.DUIBA);
				stocks.add(stock);
			}
		}
		if (!stocks.isEmpty()) {
			List<GoodsStockDto> gst = goodsCouponBO.findGoodsStockByBatch(stocks);
			Map<Long, Long> map = new HashMap<>();
			for (GoodsStockDto dto : gst) {
				map.put(dto.getGid(), dto.getStock());
			}
			for (ItemEntity entity : entitys) {
				Long stock = map.get(entity.getId());
				if (stock != null && entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
					entity.setRemaining(Integer.valueOf(stock.toString()));
				}
			}
		}
		return maps;
	}

	@Override
	public Long insert(ItemEntity itemEntity) {
		itemDao.insert(itemEntity);
        eventHomeCache.invalidItem(itemEntity.getId());
		return itemEntity.getId();
	}

	@Override
	public Boolean update(ItemEntity itemEntity) {
		int ret = itemDao.update(itemEntity);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemEntity.getId());
			advancedCacheClient.remove(getCacheKeyById(itemEntity.getId()));
			return true;
		}
		return false;
	}

	@Override
	public Boolean delete(Long itemId) {
		int ret = itemDao.delete(itemId);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			advancedCacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean updateEnable(Long itemId, Boolean enable) {
		int ret = itemDao.updateEnable(itemId, enable);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			advancedCacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean decrStock(Long itemId) {
		int ret = itemDao.decrStock(itemId);
		if (ret > 0) {
			return true;
		}
		return false;
	}

	@Override
	public Boolean incrStock(Long itemId) {
		int ret = itemDao.incrStock(itemId);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			advancedCacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean appendStock(Long itemId, Long number) {
		int ret = itemDao.appendStock(itemId, number);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			advancedCacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean deductStock(Long itemId, Long number) {
		int ret = itemDao.deductStock(itemId, number);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			advancedCacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<ItemEntity> findAutoRecommend(Long appId) {
		String key = RedisKeyFactory.K206 + String.valueOf(appId);
		List<ItemEntity> cache = redisCacheService.getObject(RedisKeyFactory.K206 + String.valueOf(appId));
		if (cache != null) {
			return cache;
		}
		// 自动推荐&非定向
		List<Long> allList = itemAutoRecommendDao.findItemAutoRecommendNoSpecify(true, true);
		// 自动推荐&定向关系存在
		List<Long> spcefyList = itemAutoRecommendDao.findItemSpecify(appId);
		// app 黑名单
		List<Long> appMaskingList = appItemMaskDao.findItemAppMasking(appId);
		// tag黑名单
		List<Long> tagsMaskingList = itemAutoRecommendDao.findItemAppTagsMasking(appId);
		// 开发者库黑名单
		List<Long> inAppList = itemAutoRecommendDao.findItemInApp(appId, false, false);

		List<Long> itemIdList = ListUtils.sum(allList, spcefyList);
		itemIdList = ListUtils.subtract(itemIdList, appMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, tagsMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, inAppList);

		// 避免自动推荐过多
		if (itemIdList != null && itemIdList.size() > 20) {
			itemIdList = itemIdList.subList(0, 20);
		}
		List<ItemEntity> entitys = new ArrayList<>();
		if (itemIdList != null && !itemIdList.isEmpty()) {
			entitys = findByIds(itemIdList);
		}
		redisCacheService.setObject(key, cache, 60);
		return entitys;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<ItemEntity> findAutoRecommendMap(Long appId, Map<String, Object> query) {
		query.put("appId", appId);
		query.put("autoRecommend", true);
		query.put("enable", true);
		// 自动推荐&非定向
		List<Long> allList = itemAutoRecommendDao.findItemAutoRecommendNoSpecifyMap(query);
		// 自动推荐&定向关系存在
		List<Long> spcefyList = itemAutoRecommendDao.findItemSpecifyMap(query);
		// APP 黑名单
		List<Long> appMaskingList = appItemMaskDao.findItemAppMasking(appId);
		// tag黑名单
		List<Long> tagsMaskingList = itemAutoRecommendDao.findItemAppTagsMasking(appId);
		// 开发者库黑名单
		List<Long> inAppList = itemAutoRecommendDao.findItemInApp(appId, false, false);

		List<Long> itemIdList = ListUtils.sum(allList, spcefyList);
		itemIdList = ListUtils.subtract(itemIdList, appMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, tagsMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, inAppList);
		// 避免自动推荐过多
		if (itemIdList != null && itemIdList.size() > 20) {
			itemIdList = itemIdList.subList(0, 20);
		}
		List<ItemEntity> entitys = new ArrayList<>();
		if (itemIdList != null && !itemIdList.isEmpty()) {
			entitys = findByIds(itemIdList);
		}
		return entitys;
	}

	@Override
	public Boolean isRechargeGoods(String type) {
		if (type == null) {
			return false;
		}
		if (ItemDto.TypeAlipay.equals(type)) {
			return true;
		}
		if (ItemDto.TypeAlipayCode.equals(type)) {
			return true;
		}
		if (ItemDto.TypeAlipayFast.equals(type)) {
			return true;
		}
		if (ItemDto.TypePhonebill.equals(type)) {
			return true;
		}
		if (ItemDto.TypePhonebillDingzhi.equals(type)) {
			return true;
		}
		if (ItemDto.TypePhoneflow.equals(type)) {
			return true;
		}
		if (ItemDto.TypeQB.equals(type)) {
			return true;
		}
		return false;
	}

	@Override
	public Long findSpecifyRemaining(Long appId, Long itemId) {
		return itemAppSpecifyDao.findSpecifyRemaining(itemId, appId);
	}

	@Override
	public int updateRemainingAndvalidEndDate(Long itemId, Long totalStock, Date validEndDate) {
		int ret = itemDao.updateRemainingAndvalidEndDate(itemId, totalStock, validEndDate);
		eventHomeCache.invalidItem(itemId);
        advancedCacheClient.remove(getCacheKeyById(itemId));
        return ret;
	}

    @Override
    public List<ItemDto> findHomeItems(List<Long> itemIds) {
        return itemDao.findHomeItems(itemIds);
    }

	@Override
	public void removeCache(Long itemId) {
		try {
			eventHomeCache.invalidItem(itemId);
			advancedCacheClient.remove(getCacheKeyById(itemId));
		} catch (Exception e) {
			LOGGER.error("removeCache:itemId=" + itemId, e);
		}
	}

	@Override
	public String getJsonValue(Long id, String key) {
		return itemDao.getJsonValue(id, key);
	}

	@Override
	public Integer setJsonValue(Long id, String key, String value) {
		if (StringUtils.isBlank(key)) {
			throw new RuntimeGoodsException(ErrorCode.E0404004);
		}
		TransactionStatus status = creditsTransactionManager.getTransaction(new DefaultTransactionDefinition());
		int ret = 0;
		try {
			String jsonText = itemDao.findJson4Update(id);
			JSONObject json;
			if (StringUtils.isBlank(jsonText)) {
				json = new JSONObject();
			} else {
				json = JSONObject.parseObject(jsonText);
			}
			if (StringUtils.isEmpty(value) && StringUtils.isNotEmpty(json.getString(key))) {
				json.remove(key);
			}
			if (null != value) {
				json.put(key, value);
			}
			ret = itemDao.setJsonValue(id, json.toJSONString());
		} catch (Exception e) {
			status.setRollbackOnly();
			throw e;
		} finally {
			creditsTransactionManager.commit(status);
		}
		advancedCacheClient.remove(getCacheKeyById(id));
		return ret;
	}

	@Override
	public ItemEntity findBySourceRelationIdAndSourceType(Long sourceRelationId, Integer sourceType) {
		return itemDao.findBySourceRelationIdAndSourceType(sourceRelationId, sourceType);
	}

	@Override
	public List<ItemDto> findByInTypes(List<String> types) {
		return itemDao.findByInTypes(types);
	}

	@Override
	public List<ItemDto> findAllByName(String name) {
		return itemDao.findAllByName(name);
	}

	@Override
	public List<ItemDto> findAllByExpressTemplateId(Long expressTemplateId) {
		return itemDao.findAllByExpressTemplateId(expressTemplateId);
	}

	@Override
	public Integer updateAutoOffDateNull(Long id) {
		return itemDao.updateAutoOffDateNull(id);
	}

	@Override
	public Integer updateLimitCountNull(Long id) {
		return itemDao.updateLimitCountNull(id);
	}

	@Override
	public Integer updateValidEndDateById(Long id, Date validEndDate) {
		return itemDao.updateValidEndDateById(id, validEndDate);
	}

	@Override
	public Integer updateOperationsTypeById(Long id, Integer operationsType) {
		return itemDao.updateOperationsTypeById(id, operationsType);
	}

	@Override
	public List<ItemDto> findAllActivityItemAndEnable(Boolean enable) {
		return itemDao.findAllActivityItemAndEnable(enable);
	}

	@Override
	public List<ItemDto> findAllForDuibaSecondsKill() {
		return itemDao.findAllForDuibaSecondsKill();
	}

	@Override
	public List<ItemDto> findAllTargetItemList(String type, Boolean deleted, Integer subType) {
		return itemDao.findAllTargetItemList(type, deleted, subType);
	}

	@Override
	public Page<ItemDto> findAdminItemPage(Map<String, Object> query) {
		Page<ItemDto> page = new Page<>(0, 0);
		page.setList(itemDao.findAdminItemPage(query));
		page.setTotalCount(itemDao.findAdminItemPageCount(query));
		return page;
	}

	@Override
	public Page<ItemDto> findRecommandItems(String itemName) {
		Page<ItemDto> page = new Page<>(0, 0);
		page.setList(itemDao.findRecommandItems(itemName));
		page.setTotalCount(itemDao.findRecommandItemsCount(itemName));
		return page;
	}

	@Override
	public Integer updateAutoRecommendById(Long id, Boolean autoRecommend) {
		return itemDao.updateAutoRecommendById(id, autoRecommend);
	}

	@Override
	public Integer updatePayloadById(Long id, Integer payload) {
		return itemDao.updatePayloadById(id, payload);
	}

	@Override
	public List<Long> findAutoRecommendAndTagsItems(Long tagsId) {
		return itemDao.findAutoRecommendAndTagsItems(tagsId);
	}

	@Override
	public List<ItemDto> findAllByNameAndType4Lottery(String name, String type) {
		return itemDao.findAllByNameAndType4Lottery(name, type);
	}

	@Override
	public List<Long> findIdAllByEnable(Boolean enable) {
		return itemDao.findIdAllByEnable(enable);
	}

	@Override
	public Integer updateItemClassifyIdNullById(Long id) {
		return itemDao.updateItemClassifyIdNullById(id);
	}

	@Override
	public List<ItemDto> findAllByClassify(List<Long> classifyIds) {
		return itemDao.findAllByClassify(classifyIds);
	}

	@Override
	public List<ItemDto> findAllEnableCoupon() {
		return itemDao.findAllEnableCoupon();
	}

	@Override
	public Page<ExchangeStockWarnDto> findExchangeStockWarnList(Map<String, Object> queryMap) {
		Page<ExchangeStockWarnDto> page = new Page<>(0, 0);
		page.setList(itemDao.findExchangeStockWarnList(queryMap));
		page.setTotalCount(itemDao.findExchangeStockWarnListCount(queryMap));
		return page;
	}

	@Override
	public List<ExchangeStockWarnDto> findItemStockWarns(Map<String, Object> queryMap) {
		return itemDao.findItemStockWarns(queryMap);
	}

	@Override
	public void emptyExchangeStockWarn() {
		itemDao.emptyExchangeStockWarn();
	}

	@Override
	public void saveExchangeStockWarn(ExchangeStockWarnDto exchangeStockWarn) {
		itemDao.saveExchangeStockWarn(exchangeStockWarn);
	}

	@Override
	public List<StockWarnEmailDto> findEmailStockList(Map<String, Object> queryMap) {
		return itemDao.findEmailStockList(queryMap);
	}

	@Override
	public void saveEmailStockWarn(String emailStr) {
		itemDao.saveEmailStockWarn(emailStr);
	}

	@Override
	public void emptyEmailStockWarn() {
		itemDao.emptyEmailStockWarn();
	}

	@Override
	public Integer updateSubTypeById(Long id, Integer subType) {
		return itemDao.updateSubTypeById(id, subType);
	}

	@Override
	public Long findDuibaItemChooseCount(Map<String, Object> paramMap) {
		return itemDao.findDuibaItemChooseCount(paramMap);
	}

	@Override
	public List<ItemDto> findAllBlacklistItem() {
		return itemDao.findAllBlacklistItem();
	}

	@Override
	public List<ItemDto> findAutoOffItem() {
		return itemDao.findAutoOffItem();
	}

	@Override
	public Boolean changeStatus(Long itemId, boolean enable) {
        ItemEntity item = find(itemId);
        if (item == null) {
        	throw new RuntimeGoodsException(ErrorCode.E0404004);
        }
        if ((ItemDto.TypeCoupon.equals(item.getType()) || ItemDto.TypeObject.equals(item.getType())) && enable) {
            if (item.getRemaining() <= 0) {
                throw new RuntimeGoodsException(ErrorCode.E0202006);
            }
        }
        if (ItemDto.TypeCoupon.equals(item.getType()) && enable) {
            if (StringUtils.isBlank(item.getName()) || StringUtils.isBlank(item.getDescription()) || StringUtils.isBlank(item.getMultiImage()) || item.getMinFacePrice() == null) {
            	throw new RuntimeGoodsException(ErrorCode.E0202006);
            }
        }
        if (ItemDto.TypeObject.equals(item.getType()) && enable) {
            if (StringUtils.isBlank(item.getName()) || StringUtils.isBlank(item.getDescription()) || StringUtils.isBlank(item.getMultiImage()) || item.getMinFacePrice() == null || item.getMarketPrice() == null) {
               throw new RuntimeGoodsException(ErrorCode.E0404018);
            }
        }
        item.setEnable(enable);
        if (item.getEnable()) {
            item.setPublishTime(new Date());
            if(item.getAutoOffDate()!=null && item.getAutoOffDate().before(new Date())){
               throw new RuntimeGoodsException(ErrorCode.E0404019);
            }
        }
        itemDao.enableById(itemId, enable, item.getPublishTime());
        if (!enable) {
            List<AppItemDto> appItems = appItemService.findOnShelfByItemId(itemId);
            for (AppItemDto ai : appItems) {
            	appBannerDao.disableByAppItemId(ai.getId());
            }
        }
        return true;
	}

}
