package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.PreStockDto;
import cn.com.duiba.goods.center.biz.dao.item.PreStockConsumeDetailDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockManualChangeDao;
import cn.com.duiba.goods.center.biz.dao.item.PreStockPointDao;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockConsumeDetailEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockManualChangeEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockPointEntity;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.biz.service.stock.MemStockService;
import cn.com.duiba.goods.center.biz.util.CacheConstants;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * 
 * PreStockServiceImpl
 *
 */
@Service
public class PreStockServiceImpl implements PreStockService {

	private static final Logger log = LoggerFactory.getLogger(PreStockServiceImpl.class);
	@Autowired
	private PreStockPointDao preStockPointDao;
	@Autowired
	private PreStockDao preStockDao;
	@Resource(name = "stringRedisTemplate03")
	private AdvancedCacheClient advancedCacheClient;
	@Autowired
	private PreStockConsumeDetailDao preStockConsumeDetailDao;
	@Autowired
	private PreStockManualChangeDao preStockManualChangeDao;
	@Autowired
	private MemStockService memStockService;

	@Override
	public void consumeStock(String bizId, Long appId, Long pointId) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException(ErrorCode.E0202009);
		}
		if (stock.getStock() < 1L) {
			throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "预分配库存不足 ");
		}
		Long preStockId = stock.getId();

		PreStockPointEntity pointStock = preStockPointDao.find(pointId);

		decrPreEverydayStock(pointStock);

		boolean success = preStockDao.reduceQuantity(stock.getId(), 1L);
		if (!success) {
			throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "预分配库存不足");
		}

		PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
		detail.setBizId(bizId);
		detail.setAction(PreStockConsumeDetailEntity.ActionPay);
		detail.setPointId(pointId);
		detail.setStockId(preStockId);
		detail.setQuantity(1L);
		detail.setAppId(appId);
		Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
		detail.setGmtCreate(now);
		detail.setGmtModified(now);
		preStockConsumeDetailDao.insert(detail);

		try {
			deletePreStockCounter(pointId);
		} catch (Exception e) {
			log.error(" method: deletePreStockCounter error", e);
		}
	}

	private void decrPreEverydayStock(PreStockPointEntity pointStock) throws GoodsException {
		if (null == pointStock) {
			return;
		}
		if (pointStock.getLimitEverydayQuantity() != null) {
			boolean limit = memStockService.consumeEverydayStock(pointStock.getId().toString(), pointStock.getLimitEverydayQuantity());
			if (!limit) {
				throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "预分配每日限量库存不足");
			}
		}
	}

	@Override
	public void paybackStock(String bizId) throws GoodsException {
		PreStockConsumeDetailEntity consume = preStockConsumeDetailDao.findByBizIdAndPayAction(bizId);
		if (consume == null) {
			return;
		}

		boolean success = preStockDao.addQuantity(consume.getStockId(), 1L);
		if (!success) {
			throw new GoodsException(ErrorCode.E0404011.getErrorCode(), "返还预分配库存失败");
		}

		PreStockConsumeDetailEntity detail = new PreStockConsumeDetailEntity();
		detail.setBizId(bizId);
		detail.setAction(PreStockConsumeDetailEntity.ActionBack);
		detail.setPointId(consume.getPointId());
		detail.setStockId(consume.getStockId());
		detail.setQuantity(consume.getQuantity());
		detail.setAppId(consume.getAppId());
		Date now = new Date(System.currentTimeMillis() / 1000 * 1000);
		detail.setGmtCreate(now);
		detail.setGmtModified(now);
		preStockConsumeDetailDao.insert(detail);

		try {
			deletePreStockCounter(consume.getPointId());
		} catch (Exception e) {
			log.error(" method: deletePreStockCounter error", e);
		}
	}

	/**
	 * 查询库存预分配的库存 为null则最兑换项本身的
	 */
	@Override
	public Long getPreStock(ItemEntity itemEntity, Long appId) {
		PreStockPointEntity point = getPointStock(itemEntity, appId);
		if (null != point) {
			PreStockEntity stock = preStockDao.findByPointId(point.getId());
			if (null != stock) {
				return stock.getStock();
			}
		}
		return null;
	}

	@Override
	public List<PreStockDto> findAllPreStock(List<Long> itemIds, Long appId) {

		List<PreStockPointEntity> points = preStockPointDao.findAllPointAppItem(itemIds, appId);
		// itemId去重
		Map<Long, PreStockPointEntity> filterMap = Maps.newLinkedHashMap();
		for (PreStockPointEntity point : points) { // 同一个兑吧商品,App私有库优先级大于共享库
			if (filterMap.containsKey(point.getItemId()) && point.getAppId() == null) {
				continue;
			}
			filterMap.put(point.getItemId(), point);
		}
		List<Long> pointIds = Lists.newArrayList();
		for (Map.Entry<Long, PreStockPointEntity> entry : filterMap.entrySet()) {
			PreStockPointEntity po = entry.getValue();
			pointIds.add(po.getId());
		}
		Map<Long, PreStockEntity> stockMap = Maps.newHashMap();// pointId->stock
		if (!pointIds.isEmpty()) {
			List<PreStockEntity> preStockList = preStockDao.findAllByPointIds(pointIds);
			for (PreStockEntity stock : preStockList) {
				stockMap.put(stock.getPointId(), stock);
			}
		}
		List<PreStockDto> returnList = Lists.newArrayList();
		for (PreStockPointEntity point : filterMap.values()) {
			PreStockDto dto = new PreStockDto();
			dto.setPointId(point.getId());
			dto.setAppId(point.getAppId());
			dto.setItemId(point.getItemId());
			dto.setLimitEverydayQuantity(point.getLimitEverydayQuantity());
			dto.setMinFacePrice(point.getMinFacePrice());
			dto.setLimitCount(point.getLimitCount());
			// 最重要的来了
			if (stockMap.containsKey(point.getId())) {
				dto.setStock(stockMap.get(point.getId()).getStock());
			} else {
				dto.setStock(0L);
			}
			returnList.add(dto);
		}
		return returnList;
	}

	@Override
	public Integer findPreEverydayStock(ItemEntity entity, Long appId) {
		if (entity == null || appId == null) {
			throw new RuntimeGoodsException(ErrorCode.E0404004);
		}
		PreStockPointEntity point = getPointStock(entity, appId);
		if (null == point || null == point.getLimitEverydayQuantity()) {
			return null;
		}
		return getPreEverydayStock(point).intValue();
	}

	private Long getPreEverydayStock(PreStockPointEntity point) {
		try {
			final String key = getKey(point.getId());
			Integer salesCount = (Integer) advancedCacheClient.get(key);
			if (null == salesCount) {
				Integer countNum = preStockConsumeDetailDao.countSalesTime(point.getId(), DateUtils.getDayDate(new Date()));
				int exp = Math.min(DateUtils.getToTomorrowSeconds(), 3600);
				advancedCacheClient.set(key, countNum, exp, TimeUnit.SECONDS);
				salesCount = countNum;
			}
			Integer limit = point.getLimitEverydayQuantity();
			Integer remaining = 0;
			if (salesCount < limit) {
				remaining = limit - salesCount;
			}
			return Long.valueOf(remaining);
		} catch (Exception e) {
			log.error("getPreEverydayStock", e);
			return 0L;
		}
	}

	/**
	 * 限量库存失效
	 * 
	 * @param pointId
	 * @return
	 */
	@Override
	public Boolean deletePreStockCounter(Long pointId) {
		try {
			String key = getKey(pointId);
			advancedCacheClient.remove(key);
			return true;
		} catch (Exception e) {
			log.error(" method: deletePreStockCounter", e);
		}
		return false;
	}

	/**
	 * 判断是否是兑吧库存预分配
	 * 
	 * @return
	 */
	@Override
	public PreStockPointEntity getPointStock(ItemEntity itemEntity, Long appId) {
		try {
			DBTimeProfile.enter("getPointStock");
			// 判断是否开启了预分配
			if (itemEntity == null || !itemEntity.isOpTypeItem(ItemDto.OpTypePreStockSwith)) {
				return null;
			}
			// 先查是否定向给某个APP
			PreStockPointEntity point = preStockPointDao.findAppIdAndItemId(appId, itemEntity.getId());
			if (null != point) {
				return point;
			}
			// 查共享库
			PreStockPointEntity pointShare = preStockPointDao.findItemIdAndNullApp(itemEntity.getId());
			if (null != pointShare) {
				return pointShare;
			}
			return null;
		} finally {
			DBTimeProfile.release();
		}
	}

	private String getKey(Long pointId) {
		return CacheConstants.MS_PRE_STOCK_QUANTITY + "-" + pointId;
	}

	@Override
	public long newStock(Long pointId, Long stockNum) throws GoodsException {
		PreStockEntity old = preStockDao.findByPointId(pointId);
		if (old != null) {
			return old.getId();
		}
		PreStockEntity stock = new PreStockEntity();
		stock.setPointId(pointId);
		stock.setStock(stockNum);
		stock = preStockDao.newStock(stock);
		return stock.getId();
	}

	@Override
	public boolean addStockQuantity(Long bizId, Long pointId, Integer quantity) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException(ErrorCode.E0202009);
		}

		preStockDao.addQuantity(stock.getId(), quantity);

		PreStockManualChangeEntity instance = new PreStockManualChangeEntity();
		instance.setBeforeStock(stock.getStock());
		instance.setAfterStock(stock.getStock() + quantity);
		instance.setChangeKind(PreStockManualChangeEntity.ChangeKindAdd);
		instance.setChangeQuantity(quantity);
		instance.setStockId(stock.getId());
		instance.setBizId(bizId);
		preStockManualChangeDao.insert(instance);
		return true;
	}

	@Override
	public boolean reduceStockQuantity(Long bizId, Long pointId, Integer quantity) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException(ErrorCode.E0404011.getErrorCode(), "预分配库存不存在");
		}

		if (stock.getStock() < quantity) {
			throw new GoodsException(ErrorCode.E0202006.getErrorCode(), "预分配库存不足");
		}

		preStockDao.reduceQuantity(stock.getId(), quantity);

		PreStockManualChangeEntity instance = new PreStockManualChangeEntity();
		instance.setBeforeStock(stock.getStock());
		instance.setAfterStock(stock.getStock() - quantity);
		instance.setChangeKind(PreStockManualChangeEntity.ChangekIndSub);
		instance.setChangeQuantity(quantity);
		instance.setStockId(stock.getId());
		instance.setBizId(bizId);
		preStockManualChangeDao.insert(instance);
		return true;
	}

	@Override
	public boolean reduceStockAll(Long bizId, Long pointId) throws GoodsException {
		PreStockEntity stock = preStockDao.findByPointId(pointId);
		if (null == stock) {
			throw new GoodsException(ErrorCode.E0202009);
		}

		preStockDao.reduceQuantity(stock.getId(), stock.getStock());

		PreStockManualChangeEntity instance = new PreStockManualChangeEntity();
		instance.setBeforeStock(stock.getStock());

		instance.setAfterStock(instance.getBeforeStock() - stock.getStock());
		instance.setChangeKind(PreStockManualChangeEntity.ChangekIndSub);
		instance.setChangeQuantity(Integer.parseInt(stock.getStock() + ""));
		instance.setStockId(stock.getId());
		instance.setBizId(bizId);
		preStockManualChangeDao.insert(instance);
		return true;
	}

	@Override
	public void submitPreStock(List<PreStockPointEntity> insertPointList, List<PreStockPointEntity> updatePointList) throws GoodsException {
		if (!insertPointList.isEmpty()) {
			for (PreStockPointEntity stock : insertPointList) {
				if (null == stock.getLimitCount()) {
					stock.setLimitCount(0);
				}
				// 库存中心插入库存
				preStockPointDao.insert(stock);
				newStock(stock.getId(), Long.parseLong(stock.getLimitCount() + ""));
			}
		}
		if (!updatePointList.isEmpty()) {
			for (PreStockPointEntity point : updatePointList) {
				preStockPointDao.update(point);
				// 库存中心更新库存
				if (null == point.getLimitCount()) {
					this.reduceStockAll(point.getId(), point.getId());
				} else {
					if (point.getLimitCount() < 0) {
						this.reduceStockQuantity(point.getId(), point.getId(), Math.abs(point.getLimitCount()));
					} else if (point.getLimitCount() > 0) {
						this.addStockQuantity(point.getId(), point.getId(), Math.abs(point.getLimitCount()));
					}
				}
				// 失效MEMCACHE
				deletePreStockCounterMem(point.getId());
			}
		}
	}

	@Override
	public Integer getMinExchangePrice(ItemEntity item, Long appId) {
		if(appId==null || item==null){
			throw new RuntimeGoodsException(ErrorCode.E0404004);
		}
		PreStockPointEntity point = getPointStock(item, appId);
		if (null != point && null != point.getMinFacePrice()) {
			return point.getMinFacePrice();
		}
		return null;
	}

	private void deletePreStockCounterMem(Long pointId) {
		if (null == pointId) {
			return;
		}
		String key = getKey(pointId);
		advancedCacheClient.remove(key);
	}

	@Override
	public List<PreStockDto> batchFindPreStock(List<Long> itemIds, Long appId) {
		if (itemIds.isEmpty()) {
			return Collections.emptyList();
		}
		//1.查找有没有设置预分配库存
		List<PreStockPointEntity> prePoints = preStockPointDao.findAllPointAppItem(itemIds, appId);
		Map<Long, Long> prePointMaps = new HashMap<>();
		for(PreStockPointEntity prePoint : prePoints){
			prePointMaps.put(prePoint.getItemId(), prePoint.getId());
		}
		List<Long> shareItemIds = new ArrayList<>();
		for(Long preItemId : itemIds){
			if(!prePointMaps.containsKey(preItemId)){
				shareItemIds.add(preItemId);
			}
		}
		//2.不存在预分配的看看有没有设置共享库存
		if(!shareItemIds.isEmpty()){
			List<PreStockPointEntity> sharePrePoints = preStockPointDao.findByItemIdsAndNullApp(shareItemIds);
			for(PreStockPointEntity sharePrePoint : sharePrePoints){
				prePointMaps.put(sharePrePoint.getItemId(), sharePrePoint.getId());
			}
		}
		//3.查询预分配的库存
		List<Long> pointIds = new ArrayList<>();
		for(Map.Entry<Long, Long> entry : prePointMaps.entrySet()){
			pointIds.add(entry.getValue());
		}
		if (pointIds.isEmpty()) {
			return Collections.emptyList();
		}
		List<PreStockEntity> preStocks = preStockDao.findAllByPointIds(pointIds);
		return BeanUtils.copyList(preStocks, PreStockDto.class);
	}
}
