package cn.com.duiba.goods.inner.api.autoconfig.excel;

import cn.com.duiba.goods.common.dto.ScrollingPageRequest;
import cn.com.duiba.goods.common.enums.ScrollingType;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.VerticalAlignment;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * excel工具
 * Created by guoyanfei .
 * 2022/12/26 .
 */
public class EasyExcelComponent {

    /**
     * xlsx文件单个工作簿上限是1048575行,这里如果超过104W需要分Sheet
     */
    private static final int SHEET_MAX = 1048575;

    public <T> List<T> readExcel(InputStream inputStream, Class<T> importParamClass) {
        CommonAnalysisEventListener listener = new CommonAnalysisEventListener(importParamClass);
        EasyExcelFactory.read(inputStream, importParamClass, listener).sheet().doRead();
        List<T> dataList = listener.getDataList();
        if (CollectionUtils.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        return dataList;
    }

    public <T> void writeExcel(List<T> dataList, OutputStream outputStream, Class<T> outputParamClass) {
        EasyExcelFactory.write(outputStream, outputParamClass).autoCloseStream(true).sheet().doWrite(dataList);
    }

    /**
     * pageSize
     *
     * @param outputStream
     * @param query
     * @param param
     * @param <P>
     * @param <R>
     */
    public <P extends ScrollingPageRequest, R> void pageWriteExcel(OutputStream outputStream, ExcelPageQuery<P, R> query, P param) {
        Class<R> headerClass = query.getHeaderClass();
        // 内容样式策略
        WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
        // 垂直居中,水平居中
        contentWriteCellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        contentWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.LEFT);
        contentWriteCellStyle.setBorderLeft(BorderStyle.THIN);
        contentWriteCellStyle.setBorderTop(BorderStyle.THIN);
        contentWriteCellStyle.setBorderRight(BorderStyle.THIN);
        contentWriteCellStyle.setBorderBottom(BorderStyle.THIN);
        HorizontalCellStyleStrategy contentStyleStrategy = new HorizontalCellStyleStrategy();
        contentStyleStrategy.setContentWriteCellStyleList(Arrays.asList(new WriteCellStyle[] { contentWriteCellStyle }));

        ExcelWriter excelWriter = new ExcelWriterBuilder().autoCloseStream(true).automaticMergeHead(true) // 这里一定要设置为true，否则在无法实现表头列合并
                                                          .excelType(ExcelTypeEnum.XLSX).file(outputStream).head(headerClass).registerWriteHandler(contentStyleStrategy).build();

        // 工作簿编号
        int sheetNum = 1;
        WriteSheet writeSheet = EasyExcel.writerSheet("sheet" + sheetNum).head(headerClass).build();

        // 每一个工作簿当前写入的数据量，需要把表头所占的行算上
        int currentSheetTotal = 1;

        // 滚动方向，没传就是倒序
        int scrollingType = param.getScrollingType() != null ? param.getScrollingType() : ScrollingType.DESC.code();
        param.setScrollingType(scrollingType);

        while (true) {
            // 分页捞数据
            Pair<List<R>, Long> dataPair = query.findPage(param);

            // 结果集
            List<R> dataList = dataPair.getLeft();
            int dataSize = dataList.size();

            // 判断当前sheet是不是写满，写满就写一个新的
            currentSheetTotal += dataSize;
            if (currentSheetTotal >= SHEET_MAX) {
                currentSheetTotal = 1 + dataSize;
                sheetNum++;
                writeSheet = EasyExcel.writerSheet("sheet" + sheetNum).head(headerClass).build();
            }

            // 写数据
            excelWriter.write(dataList, writeSheet);

            // 边界值设置到 param 中，下一次 大于/小于 边界值
            param.setBoundaryValue(dataPair.getRight());

            // 当前页数量小于每页条数，那么认为当前是最后一页，直接走了
            if (dataSize < param.getPageSize()) {
                break;
            }
        }
        excelWriter.finish();
    }
}
