package cn.com.duiba.kjy.base.customweb.web.factory;

import cn.com.duiba.kjy.base.customweb.web.bean.ParameterBean;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingHandler;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingHandlerBuilder;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingRegister;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.RequestMappingInfo;
import cn.com.duiba.kjy.base.customweb.web.handler.response.ResponseHandler;
import cn.com.duiba.kjy.base.customweb.web.processor.parameter.ParameterPostProcessor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.support.ApplicationObjectSupport;
import org.springframework.core.MethodIntrospector;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.stereotype.Controller;
import org.springframework.util.ClassUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author dugq
 * @date 2021/3/23 7:44 下午
 */
@Slf4j
public class HandlerMappingBeanFactory extends ApplicationObjectSupport implements InitializingBean {

    @Resource
    private ControllerMappingRegister controllerMappingRegister;

    private final List<ParameterPostProcessor> parameterPostProcessors;

    private final List<ResponseHandler> responseHandlers;

    public HandlerMappingBeanFactory(List<ParameterPostProcessor> parameterPostProcessors, List<ResponseHandler> responseHandlers) {
        this.parameterPostProcessors = parameterPostProcessors;
        this.responseHandlers = responseHandlers;
    }


    public void builderHandlerMappingAndRegister(Class<?> handlerClass, Method method, String handler, RequestMappingInfo requestMappingInfo){
        final ControllerMappingHandler controllerMappingHandler = buildOneHandlerMapping(handlerClass, method,handler,requestMappingInfo);
        controllerMappingRegister.registerHandler(controllerMappingHandler);
    }

    private ControllerMappingHandler buildOneHandlerMapping(Class<?> handlerClass, Method handlerMappingMethod, String handler, RequestMappingInfo requestMappingInfo) {
        ControllerMappingHandlerBuilder controllerMappingHandlerBuilder = ControllerMappingHandlerBuilder.create(handlerClass, handlerMappingMethod);
        controllerMappingHandlerBuilder.setRequestMappingInfo(requestMappingInfo);
        final Class<?> returnType = handlerMappingMethod.getReturnType();
        controllerMappingHandlerBuilder.setReturnType(returnType);
        controllerMappingHandlerBuilder.setResponseHandler(getResponseHandler(handlerMappingMethod,returnType));
        controllerMappingHandlerBuilder.setHandler(handler);
        fillParameterList(handlerClass, handlerMappingMethod, controllerMappingHandlerBuilder);
        return controllerMappingHandlerBuilder.build();
    }

    private void fillParameterList(Class<?> handlerClass, Method handlerMappingMethod, ControllerMappingHandlerBuilder controllerMappingHandlerBuilder) {
        final Parameter[] parameters = handlerMappingMethod.getParameters();
        List<ParameterBean> parameterBeans = new ArrayList<>();
        int index = 0;
        for (Parameter parameter : parameters) {
            ParameterBean parameterBean = new ParameterBean();
            final Class<?> type = parameter.getType();
            parameterBean.setType(type);
            parameterBean.setName(parameter.getName());
            parameterBean.setPrimitive(type.isPrimitive());
            parameterBean.setMethodParameter(new MethodParameter(handlerMappingMethod,index++));
            processorParameter(parameterBean, handlerClass, handlerMappingMethod);
            parameterBeans.add(parameterBean);
        }
        controllerMappingHandlerBuilder.setParamList(parameterBeans);
    }

    private void processorParameter(ParameterBean parameterBean, Class<?> handlerBeanClass, Method method) {
        for (ParameterPostProcessor parameterPostProcessor : parameterPostProcessors) {
            parameterPostProcessor.postProcessorParameter(parameterBean,handlerBeanClass,method);
        }
    }


    private ResponseHandler getResponseHandler(Method method, Class<?> returnType){
        for (ResponseHandler responseHandler : responseHandlers) {
            if (responseHandler.canWrite(method,returnType)){
                return responseHandler;
            }
        }
        throw new UnsupportedOperationException("can not write this type "+returnType.getName()+"in method"+method.getName());
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        init();
    }

    private void init() {
        final String[] list = obtainApplicationContext().getBeanNamesForType(Object.class);
        for (String beanName : list){

            final Class<?> beanClass = obtainApplicationContext().getType(beanName);
            if (Objects.isNull(beanClass)){
                continue;
            }
            if (!AnnotatedElementUtils.hasAnnotation(beanClass, Controller.class)){
                continue;
            }
            Class<?> userType = ClassUtils.getUserClass(beanClass);
            MethodIntrospector.selectMethods(userType,
                    (ReflectionUtils.MethodFilter) method -> Objects.nonNull(AnnotatedElementUtils.findMergedAnnotation(method, RequestMapping.class)))
                    .forEach((method -> {
                        final RequestMapping requestMapping = AnnotatedElementUtils.findMergedAnnotation(method, RequestMapping.class);
                        builderHandlerMappingAndRegister(beanClass,method,beanName,getMappingForMethod(beanClass,requestMapping));
                    }));
        }
    }


    private RequestMappingInfo getMappingForMethod(Class<?> beanClass, RequestMapping requestMapping) {
        RequestMappingInfo info = createRequestMappingInfo(requestMapping);
        if (info != null) {
            final RequestMapping ctrlMapping = AnnotatedElementUtils.findMergedAnnotation(beanClass, RequestMapping.class);
            RequestMappingInfo typeInfo = createRequestMappingInfo(ctrlMapping);
            if (typeInfo != null) {
                info = typeInfo.combine(info);
            }
            info = RequestMappingInfo.paths("/").build().combine(info);
        }
        return info;
    }


    protected RequestMappingInfo createRequestMappingInfo(
            RequestMapping requestMapping) {
        if (Objects.isNull(requestMapping)){
            return null;
        }
        return RequestMappingInfo
                .paths(requestMapping.path())
                .methods(requestMapping.method())
                .mappingName(requestMapping.name())
                .build();
    }
}
