package cn.com.duiba.kjy.base.customweb.web.handler.exception;

import cn.com.duiba.kjy.base.customweb.web.bean.KjjHttpRequest;
import cn.com.duiba.kjy.base.customweb.web.bean.KjjHttpResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.entity.ContentType;
import org.springframework.web.method.ControllerAdviceBean;
import org.springframework.web.method.annotation.ExceptionHandlerMethodResolver;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author dugq
 * @date 2021/3/29 12:03 上午
 */
@Slf4j
public class ControllerAdviceHandler implements ExceptionHandler  {
    private final Map<ControllerAdviceBean, ExceptionHandlerMethodResolver> exceptionHandler = new HashMap<>();

    private final  ObjectMapper objectMapper;

    public ControllerAdviceHandler(List<ControllerAdviceBean> adviceBeans, ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
        adviceBeans.stream()
                .filter(adviceBean->Objects.nonNull(adviceBean.getBeanType()))
                .forEach(adviceBean->{
                    final ExceptionHandlerMethodResolver exceptionHandlerMethodResolver = new ExceptionHandlerMethodResolver(adviceBean.getBeanType());
                    exceptionHandler.put(adviceBean,exceptionHandlerMethodResolver);
                });
    }


    @Override
    public boolean doHandler(KjjHttpRequest request, KjjHttpResponse response, Throwable throwable) {
        if (!(throwable instanceof Exception)){
            return false;
        }
        for (Map.Entry<ControllerAdviceBean, ExceptionHandlerMethodResolver> entry : exceptionHandler.entrySet()) {
            final Method method = entry.getValue().resolveMethodByExceptionType(throwable.getClass());
            if (Objects.isNull(method)){
                continue;
            }
            try {
                final Object[] args = getArgs(request, response, (Exception) throwable, method);
                final Object result = method.invoke(entry.getKey().resolveBean(), args);
                if (Objects.nonNull(result)){
                    final String responseBody = objectMapper.writeValueAsString(result);
                    response.write(responseBody);
                }
                response.setContentType(ContentType.APPLICATION_JSON.getMimeType());
                response.flushAndClose();
                return true;
            }catch (Exception e){
                log.error("handler exception has error in controller advice",e);
                return false;
            }
        }
        return false;

    }

    private Object[] getArgs(KjjHttpRequest request, KjjHttpResponse response, Exception exception, Method method) {
        final int parameterCount = method.getParameterCount();
        Object[] args = new Object[parameterCount];
        final Parameter[] parameters = method.getParameters();
        int index = 0;
        for (Parameter parameter : parameters) {
            if (Objects.equals(parameter.getType(),KjjHttpRequest.class)){
                args[index++] = request;
            }else if(Objects.equals(parameter.getType(),KjjHttpResponse.class)){
                args[index++] = response;
            }else if(Exception.class.isAssignableFrom(parameter.getType())){
                args[index++] = exception;
            }else{
                throw new UnsupportedOperationException("目前暂不支持其他额外的参数。");
            }
        }
        return args;
    }
}
