package cn.com.duibaboot.kjj.oss.template;

import cn.com.duibaboot.kjj.oss.template.operation.MultipartTask;
import cn.com.duibaboot.kjj.oss.template.operation.MultipartTaskImpl;
import cn.com.duibaboot.kjj.oss.template.support.MultiUploadSupport;
import cn.com.duibaboot.kjj.oss.template.util.OssUtils;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PutObjectResult;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;

/**
 * @author dugq
 * @date 2021/7/21 4:03 下午
 */
@Data
@Slf4j
public abstract class AbstractOssTemplate implements OssTemplate{
    /**
     * OSS 客户端
     */
    private OSSClient ossClient;

    /**
     * bucket的端点，也是访问域名。
     */
    private String endPoint;

    /**
     * 当前客户端操作的分区名称
     */
    private String bucketName;

    /**
     * bucket的加速访问域名
     */
    protected String cdnDomain;

    @Resource
    private ExecutorService ossExecutorService;

    @Override
    public MultipartTask createMultipartTask() {
        return new MultipartTaskImpl(this);
    }

    @Override
    public String simpleMultipartUpload(InputStream inputStream, String objectName, ObjectMetadata meta) {
        final MultipartTask multipartTask = createMultipartTask();
        try {
            final boolean result = MultiUploadSupport.simpleMultipartUpload(multipartTask, inputStream, objectName, meta);
            if (result){
                getUrl(objectName);
            }
            return null;
        } catch (IOException e) {
            log.error("上传oss发生错误，输入流读取失败 objectName={}",objectName,e);
            return null;
        }
    }

    @Override
    public <R> R downloadStream(String objectName, Function<OSSObject, R> callback) {
        OssUtils.validatorObjectName(objectName);
        final OSSObject object = ossClient.getObject(bucketName, objectName);
        try {
            return callback.apply(object);
        }finally {
            final InputStream objectContent = object.getObjectContent();
            try {
                objectContent.close();
            } catch (IOException e) {
                //ignore.如论如何，都得把inputStream关闭了。
            }
        }
    }

    @Override
    public String uploadStream(InputStream inputStream, String objectName, ObjectMetadata meta) {
        OssUtils.validatorObjectName(objectName);
        final PutObjectResult result = ossClient.putObject(bucketName, objectName, inputStream, meta);
        if (OssUtils.successful(result)){
            return getUrl(objectName);
        }
        log.error("上传oss发生错误.object name = {} msg = {}",objectName,OssUtils.getErrorMessage(result));
        return null;
    }

    @Override
    public String uploadString(String content, String objectName, ObjectMetadata meta) {
        if (StringUtils.isBlank(content)){
            log.error("上传oss发生错误:文件内容为空.object name = {}",objectName);
            return null;
        }
        final byte[] bytes = content.getBytes();
        return uploadStream(new ByteArrayInputStream(bytes),objectName,meta);
    }

    protected String getBucketUri(){
        if (!endPoint.endsWith("/")){
            return endPoint+"/"+bucketName+"/";
        }
        return endPoint+bucketName+"/";
    }
}
