package cn.com.duibaboot.kjj.oss.template.operation;

import cn.com.duibaboot.kjj.oss.template.AbstractOssTemplate;
import cn.com.duibaboot.kjj.oss.template.util.OssUtils;
import com.aliyun.oss.internal.OSSHeaders;
import com.aliyun.oss.model.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import java.io.InputStream;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * @author dugq
 * @date 2021/7/22 4:58 下午
 */
@Slf4j
public class MultipartTaskImpl implements MultipartTask {

    /**
     * oss封装类
     */
    private final AbstractOssTemplate ossTemplate;

    /**
     * objectName
     */
    private String key;

    /**
     * 它是分片上传事件的唯一标识，您可以根据这个ID来发起相关的操作，如取消分片上传、查询分片上传等。
     */
    private String uploadId;
    /**
     * 是否关闭。
     * 分片上传应该是一个闭环，要么上传成功，要么取消上传。除了支持断点续传的场景外，不应该出现上传中断的情况。继承Closeable接口，以实现此逻辑
     */
    private volatile boolean closed;

    /**
     * 分片上传成功的分片列表
     */
    private final List<PartETag> partETagList = new CopyOnWriteArrayList<>();

    /**
     * 任务是否开始
     */
    private volatile boolean created;

    private boolean success;

    public MultipartTaskImpl(AbstractOssTemplate ossTemplate){
        this.ossTemplate = ossTemplate;
    }

    @Override
    public boolean startMultipartPartUpload(String objectName, ObjectMetadata objectMetadata) {
        OssUtils.validatorObjectName(objectName);
        if (created || closed){
            log.error("分片上传任务不可重复使用。objectName={}",key);
            return false;
        }
        created = true;
        this.key = objectName;
        InitiateMultipartUploadRequest request = new InitiateMultipartUploadRequest(ossTemplate.getBucketName(), objectName);
        //强制指定只支持标准存储。
        objectMetadata.setHeader(OSSHeaders.OSS_STORAGE_CLASS, StorageClass.Standard.toString());
        request.setObjectMetadata(objectMetadata);
        try {
            // 初始化分片。
            InitiateMultipartUploadResult result = ossTemplate.getOssClient().initiateMultipartUpload(request);
            if (!OssUtils.successful(result)){
                log.error("初始化分片上传失败。objectName={} message = {}",objectName,result.getResponse().getErrorResponseAsString());
                return created = false;
            }
            if (StringUtils.isBlank(result.getUploadId())){
                log.error("初始化分片上传失败。未知错误：【uploadId】when 【init oss multipart operation】 is null! objectName={}",objectName);
                return created = false;
            }
            this.uploadId = result.getUploadId();
            return created = true;
        }catch (Exception e){
            log.error("初始化分片上传失败。objectName={}",objectName,e);
            return created = false;
        }

    }

    @Override
    public boolean startContinuePartUpload(String uploadId, String objectName) {
        OssUtils.validatorObjectName(objectName);
        if (created || closed){
            log.error("分片上传任务不可重复使用。objectName={}",key);
            return false;
        }
        created=true;
        this.uploadId = uploadId;
        this.key = objectName;
        return true;
    }

    @Override
    public boolean multipartPartUpload(InputStream inputStream, Long partSize, int partNumber) {
        if (partNumber<1 || partNumber>10000){
            log.error("分片下标超出范围。【1～10000】当前：{} objectName={} uploadId={}",partNumber,key,uploadId);
            return false;
        }
        validator();
        UploadPartRequest uploadPartRequest = new UploadPartRequest();
        uploadPartRequest.setBucketName(ossTemplate.getBucketName());
        uploadPartRequest.setKey(key);
        uploadPartRequest.setUploadId(uploadId);
        uploadPartRequest.setInputStream(inputStream);
        uploadPartRequest.setPartSize(partSize);
        uploadPartRequest.setPartNumber(partNumber);
        try {
            // 每个分片不需要按顺序上传，甚至可以在不同客户端上传，OSS会按照分片号排序组成完整的文件。
            UploadPartResult uploadPartResult = ossTemplate.getOssClient().uploadPart(uploadPartRequest);
            if (!OssUtils.successful(uploadPartResult)){
                log.error("oss分片上传失败。objectName={} uploadId={} msg={}",key,uploadId, OssUtils.getErrorMessage(uploadPartResult));
                return false;
            }
            // 每次上传分片之后，OSS的返回结果会包含一个PartETag。PartETag将被保存到partETags中。
            addPartTag(uploadPartResult.getPartETag());
            return true;
        }catch (Exception e){
            log.error("上传oss分片失败 objectName={}",key,e);
            return false;
        }

    }

    private void validator(){
        if (closed){
            throw new UnsupportedOperationException("分片上传任务不可重复使用。");
        }
        if (!created){
            throw new UnsupportedOperationException("分片任务未初始化");
        }
    }

    @Override
    public List<PartETag> getSuccessfulUploadPart(){
        return partETagList;
    }

    //为了支持并发上传
    private synchronized void addPartTag(PartETag eTag){
        partETagList.add(eTag);
    }

    @Override
    public boolean multipartUpload() {
        validator();
        // 创建CompleteMultipartUploadRequest对象。
        // 在执行完成分片上传操作时，需要提供所有有效的partETags。OSS收到提交的partETags后，会逐一验证每个分片的有效性。当所有的数据分片验证通过后，OSS将把这些分片组合成一个完整的文件。
        CompleteMultipartUploadRequest completeMultipartUploadRequest =
                new CompleteMultipartUploadRequest(ossTemplate.getBucketName(), key, uploadId, partETagList);
        try {
            final CompleteMultipartUploadResult result = ossTemplate.getOssClient().completeMultipartUpload(completeMultipartUploadRequest);
            if (!OssUtils.successful(result)){
                log.error("oss分片上传合成失败。objectName={} uploadId={} msg={}",key,uploadId, OssUtils.getErrorMessage(result));
                return false;
            }
            return success = closed = true;
        }catch (Exception e){
            log.error("oss分片上传合成失败。objectName={} uploadId={}",key,uploadId,e);
            return false;
        }
    }

    @Override
    public void cancel() {
        if (closed){
            return;
        }
        //not start
        if (!created && StringUtils.isBlank(uploadId)){
            closed = true;
            return;
        }
        try {
            AbortMultipartUploadRequest request = new AbortMultipartUploadRequest(ossTemplate.getBucketName(),key,uploadId);
            ossTemplate.getOssClient().abortMultipartUpload(request);
            closed =true;
        }catch (Exception e){
            log.error("oss取消分片上传失败。objectName={} uploadId={}",key,uploadId);
        }
    }

    @Override
    public String getFullUrl() {
        if (success){
           return ossTemplate.getUrl(this.key);
        }
        return null;
    }

}
