package cn.com.duiba.kjy.base.reactive.util;

import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpCookie;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.server.ServerRequest;

import java.net.InetSocketAddress;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @author dugq
 * @date 2021/3/15 5:23 下午
 * 覆盖 tomcat中httpRequest 的通用方法
 */
public final class ReactRequestTool {

    private ReactRequestTool(){}


    private static final String UNKNOWN = "unknown";
    private static final String IPHONE = "iphone";
    private static final String ANDROID = "android";
    private static final String IOS = "ios";
    private static final String IPAD = "ipad";
    private static final String MACINTOSH = "macintosh";
    private static final String LINUX = "linux";
    private static final String WINDOWS = "windows";


    /**
     * OS map
     */
    private static Map<String,String> osMap = new LinkedHashMap<>();
    static {
        osMap.put(IPHONE,"iPhone");
        osMap.put(ANDROID,"Android");
        osMap.put(IPAD,"iPad");
        osMap.put(MACINTOSH,"Mac");
        osMap.put(WINDOWS,"Windows");
        osMap.put(LINUX,"Linux");
    }

    /**
     * 获得User-Agent
     * @param req
     * @return
     */
    public static String getUserAgent(ServerRequest req) {
        if (req != null) {
            return req.headers().firstHeader("User-Agent");
        }
        return null;
    }

    public static String getUserAgent(ServerHttpRequest req) {
        if (req != null) {
            return req.getHeaders().getFirst("User-Agent");
        }
        return null;
    }

    /**
     * 获得完整的url（包含请求url中的参数，但是不包含post提交的参数部分）
     * @param request
     * @return
     */
    public static String getFullRequestUrl(ServerRequest request) {
        String url = request.uri().getPath();
        String queryString = request.uri().getQuery();
        if (queryString == null) {
            return url;
        } else {
            return url  +"?"+ queryString;
        }
    }

    public static String getFullRequestUrl(ServerHttpRequest request) {
        String url = request.getURI().getPath();
        String queryString = request.getURI().getQuery();

        if (queryString == null) {
            return url;
        } else {
            return url  +"?"+ queryString;
        }
    }



    /**
     * 获取用户的ip地址
     * @param request
     * @return
     */
    public static String getIpAddr(ServerRequest request) {
        String ip = request.headers().firstHeader("x-forwarded-for");
        if (ip != null && ip.trim().length() > 0) {
            String[] ips = ip.trim().split(",");
            int size = ips.length;
            if (size > 0) {
                ip = ips[0].trim();
            }
        }
        if (isInvalidIp(ip)) {
            ip = request.headers().firstHeader("X-Real-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.headers().firstHeader("Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.headers().firstHeader("WL-Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.headers().firstHeader("Cdn-Src-Ip");
        }
        if (isInvalidIp(ip)) {
            ip = request.remoteAddress().map(InetSocketAddress::getHostName).orElse(null);
        }
        if (ip != null && ip.startsWith("0:0:0:0")) {
            ip = "127.0.0.1";//NOSONAR
        }
        return ip;
    }

    public static String getIpAddr(ServerHttpRequest request) {
        String ip = request.getHeaders().getFirst("x-forwarded-for");
        if (ip != null && ip.trim().length() > 0) {
            String[] ips = ip.trim().split(",");
            int size = ips.length;
            if (size > 0) {
                ip = ips[0].trim();
            }
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("X-Real-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("WL-Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("Cdn-Src-Ip");
        }
        if (isInvalidIp(ip)) {
            ip = Objects.isNull(request.getRemoteAddress())?null:request.getRemoteAddress().getHostName();
        }
        if (ip != null && ip.startsWith("0:0:0:0")) {
            ip = "127.0.0.1";//NOSONAR
        }
        if (StringUtils.equals("localhost",ip)){
            ip = "127.0.0.1";//NOSONAR
        }
        return ip;
    }

    private static boolean isInvalidIp(String ip){
        return ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip);
    }

    /**
     * 是否经过了HTTP代理
     * @param request
     * @return
     */
    public static boolean isProxy(ServerRequest request) {
        //代理服务器ip
        String httpVia = request.headers().firstHeader("HTTP_VIA");
        if(httpVia != null && !"".equals(httpVia)){
            return true;
        }

        String xForwardedFor = request.headers().firstHeader("x-forwarded-for");
        if (xForwardedFor != null && xForwardedFor.trim().length() > 0) {
            String[] ips = xForwardedFor.trim().split(",");
            long sizeOfForwardFor = ips.length;
            if(sizeOfForwardFor > 1){
                return true;
            }
        }

        return false;
    }

    /**
     * 获取操作系统类型： Android,iPhone,iPad,Mac,Windows,Linux，Unknown
     *
     * @param request
     * @return
     */
    public static String getOS(ServerHttpRequest request) {
        String ua = getUserAgent(request);
        return getOS(ua);
    }

    public static String getOS(ServerRequest request) {
        String ua = getUserAgent(request);
        return getOS(ua);
    }

    /**
     * 获取操作系统类型： Android,iPhone,iPad,Mac,Windows,Linux，Unknown
     *
     * @return
     */
    public static String getOS(String ua) {
        if (ua == null) {
            return UNKNOWN;
        }
        String ua1 = ua.toLowerCase();
        for (Map.Entry<String,String> entry : osMap.entrySet()){
            if (ua1.contains(entry.getKey())){
                return entry.getValue();
            }
        }
        return UNKNOWN;
    }

    /**
     *
     * 获取简洁ua,对ios,mac,ipad合并为ios，返回ios/android/unknown
     *
     * @param request
     * @return
     * @since JDK 1.6
     */
    public static String getSimpleOs(ServerHttpRequest request){
        String ua = getUserAgent(request);
        if(ua != null){
            String ua1=ua.toLowerCase();
            if(ua1.contains(IPAD) || ua1.contains(IPHONE) || ua1.contains(MACINTOSH)){
                return IOS;
            }else if(ua1.contains(ANDROID)){
                return ANDROID;
            }
        }
        return UNKNOWN;
    }

    public static String getSimpleOs(ServerRequest request){
        String ua = getUserAgent(request);
        if(ua != null){
            String ua1=ua.toLowerCase();
            if(ua1.contains(IPAD) || ua1.contains(IPHONE) || ua1.contains(MACINTOSH)){
                return IOS;
            }else if(ua1.contains(ANDROID)){
                return ANDROID;
            }
        }
        return UNKNOWN;
    }

    /**
     * 获取操作系统类型：
     * 	Android,iPhone,iPad,Mac,Windows,Linux
     * @param request
     * @return
     */
    public static String getOSNew(ServerRequest request){
        String ua = getUserAgent(request);
        return getOSNew(ua);
    }

    /**
     * 获取操作系统类型，返回数字类型：4：iPad，1：iPhone，0：Android，5：Linux，2：Windows,3：Mac，6：Unknown，
     * 	Android,iPhone,iPad,Mac,Windows,Linux
     * @return
     */
    public static String getOSNew(String ua){
        //Unknown
        if(ua==null){
            return "6";
        }
        String ua1=ua.toLowerCase();
        if(ua1.contains("ipad")){
            //iPad
            return "4";
        }else if(ua1.contains("iphone")){
            //iPhone
            return "1";
        }else if(ua1.contains("android")){
            //Android
            return "0";
        }else if(ua1.contains("linux")){
            //Linux
            return "5";
        }else if(ua1.contains("windows")){
            //Windows
            return "2";
        }else if(ua1.contains("macintosh")){
            //Mac
            return "3";
        }else if(ua1.contains("ios")){
            //ios
            return "6";
        }
        return "6";
    }

    /**
     * 判断来源ip是不是本地ip(126.0.0.1)
     * @param request
     * @return
     */
    public static boolean isLocalRequest(ServerRequest request) {
        String ip = getIpAddr(request);
        if (ip.startsWith("127.0.0.1") || ip.startsWith("0:0:0:0")) {//NOSONAR
            return true;
        }
        return false;
    }


    /**
     * 获取指定的cookie
     * @param request
     * @param name
     * @return
     */
    public static String getCookie(ServerRequest request, String name) {
        final MultiValueMap<String, HttpCookie> cookies = request.cookies();
        final HttpCookie cookiesFirst = cookies.getFirst(name);
        if (Objects.isNull(cookiesFirst)){
            return null;
        }
        return cookiesFirst.getValue();
    }

    public static String getCookie(ServerHttpRequest request, String name) {
        final MultiValueMap<String, HttpCookie> cookies = request.getCookies();
        final HttpCookie cookiesFirst = cookies.getFirst(name);
        if (Objects.isNull(cookiesFirst)){
            return null;
        }
        return cookiesFirst.getValue();
    }

    /**
     * 判断请求来源是否是ios系统
     *
     * @return
     */
    public static boolean isIos(ServerRequest request){
        if (request==null) {
            return false;
        }
        String ua = getUserAgent(request);
        if(ua==null){
            return false;
        }
        ua = ua.toLowerCase();
        return ua.contains(IPHONE) || ua.contains(IPAD)|| ua.contains(MACINTOSH);
    }

    /**
     * 是否是android系统
     * @return
     */
    public static boolean isAndroid(ServerRequest request){
        if (request==null) {
            return false;
        }
        String ua = getUserAgent(request);
        if(ua==null){
            return false;
        }
        ua = ua.toLowerCase();
        return ua.contains(ANDROID);
    }

    /**
     * 从request header获取referer头
     *
     * @param request
     * @return
     */
    public static String getReferer(ServerRequest request) {
        if (request == null) {
            return null;
        }
        return request.headers().firstHeader("referer");
    }

    public static String getReferer(ServerHttpRequest request) {
        if (request == null) {
            return null;
        }
        return request.getHeaders().getFirst("referer");
    }

}
