package cn.com.duibaboot.kjj.oss.template;

import cn.com.duibaboot.kjj.oss.template.operation.MultipartTask;
import cn.com.duibaboot.kjj.oss.template.operation.MultipartTaskImpl;
import cn.com.duibaboot.kjj.oss.template.support.MultiUploadSupport;
import cn.com.duibaboot.kjj.oss.template.util.OssUtils;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PutObjectResult;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;

/**
 * @author dugq
 * @date 2021/7/21 4:03 下午
 */
@Data
@Slf4j
public abstract class AbstractOssTemplate implements OssTemplate, InitializingBean {
    /**
     * OSS 客户端
     */
    private OSSClient ossClient;

    /**
     * bucket的端点，也是访问域名。
     */
    private String endPoint;

    /**
     * 当前客户端操作的分区名称
     */
    private String bucketName;

    /**
     * bucket的加速访问域名
     */
    protected String cdnDomain;

    /**
     * bucket的加速域名，不带协议等
     */
    private String cdnHost;

    /**
     * bucket的访问域名： http://bucketName.endPoint
     */
    private String bucketDomain;

    @Resource
    private ExecutorService ossExecutorService;

    @Override
    public void afterPropertiesSet() throws Exception {
        URI uri = URI.create(endPoint);
        bucketDomain = uri.getScheme()+"://"+bucketName+"."+uri.getHost();
        if (StringUtils.isNotBlank(cdnDomain)){
            URI uri2 = URI.create(cdnDomain);
            cdnHost = uri2.getHost();
        }
    }

    @Override
    public MultipartTask createMultipartTask() {
        return new MultipartTaskImpl(this);
    }

    @Override
    public String simpleMultipartUpload(InputStream inputStream, String objectName, ObjectMetadata meta) {
        final MultipartTask multipartTask = createMultipartTask();
        try {
            final boolean result = MultiUploadSupport.simpleMultipartUpload(multipartTask, inputStream, objectName, meta);
            if (result){
                getUrl(objectName);
            }
            return null;
        } catch (IOException e) {
            log.error("上传oss发生错误，输入流读取失败 objectName={}",objectName,e);
            return null;
        }
    }

    @Override
    public <R> R downloadStream(String objectName, @NotNull Function<OSSObject, R> callback) {
        if(StringUtils.isBlank(objectName)){
            return null;
        }
        String key;
        URI uri = URI.create(objectName);
        //兼容传入的是文件地址的场景，对域名进行校验后，再进行下载操作
        if (Objects.nonNull(uri.getHost())){
            // 1、cdn加速域名拼接的地址
            if (cdnDomain.contains(uri.getHost())){
                //objectName要求不能以/开头
                key = uri.getPath().replaceFirst("/","");
            }
            // 2、oss endpoint拼接的地址。同时也包含对bucket的验证
            else if (bucketDomain.contains(uri.getHost())){
                key = uri.getPath().replaceFirst("/","");
            }
            // 3、其他一律不允许下载
            else{
                log.error("非法的Oss下载地址：{}.  this template cdnDomain={} endpoint={} bucket={}",objectName,cdnDomain,endPoint,bucketName);
                return null;
            }

        }else{
            //没有加域名的，默认直接进行下载
            key = objectName;
        }
        OssUtils.validatorObjectName(key);
        final OSSObject object = ossClient.getObject(bucketName, key);
        try {
            return callback.apply(object);
        }finally {
            final InputStream objectContent = object.getObjectContent();
            try {
                objectContent.close();
            } catch (IOException e) {
                //ignore.如论如何，都得把inputStream关闭了。
            }
        }
    }



    @Override
    public String uploadStream(InputStream inputStream, String objectName, ObjectMetadata meta) {
        OssUtils.validatorObjectName(objectName);
        final PutObjectResult result = ossClient.putObject(bucketName, objectName, inputStream, meta);
        if (OssUtils.successful(result)){
            return getUrl(objectName);
        }
        log.error("上传oss发生错误.object name = {} msg = {}",objectName,OssUtils.getErrorMessage(result));
        return null;
    }

    @Override
    public String uploadFile(File file, String objectName, ObjectMetadata meta) {
        OssUtils.validatorObjectName(objectName);
        final PutObjectResult result = ossClient.putObject(bucketName, objectName, file, meta);
        if (OssUtils.successful(result)){
            return getUrl(objectName);
        }
        log.error("上传oss发生错误.object name = {} msg = {}",objectName,OssUtils.getErrorMessage(result));
        return null;
    }

    @Override
    public String uploadString(String content, String objectName, ObjectMetadata meta) {
        if (StringUtils.isBlank(content)){
            log.error("上传oss发生错误:文件内容为空.object name = {}",objectName);
            return null;
        }
        final byte[] bytes = content.getBytes();
        return uploadStream(new ByteArrayInputStream(bytes),objectName,meta);
    }

    protected String getBucketUri(){
        return bucketDomain+"/";
    }

    @Override
    public boolean doesObjectExist(String objectName) {
        return ossClient.doesObjectExist(bucketName, objectName);
    }

    @Override
    public String getUrlWithOutScheme(String objectName) {
        return getUrl(objectName).replace("https:", "").replace("http:", "");
    }

    @Override
    public String getCDNHost() {
        return cdnHost;
    }
}
