package cn.com.duibaboot.kjj.oss.template.operation;

import cn.com.duibaboot.kjj.oss.template.metadata.ObjectMetadataHelper;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PartETag;

import java.io.InputStream;
import java.util.List;

/**
 * <h1>分片上传&断点续传</h1>
 *  e.g:
 *   <li>分片上传</li>
 *   <pre>
 *  MultipartTask task = ossTemplate.createMultipartTask();
 *  task.startMultipartPartUpload("objectName",ObjectMetadataHelper.create().getInstance());
 *  for(int i =0 ;i < 10; i++){
 *    task.multipartPartUpload(in,1024,i);
 *      }
 *  task.multipartUpload();
 *  String ossUrl = task.getFullUrl();
 *  </pre>
 *   <li>断点上传</li>
 *   <ul>
 *       <li>1.创建任务</li>
 *   <code>
 *    MultipartTask sourceTask = ossTemplate.createMultipartTask();
 *    sourceTask.startMultipartPartUpload("objectName",ObjectMetadataHelper.create().getInstance());
 *   </code>
 *        <li>2.断点续传</li>
 *   <code>
 *    MultipartTask task = ossTemplate.createMultipartTask();
 *    task.startContinuePartUpload(sourceTask.getUploadId(),"objectName");
 *    task.multipartPartUpload(in,1024,partNum);
 *    List<PartETag> successTags = task.getSuccessfulUploadPart();
 *   </code>
 *         <li>3.完成断点续传</li>
 *   <code>
 *    MultipartTask task = ossTemplate.createMultipartTask();
 *    task.startContinuePartUpload(sourceTask.getUploadId(),"objectName");
 *    task.multipartUpload(successTags);
 *   </code>
 *   </ul>
 * 我们暂不开启大文件分片上传时的异步上传。
 * 因为我们无法判断客户端调用时的文件大小、分片数量等引起的文件流到底有多大。这可能会给服务器的网络IO造成困扰
 * 当然在某些独立的服务器上或者特殊的业务，通过您自己的判断，确实需要时，您可以自行实现异步多线程同时上传多个分片。
 */
public interface MultipartTask{

    /**
     * 开始分片上传，创建任务ID
     * 定义 对象 名称，格式等
     * @param objectName 对象名称（key）
     * @param objectMetadata 对象元空间。{@link ObjectMetadataHelper}
     * @return 任务是否创建成功
     */
    boolean startMultipartPartUpload(String objectName, ObjectMetadata objectMetadata);

    /**
     * 创建一个断点续传任务
     * @param uploadId 任务ID
     * @param objectName 对象名称
     * @return 是否创建成功。失败主要是因为对象复用
     */
    boolean startContinuePartUpload(String uploadId,String objectName);

    /**
     * 分片上传
     * 将文件按需切割成多个分片后，每个分片逐个上传。再此过程，您可以开启多线程并发上传。
     * 每个文件的必须大于100K小于5GB。我们希望您认真考虑如何切片。
     * 分片最大数量为1万。
     * 分片大小 * 最大分片数量 = 文件最大长度。 e.g: partSize=1M => maxSize=10GB
     * 一般文件不大的情况下我们建议您设置再1-100MB
     * 对于上传失败的，您可以选择再此上传的。
     *
     * @param inputStream 分片流
     * @param partSize  流的大小。 决定oss本次上传开辟多少空间存储。除了最后一个分片以外，其他的必须 > 100*1024 (100KB)
     * @param partNumber 分片序号。 决定最终合成时的顺序。取值范围是1~10000.
     */
    boolean multipartPartUpload(InputStream inputStream,Long partSize, int partNumber);

    /**
     * 获取已经上传成功的分片
     * @return 分片列表
     */
    List<PartETag> getSuccessfulUploadPart();

    /**
     * 完成所有分片上传
     */
    boolean multipartUpload();

    /**
     * 完成断点续传
     */
    boolean multipartUpload(List<PartETag> tags);

    /**
     * 取消本次任务。不支持断点续传时，发生上传错误后，必须调用cancel
     */
    void cancel();

    /**
     * 分片上传完成后，我们可以通过此方法获取到oss的完整地址
     * @return null:上传未完成或者失败
     */
    String getFullUrl();

    /**
     * 获取uploadId
     * @return uploadId
     */
    String getUploadId();
}
