package cn.com.duibaboot.kjj.oss.template;

import cn.com.duibaboot.kjj.oss.template.metadata.ObjectMetadataHelper;
import cn.com.duibaboot.kjj.oss.template.operation.MultipartTask;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectMetadata;

import javax.annotation.Nullable;
import java.io.File;
import java.io.InputStream;
import java.util.function.Function;

/**
 * <h1>OSS操作封装类</h1>
 *
 *  <h3>特性</h3>
 *  <li>OSS：面向对象存储。把每个文件当成一个对象看待，文件的path就是ObjectName，objectName不能以/开头。这样它才像object，而不是path</li>
 *  <li>OSS Bucket: 所有的对象按桶分组，实现业务隔离</li>
 *  <li>文件的URL：OSS-endpoint/bucketName/objectName</li>
 *  <li>本对象封装object的CURD操作，但屏蔽掉了bucket的相关操作</li>
 *  <li>OSS的每个bucket都可以开启CDN加速功能，开启后的文件地址：CDNDomain/objectName </li>
 *  <li>通过CDN缓存加快文件的访问速度，同时也造成了文件缓存，对于文件的修改、删除生效时间不详。受地域和访问情况影响</li>
 *
 *  <h3>功能</h3>
 *  <ul>
 *  <li>上传文件</li>
 *   <ul>
 *       <li>通过流上传。支持字节流，字符流</li>
 *       <li>上传字符串。内部转化为字节流上传。方便使用</li>
 *       <li>大文件分片上传{@link MultipartTask}</li>
 *       <li>追加文件内容。可扩展为断点续传</li>
 *       <li>异步上传</li>
 *   </ul>
 *  </ul>
 * <li>下载文件</li>
 * <li>查看文件是否存在</li>
 * <li>删除文件</li>
 *
 * <h3>配置{@link cn.com.duibaboot.kjj.oss.conf.OssProps}</h3>
 *
 *
 * @author dugq
 * @date 2021/7/21 3:52 下午
 */
public interface OssTemplate {

    /**
     * 通过流 上传到oss
     * @param inputStream 输入流
     * @param objectName 文件key
     * @param meta 对象元空间 {@link ObjectMetadataHelper}
     * @return 完整访问路径地址 null:上传失败
     */
    @Nullable
    String uploadStream(InputStream inputStream, String objectName, ObjectMetadata meta);

    /**
     * 上传字符串到oss
     * @param content 上传的字符串
     * @param objectName 文件key
     * @param meta 对象元空间 {@link ObjectMetadataHelper}
     * @return 完整访问路径地址 null:上传失败
     */
    @Nullable
    String uploadString(String content,String objectName, ObjectMetadata meta);

    /**
     * 创建一个分片上传任务
     * 如果只是对一个文件流进行简单的分片上传，可使用下面这个方法
     * @return 新的任务
     */
    MultipartTask createMultipartTask();

    /**
     * 获取下载流
     * @param objectName 文件对象的地址。不能以/开头。当传入的是文件的地址时，会进行一次域名校验，不通过的不予下载
     * @param callback 本方法处理文件流的获取以及释放，文件流的处理需要业务方自己完成。所以我们不接受在callback中再异步处理inputStream。
     *                 模版操作见：{@link cn.com.duibaboot.kjj.oss.template.support.DownloadHelper}
     * @return callback的返回值
     */
    <R> R downloadStream(String objectName, Function<OSSObject,R> callback);

    /**
     * 对象访问地址
     * 有CDN返回CDN地址，私有bucket和没有CDN加速域名的client返回：endpoint+bucket+objectName
     * 注意：私有bucket返回的地址是不可以直接访问的，只能通过本类进行下载。
     * @param objectName 文件地址，支持附带域名的情况，智能补全。
     * @return 对象访问地址
     */
     String getUrl(String objectName);

    /**
     * 从ossUrl中提取出objectName
     * @param ossUrl URL
     * @return objectName null: 非法的URL
     */
     String getObjectName(String ossUrl);

    /**
     * 同上，返回不带http等协议的URL
     * @param objectName key
     * @return 对象访问地址
     */
    String getUrlWithOutScheme(String objectName);


    /**
     * 文件是否存在
     *
     * @param objectName 文件名称。为全路径时会额外验证域名预计bucket的正确性
     * @return 是否存在
     */
     boolean doesObjectExist(String objectName);

    /**
     * 通过文件 上传到oss
     * @param file 输入文件
     * @param objectName 文件key
     * @param meta 对象元空间 {@link ObjectMetadataHelper}
     * @return 完整访问路径地址 null:上传失败
     */
    @Nullable
    String uploadFile(File file, String objectName, ObjectMetadata meta);

    /**
     * 获取cdnHost
     * @return cdnHost
     */
    String getCDNHost();
}
