package cn.com.duiba.kjy.base.customweb.web.bean;

import cn.com.duiba.kjy.base.customweb.exception.ResponseClosedException;
import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaderValues;
import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.util.AsciiString;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.nio.charset.Charset;

/**
 * @author dugq
 * @date 2021/3/25 11:05 上午
 * response的本身属于单个请求，所以并未做线程安全处理
 */
@Data
@Slf4j
public class KjjHttpResponse {

    private final ChannelHandlerContext ctx;

    private final FullHttpRequest httpRequest;

    private boolean close;

    private final ByteBuf content;

    private final FullHttpResponse response;

    public KjjHttpResponse(ChannelHandlerContext ctx, FullHttpRequest httpRequest) {
        this.ctx = ctx;
        this.httpRequest = httpRequest;
        this.content = ctx.alloc().buffer();
        response = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.OK,this.content);
    }

    public boolean isClosed(){
        return close;
    }


    public void addCookie(Cookie cookie){
        validator();
        addHeader(HttpHeaderNames.SET_COOKIE,cookie);
    }

    public void addHeader(HttpHeaders headers){
        validator();
        response.headers().add(headers);
    }

    /**
     * @param name 请从{@link io.netty.handler.codec.http.HttpHeaderNames} 常量池中获取
     * @param value value
     */
    public void addHeader(AsciiString name, Object value){
        validator();
        response.headers().add(name,value);
    }

    public ByteBuf writeBoolean(boolean value) {
        validator();
        return content.writeBoolean(value);
    }

    public ByteBuf writeByte(int value) {
        validator();
        return content.writeByte(value);
    }

    public ByteBuf writeShort(int value) {
        validator();
        return content.writeShort(value);
    }

    public ByteBuf writeInt(int value) {
        validator();
        return content.writeInt(value);
    }

    public ByteBuf writeLong(long value) {
        validator();
        return content.writeLong(value);
    }

    public ByteBuf writeChar(int value) {
        validator();
        return content.writeChar(value);
    }

    public ByteBuf writeFloat(float value) {
        validator();
        return content.writeFloat(value);
    }


    public ByteBuf writeDouble(double value) {
        validator();
        return content.writeDouble(value);
    }

    public ByteBuf writeBytes(byte[] src) {
        validator();
        return content.writeBytes(src);
    }

    public int writeCharSequence(CharSequence sequence, Charset charset) {
        validator();
        return content.writeCharSequence(sequence, charset);
    }

    /**
     * write 不支持序列化，如果需要写入对象，请引用{@link cn.com.duiba.kjy.base.customweb.web.handler.response.impl.JacksonHandler}进行序列化
     * 在写完之后，需要手动调用{@link #flushAndClose}方法进行结束response流
     * @param response
     */
    public void write(String response){
        validator();
        this.content.writeCharSequence(response, Charset.defaultCharset());
    }

    /**
     *  写完数据，关闭response
     */
    public void flushAndClose(){
        if (close){
            return;
        }
        close = true;
        //http 协议规范，必须在response header中正确的返回 responsebody的大小。否则标准浏览器无法解析
        response.headers().set(HttpHeaderNames.CONTENT_LENGTH, response.content().readableBytes());
        if (httpRequest.headers().containsValue(HttpHeaderNames.CONNECTION, HttpHeaderValues.KEEP_ALIVE, true)) {
            //保持TCP链接，写入http response
            response.headers().set(HttpHeaderNames.CONNECTION, HttpHeaderValues.KEEP_ALIVE);
            ctx.writeAndFlush(response);
        } else {
            //往通道中写入数据，写完立刻关闭通道
            ctx.writeAndFlush(response).addListener(ChannelFutureListener.CLOSE);
        }
    }

    private void validator() {
        if (close){
            throw new ResponseClosedException();
        }
    }

    public void setContentType(String contentType) {
        addHeader(HttpHeaderNames.CONTENT_TYPE,contentType);
    }

    public void setStatus(HttpResponseStatus status) {
        response.setStatus(status);
    }

}
