package cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller;

import cn.com.duiba.kjy.base.customweb.web.bean.KjjHttpRequest;
import cn.com.duiba.kjy.base.customweb.web.condition.PatternsRequestCondition;
import cn.com.duiba.kjy.base.customweb.web.condition.RequestCondition;
import cn.com.duiba.kjy.base.customweb.web.condition.RequestMethodsRequestCondition;
import org.springframework.lang.Nullable;
import org.springframework.util.PathMatcher;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import java.util.Set;

/**
 * @author dugq
 * @date 2021/4/1 4:42 下午
 */
public class RequestMappingInfo implements RequestCondition<RequestMappingInfo> {

    @Nullable
    private final String name;

    private final PatternsRequestCondition patternsCondition;

    private final RequestMethodsRequestCondition methodCondition;


    public RequestMappingInfo(@Nullable String name, @Nullable PatternsRequestCondition patterns, RequestMethodsRequestCondition methodCondition) {

        this.name = (StringUtils.hasText(name) ? name : null);
        this.patternsCondition = (patterns != null ? patterns : new PatternsRequestCondition());
        this.methodCondition = methodCondition;
    }

    /**
     * Creates a new instance with the given request conditions.
     */
    public RequestMappingInfo(@Nullable PatternsRequestCondition patterns, RequestMethodsRequestCondition methodCondition) {
        this(null, patterns,methodCondition);
    }

    /**
     * Re-create a RequestMappingInfo with the given custom request condition.
     */
    public RequestMappingInfo(RequestMappingInfo info) {
        this(info.name, info.patternsCondition,info.methodCondition);
    }


    /**
     * Return the name for this mapping, or {@code null}.
     */
    @Nullable
    public String getName() {
        return this.name;
    }

    public PatternsRequestCondition getPatternsCondition() {
        return this.patternsCondition;
    }

    /**
     * Combine "this" request mapping info (i.e. the current instance) with another request mapping info instance.
     * <p>Example: combine type- and method-level request mappings.
     * @return a new request mapping info instance; never {@code null}
     */
    @Override
    public RequestMappingInfo combine(RequestMappingInfo other) {
        String name = combineNames(other);
        PatternsRequestCondition patterns = this.patternsCondition.combine(other.patternsCondition);
        RequestMethodsRequestCondition methods = this.methodCondition.combine(other.methodCondition);
        return new RequestMappingInfo(name, patterns,methods);
    }

    @Nullable
    private String combineNames(RequestMappingInfo other) {
        if (this.name != null && other.name != null) {
            String separator = "#";
            return this.name + separator + other.name;
        }
        else if (this.name != null) {
            return this.name;
        }
        else {
            return other.name;
        }
    }

    /**
     * Checks if all conditions in this request mapping info match the provided request and returns
     * a potentially new request mapping info with conditions tailored to the current request.
     * <p>For example the returned instance may contain the subset of URL patterns that match to
     * the current request, sorted with best matching patterns on top.
     * @return a new instance in case all conditions match; or {@code null} otherwise
     */
    @Override
    public boolean getMatchingCondition(KjjHttpRequest request) {
        final boolean urlMath = this.patternsCondition.getMatchingCondition(request);
        if (!urlMath) {
            return false;
        }
        return this.methodCondition.getMatchingCondition(request);
    }

    @Override
    public int compareTo(RequestMappingInfo other, KjjHttpRequest request) {
        int result;
        result = this.patternsCondition.compareTo(other.getPatternsCondition(), request);
        if (result != 0) {
            return result;
        }
        return 0;
    }

    @Override
    public boolean equals(@Nullable Object other) {
        if (this == other) {
            return true;
        }
        if (!(other instanceof RequestMappingInfo)) {
            return false;
        }
        RequestMappingInfo otherInfo = (RequestMappingInfo) other;
        return (this.patternsCondition.equals(otherInfo.patternsCondition) );
    }

    @Override
    public int hashCode() {
        return (this.patternsCondition.hashCode() * 31);
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder("{");
        if (!this.patternsCondition.isEmpty()) {
            Set<String> patterns = this.patternsCondition.getPatterns();
            builder.append(" ").append(patterns.size() == 1 ? patterns.iterator().next() : patterns);
        }
        builder.append('}');
        return builder.toString();
    }


    /**
     * Create a new {@code RequestMappingInfo.Builder} with the given paths.
     * @param paths the paths to use
     * @since 4.2
     */
    public static Builder paths(String... paths) {
        return new DefaultBuilder(paths);
    }


    /**
     * Defines a builder for creating a RequestMappingInfo.
     * @since 4.2
     */
    public interface Builder {

        /**
         * Set the path patterns.
         */
        Builder paths(String... paths);

        /**
         * Set the request method conditions.
         */
        Builder methods(RequestMethod... methods);

        /**
         * Set the mapping name.
         */
        Builder mappingName(String name);

        /**
         * Provide additional configuration needed for request mapping purposes.
         */
        Builder options(BuilderConfiguration options);

        /**
         * Build the RequestMappingInfo.
         */
        RequestMappingInfo build();
    }


    private static class DefaultBuilder implements Builder {

        private String[] paths = new String[0];

        private RequestMethod[] methods = new RequestMethod[0];

        @Nullable
        private String mappingName;

        private BuilderConfiguration options = new BuilderConfiguration();

        public DefaultBuilder(String... paths) {
            this.paths = paths;
        }

        @Override
        public Builder paths(String... paths) {
            this.paths = paths;
            return this;
        }

        @Override
        public DefaultBuilder methods(RequestMethod... methods) {
            this.methods = methods;
            return this;
        }


        @Override
        public DefaultBuilder mappingName(String name) {
            this.mappingName = name;
            return this;
        }

        @Override
        public Builder options(BuilderConfiguration options) {
            this.options = options;
            return this;
        }

        @Override
        public RequestMappingInfo build() {
            PatternsRequestCondition patternsCondition = new PatternsRequestCondition(this.paths, this.options.getPathMatcher());

            return new RequestMappingInfo(this.mappingName, patternsCondition,new RequestMethodsRequestCondition(this.methods));
        }
    }


    /**
     * Container for configuration options used for request mapping purposes.
     * Such configuration is required to create RequestMappingInfo instances but
     * is typically used across all RequestMappingInfo instances.
     * @since 4.2
     * @see Builder#options
     */
    public static class BuilderConfiguration {

        @Nullable
        private PathMatcher pathMatcher;

        /**
         * Set a custom PathMatcher to use for the PatternsRequestCondition.
         * <p>By default this is not set.
         */
        public void setPathMatcher(@Nullable PathMatcher pathMatcher) {
            this.pathMatcher = pathMatcher;
        }

        /**
         * Return a custom PathMatcher to use for the PatternsRequestCondition, if any.
         */
        @Nullable
        public PathMatcher getPathMatcher() {
            return this.pathMatcher;
        }


    }
}
