package cn.com.duiba.kjy.base.reactive.config;

import cn.com.duiba.kjy.base.reactive.adaptor.CustomRequestMappingHandlerAdapter;
import cn.com.duiba.kjy.base.reactive.convert.HttpReader;
import cn.com.duiba.kjy.base.reactive.convert.IdArgumentResolver;
import cn.com.duiba.kjy.base.reactive.fitler.MonitorWebFilter;
import cn.com.duiba.kjy.base.reactive.interceptor.ReactiveInterceptor;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.web.mvc.CustomHttpMessageConverters;
import org.hibernate.validator.HibernateValidator;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.boot.autoconfigure.web.reactive.WebFluxRegistrations;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.reactive.config.WebFluxConfigurer;
import org.springframework.web.reactive.result.method.annotation.RequestMappingHandlerAdapter;

import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author dugq
 * @date 2021/3/18 7:11 下午
 */
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.REACTIVE)
@Configuration
public class ReactiveAutoConfiguration {

    /**
     * 自定义 reactive server config.
     * 使得ID自动解密生效，如若在系统中自定义config，务必继承此类。
     */
    @ConditionalOnMissingBean(WebFluxConfigurer.class)
    @Bean
    public ReactiveWebConfig reactiveWebConfig(){
        return new ReactiveWebConfig();
    }

    /**
     * 自定义参数解析器，用户处理ID自动解密并转换为Long类型。针对与Get方式
     */
    @Bean
    @Order(Integer.MIN_VALUE)
    public IdArgumentResolver getIdArgumentResolver(){
        return new IdArgumentResolver();
    }

    /**
     * 自定义参数解析器，用户处理ID自动解密并转换为Long类型。针对与Post方式
     */
    @Bean
    @Order(Integer.MIN_VALUE)
    public HttpReader getHttpReader() {
        return new HttpReader();
    }

    /**
     * 把HttpMessageConvertersAutoConfiguration中定义的HttpMessageConverters替换为使用自定义的HttpMessageConverters，以在最后增加fst和kryo的HttpMessageConverter.(顺序很重要，一定要在最后)
     * @return
     */
    @Bean
    public static SpecifiedBeanPostProcessor httpMessageConvertersPostProcessor(){
        return new SpecifiedBeanPostProcessor<HttpMessageConverters>(){

            @Override
            public int getOrder() {
                return 0;
            }

            @Override
            public Class<HttpMessageConverters> getBeanType() {
                return HttpMessageConverters.class;
            }

            @Override
            public Object postProcessBeforeInitialization(HttpMessageConverters bean, String beanName) throws BeansException {
                return bean;
            }

            @Override
            public Object postProcessAfterInitialization(HttpMessageConverters bean, String beanName) throws BeansException {
                List<HttpMessageConverter<?>> converters = bean.getConverters();
                return new CustomHttpMessageConverters(false, converters == null
                        ? Collections.emptyList() : converters);
            }
        };
    }

    /**
     * 重新注册http消息转换类。注意：此类是spring-web 框架使用的，实则适用tomcat的，但由于RPC也使用到，所以再加回来。
     * 有此类，上述的自定义converter才生效
     */
    @Bean
    @ConditionalOnMissingBean
    public HttpMessageConverters messageConverters(ObjectProvider<HttpMessageConverter<?>> converters) {
        return new HttpMessageConverters(converters.orderedStream().collect(Collectors.toList()));
    }

    /**
     * 重写 RequestMappingHandlerAdapter。类似spring-web 增加interceptor 功能。
     */
    @Bean
    @ConditionalOnBean(ReactiveInterceptor.class)
    public WebFluxRegistrations getWebFluxRegistrations(){
        return new WebFluxRegistrations() {
            @Override
            public RequestMappingHandlerAdapter getRequestMappingHandlerAdapter() {
                return new CustomRequestMappingHandlerAdapter();
            }
        };
    }

    /**
     * 此类为了容器心跳监控服务。同时zuul的心跳检测也是这里处理。
     */
    @Bean
    public MonitorWebFilter getMonitorWebFilter(){
        return new MonitorWebFilter();
    }

    /**
     * 配置 validation 为快速失败模式
     */
    @Bean
    public Validator validator(){
        ValidatorFactory validatorFactory = Validation.byProvider( HibernateValidator.class )
                .configure()
                .addProperty( "hibernate.validator.fail_fast", "true" )
                .buildValidatorFactory();
        return validatorFactory.getValidator();
    }
}
