package cn.com.duiba.kjy.base.api.queue;

import com.dianping.cat.Cat;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 队列批量消费抽象类
 * @author lizhi
 * @date 2020/4/5 11:40 AM
 */
@Slf4j
public abstract class AbstractQueueService<T> implements QueueConsumeService<T>, DisposableBean {


    private final BatchConsumeBlockingQueue<T> queue;

    /**
     * 构造一个支持压测、并发、批量消费的Queue
     * @param capacity BlockingQueue的最大允许容量, 如果<=0 视为无界队列
     * @param batchSize 一次批量消费的个数
     * @param maxWaitSize 一次批量消费的阈值，当已入队的数据个数达到此阈值时，触发批量消费（若消费回调阻塞，则无法消费）
     *                  如果此阈值 <= 0则不会触发个数阈值消费（这种情况下数据只会在时间到达maxWaitTimeMillis阈值后才会被消费，这意味着数据可能会在队列中存在比较长的时间）
     * @param maxWaitTimeMillis 数据最大延迟消费时间的阈值(ms)，当有数据在队列中存在时间超过此阈值时，触发批量消费（若消费回调阻塞，则无法消费）。
     *                          如果此阈值 <= 0则不会触发延时消费（这种情况下数据只会在堆积到达maxWaitSize阈值后才会被消费，这意味着数据可能会在队列中存在非常长的时间）
     */
    public AbstractQueueService(int capacity, int batchSize, int maxWaitSize, int maxWaitTimeMillis) {
        this.queue = new BatchConsumeBlockingQueue<>(capacity, batchSize, maxWaitSize, maxWaitTimeMillis, this::batchConsume);
        this.queue.start();
    }

    @Override
    public void addQueue(T t) {
        String simpleClassName = this.getClass().getSimpleName();
        Cat.logEvent("batchConsumerQueue", simpleClassName);
        try {
            queue.add(t);
        }catch (Throwable e){
            Cat.logEvent("batchConsumerQueue", simpleClassName,"fail","");
            throw  e;
        }
    }

    /**
     * 批量消费队列消息
     * @param list 消息集合
     */
    protected void batchConsume(List<T> list) {
        batchConsumeMsg(list);
    }

    /**
     * 批量消费队列消息
     * @param list 消息集合
     */
    protected abstract void batchConsumeMsg(List<T> list);

    /**
     * 避免容器停止时消息未处理完
     */
    @Override
    public void destroy() throws Exception {
        int sleepCnt = 1;
        while (!queue.isEmpty()) {
            if (sleepCnt++ > 3) {
                return;
            }
            TimeUnit.SECONDS.sleep(2);
        }
    }
}
