package cn.com.duiba.kjy.base.api.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;

/**
 * 加解密工具类
 * @author lizhi
 * @date 2021/12/10 3:13 下午
 */
@Slf4j
public class KjyEncryptUtil {

    private KjyEncryptUtil() {}

    private static final String AES_CIPHER_NAME = "AES/CBC/PKCS7Padding";

    private static final String AES_KEY_SPEC_NAME = "AES";

    /**
     * 加密：AES-128-CBC，PKCS#7填充
     * 1.偏移量：key MD5加密后，取前16位
     * 2.AES加密
     * 3.Base64编码
     * @param data 需要加密的数据
     * @param key 密钥
     * @return 加密后的数据
     * @throws IllegalStateException 构建Cipher失败、加密失败
     */
    public static String encodeByAesBase64(String data, String key) {
        if (StringUtils.isAnyBlank(data, key)) {
            return null;
        }
        String iv = StringUtils.substring(DigestUtils.md5Hex(key), 0,16);
        byte[] bytes = encodeByAes(data.getBytes(StandardCharsets.UTF_8), key.getBytes(StandardCharsets.UTF_8), iv.getBytes(StandardCharsets.UTF_8));
        return Base64.encodeBase64String(bytes);
    }

    /**
     * 解密：AES-128-CBC，PKCS#7填充
     * 1.偏移量：key MD5加密后，取前16位
     * 2.Base64解码
     * 3.AES解密
     * 4.将解密后的字节数组转成字符串
     * @param data 需要解密的数据
     * @param key 密钥
     * @return 解密后的数据
     * @throws IllegalStateException 构建Cipher失败、解密失败
     */
    public static String decodeByBase64Aes(String data, String key) {
        if (StringUtils.isAnyBlank(data, key)) {
            return null;
        }
        String iv = StringUtils.substring(DigestUtils.md5Hex(key), 0,16);
        byte[] dataBytes = Base64.decodeBase64(data);
        byte[] bytes = decodeByAes(dataBytes, key.getBytes(StandardCharsets.UTF_8), iv.getBytes(StandardCharsets.UTF_8));
        return new String(bytes);
    }

    /**
     * 加密：AES-128-CBC，PKCS#7填充
     * @param dataBytes 需要加密的数据字节数组
     * @param keyBytes 密钥字节数组
     * @param ivBytes 偏移量字节数组
     * @return 加密后的数据字节数组
     * @throws IllegalStateException 构建Cipher失败、加密失败
     */
    public static byte[] encodeByAes(@Nonnull byte[] dataBytes, @Nonnull byte[] keyBytes, @Nonnull byte[] ivBytes) {
        Cipher cipher = generateCipher(keyBytes, ivBytes, AES_KEY_SPEC_NAME, AES_CIPHER_NAME, true);
        try {
            return cipher.doFinal(dataBytes);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * 解密：AES-128-CBC，PKCS#7填充
     * @param dataBytes 需要解密的数据字节数组
     * @param keyBytes 密钥字节数组
     * @param ivBytes 偏移量字节数组
     * @return 解密后的数据字节数组
     * @throws IllegalStateException 构建Cipher失败、解密失败
     */
    public static byte[] decodeByAes(@Nonnull byte[] dataBytes, @Nonnull byte[] keyBytes, @Nonnull byte[] ivBytes) {
        Cipher cipher = generateCipher(keyBytes, ivBytes, AES_KEY_SPEC_NAME, AES_CIPHER_NAME, false);
        try {
            return cipher.doFinal(dataBytes);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    private static Cipher generateCipher(byte[] keyBytes, byte[] ivBytes, String keySpecName, String cipherName, boolean encrypt) {
        SecretKeySpec secretKeySpec=new SecretKeySpec(keyBytes, keySpecName);
        IvParameterSpec ivParameterSpec= new IvParameterSpec(ivBytes);
        try {
            Cipher cipher = Cipher.getInstance(cipherName);
            if (encrypt) {
                cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, ivParameterSpec);
            } else {
                cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, ivParameterSpec);
            }
            return cipher;
        }catch(Exception e){
            log.error("SignUtil, generate cipher error, key={}, iv={}, keySpecName={}, cipherName={}, encrypt={}", new String(keyBytes), new String(ivBytes), keySpecName, cipherName, encrypt, e);
            throw new IllegalStateException(e);
        }
    }
}
