package cn.com.duibaboot.kjj.filter;

import cn.com.duiba.boot.perftest.PerfTestUtils;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.wolf.utils.SecurityUtils;
import cn.com.duibaboot.ext.autoconfigure.core.utils.HttpRequestUtils;
import cn.com.duibaboot.ext.autoconfigure.logger.filter.LoggerSizeFilter;
import cn.com.duibaboot.ext.autoconfigure.web.wrapper.BodyReaderHttpServletRequestWrapper;
import cn.com.duibaboot.ext.autoconfigure.web.wrapper.BodyWriterHttpServletResponseWrapper;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.EnumerationUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.AntPathMatcher;

import javax.annotation.Resource;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <h1>accessLog收集打印类</h1>
 *
 * 该类修改自中台提供的日志处理类：{@link cn.com.duibaboot.ext.autoconfigure.accesslog.AccessLogFilter }
 *
 *
 * <h2>使用说明</h2>
 *  中台提供的类默认加载，如果需要使用本自定义的类，请在Application.java中移除{@link cn.com.duibaboot.ext.autoconfigure.accesslog.AccessLogAutoConfiguration}
 *  添加自动装配：{@link cn.com.duibaboot.kjj.filter.KjjAccessLogAutoConfiguration }
 *  <pre>
 *      e.g:
 *      {@code
 *        @SpringBootApplication(exclude = {AccessLogAutoConfiguration.class})
 *        @Import({KjjAccessLogAutoConfiguration.class})
 *      }
 *  </pre>
 *
 *  <h2>打印accessLog到application日志中</h2>
 *  通过属性{@code access.log.print.into.application}控制。默认开启
 *
 *  <h2>过滤请求不打印日志</h2>
 *  <li>host黑白名单模式</li>
 *  通过属性配置 accesslog.mode=black/white 设置黑白模式，默认黑名单模式<br/>
 *  通过属性 accesslog.hosts 添加域名配置，通过英文逗号隔开<br/>
 *  <li>url黑白名单模式</li>
 *  在域名过滤之外，也有URL的黑名单过滤，它不支持白名单。<br/>
 *  可通过属性 accesslog.blackpaths 添加，多个用英文逗号隔开<br/>
 *  <li>单个请求不打印日志</li>
 *  在request的生命周期内，在filter结束之前，您都可以通过调用{@link #dontWriteCurrentLog} 方法来禁用本次请求的日志打印
 *  <pre>
 *      e.g:
 *  {@code
 *      @Resource
 *      private KjjCustomAccessLogFilter kjjCustomAccessLogFilter;
 *
 *      @GetMapping("test")
 *      public Result<Void> test(HttServletRequest request){
 *          kjjCustomAccessLogFilter.dontWriteCurrentLog(request);
 *          return ResultBuilders.success();
 *      }
 *
 *  }
 *  </pre>
 *  <h2>追加日志内容</h2>
 *  日志打印的是一个JSON模版如下：
 *  <pre>
 *      {@code
 *          {
 *              "url_query":"参数列表：scid=1&t=1",
 *              "referer":"request header中的referer字段",
 *              "rt":"基本可以认为是您的请求耗时：单位毫秒",
 *              "ip":"客户端IP",
 *              "app_name":"kjy-web",
 *              "rc":"返回值状态码： 200",
 *              "http_method":"GET",
 *              "ex":{
 *                  "t":"1624073199458",
 *                  "scid":"Kj20MTE4MjIzODcx"
 *                  },
 *              "mip":"服务器内网IP",
 *              "time":"请求时间 2021-06-19 11:26:39",
 *              "url_host":"wx.kjjcrm.com",
 *              "user_agent":"request  header中的ua",
 *              "rb":"request body",
 *              "pb":"response body"
 *              "url_path":"URL ： /kjy/mp/chat/checkUnread"
 *          }
 *      }
 *  </pre>
 *  JSON的直接属性是包含Request的基本信息，其中"ex"属性就是提供给用户自定义属性使用的。
 *  <h3>添加全局自定义属性</h3>
 *  我们可以通过Interceptor来拦截所有的请求，在Interceptor中进行额外属性的配置。当然，您也可以在您需要的地方进行切面编程。这里，我们只关注于如何添加自定义属性。
 *  <pre>
 *      e.g:
 *      {@code
 *      public class AccessLogInterceptor implements HandlerInterceptor {
 *     @Resource
 *     private KjjCustomAccessLogFilter accessLogFilter;
 *
 *     @Override
 *     public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
 *        try {
 *            // add transId
 *            String transactionId = request.getHeader("X-Transaction-ID");
 *            if (StringUtils.isNotBlank(transactionId)){
 *                accessLogFilter.putExPair(request, "transId", transactionId);
 *            }
 *            //把加密的ID参数，解密后放入ex中
 *            Map<String, String[]> paramMap = request.getParameterMap();
 *            if (MapUtils.isNotEmpty(paramMap)) {
 *                Map<String, String[]> params = embedService.dealEncodingFiles(paramMap);
 *                params.forEach((k, v) -> accessLogFilter.putExPair(request, k, (1 == v.length) ? v[0] : v));
 *            }
 *        }catch (Exception e){
 *            log.info("print access log has error!",e);
 *        }
 *         return true;
 *     }
 *
 *      }
 *     }
 *  </pre>
 *  <h3>单独添加自定义属性</h3>
 *  <pre>
 *      e.g:
 *  {@code
 *      @Resource
 *      private KjjCustomAccessLogFilter kjjCustomAccessLogFilter;
 *
 *      @GetMapping("test")
 *      public Result<Void> test(HttServletRequest request){
 *          kjjCustomAccessLogFilter.putExPair(request,"myKey","this is a simple value!");
 *          return ResultBuilders.success();
 *      }
 *
 *  }
 *  </pre>
 *
 *  <h2>开启response body打印</h2>
 *  利用阿波罗配置 access.log.pb.open = true 可开启。开启后日志中多出 "pb"属性。<br/>
 *  一般情况，我们是不建议去开启的。尤其对于高并发系统，因为这象功能需要开启response的双写功能。
 *
 *  <h2>对异步servlet的支持</h2>
 *  异步servlet意味着对 request的异步处理，request本身跨越了原本属于它的范围，而response直接不在常规的流程中做处理，需要自行控制输出流。
 *  当某些场景下，我们需要打印responseBody时，可以将本请求设置在黑名单中（{@code accesslog.blackpaths=XXXX}），
 *  然后在response steam write完通过调用{@link #printAsyncAccessLog}来完成access log的打印
 * <h1 color="red">重要的事情：</h1>
 * <span color="red">
 *    <li>本类处理的是Http请求的信息打印，所以务必在request的生命周期内进行日志处理</li>
 *    <li>为了保证request的安全性，代码加了校验，要求日志的处理必须的容器的线程内处理。请求异步处理后不可以再使用。</li>
 *    <li>accessLog依赖的依然是logback插件，本类使用的依然是中台ext中提供 logback/access-logback.xml配置</li>
 *    <li>为了线程安全，本类属性存储依赖request域。请不要乱用，影响request本身！</li>
 * </span>
 */
@Slf4j
public class KjjCustomAccessLogFilter implements Filter, InitializingBean {
    private static final Logger accessLog = LoggerFactory.getLogger("duiba_access_log");

    public static final String OW_HOST="host";
    private static final String OW_PATH="accesslog_overwrite_path";

    private static final String ModeBlack="black";//黑名单模式
    private static final String ModeWhite="white";//白名单模式

    private static final String COLLECT_COOKIE_MAP_KEY="_accesslog_collect_cookie_map";

    private static final String LOG_COOKIE_ATTRIBUTE_KEY = "_accesslog_log_cookie";
    private static final String EX_ATTRIBUTE_KEY = "_accesslog_ex";
    private static final String OVERWRITE_ATTRIBUTE_KEY = "_accesslog_overwrite";
    private static final String DONT_WRITE_CURRENT_ACCESSLOG="dont_write_current_accesslog";

    private String mode=ModeBlack;
    private Set<String> inhosts=new HashSet<>();

    private Set<String> blackPaths=new HashSet<>();
    private Set<String> blackUriPatterns = new HashSet<>();
    private Map<String,String> cookieHosts=new ConcurrentHashMap<>();
    private AntPathMatcher pathMatcher = new AntPathMatcher();

    @Resource
    private LoggerSizeFilter loggerSizeFilter;

    @Value("${spring.application.name}")
    private String appName;

    @Resource
    private AccessLogProps accessLogProps;

    private static final String KJJ_ACCESS_UID = "kjj_access_uid";

    private static final String KJJ_ACCESS_SID = "kjj_access_sid";
    @Override
    public void afterPropertiesSet(){
        try {
            Properties p = new Properties();
            ClassPathResource res = new ClassPathResource("/accesslog_filter.properties");
            if (!res.exists()) {
                return;
            }
            p.load(res.getInputStream());
            if(!p.isEmpty()){
                loadConfig(p);
                log.info("load accesslog_filter.properties success");
            }
        }catch (Exception e){
            log.warn("load accesslog_filter.properties failed", e);
        }
    }


    /**
     * 加载配置项
     * @param properties
     */
    private void loadConfig(Properties properties){
        mode=properties.getProperty("accesslog.mode");
        String hoststring=properties.getProperty("accesslog.hosts");
        if(hoststring!=null){
            List<String> hosts= Arrays.asList(hoststring.trim().split(","));
            if(!CollectionUtils.isEmpty(hosts)){
                inhosts=new HashSet<>(hosts);
            }
        }
        String blackpaths=properties.getProperty("accesslog.blackpaths");
        List<String> paths=Arrays.asList(StringUtils.split(StringUtils.trimToEmpty(blackpaths),","));
        for(String path : paths){
            if(path != null && !path.contains("*")){
                blackPaths.add(path);
            }else if(path != null && path.contains("*")){
                blackUriPatterns.add(path);
            }
        }
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        HttpServletRequest req = getServletRequest((HttpServletRequest) servletRequest);
        HttpServletResponse resp = getResp((HttpServletResponse) servletResponse);
        long start = System.currentTimeMillis();

        try {
            doBefore(req, resp);
            filterChain.doFilter(req, resp);
        }finally {
            doAfter(req, resp, System.currentTimeMillis() - start);
        }
    }

    public void printAsyncAccessLog(HttpServletRequest req, HttpServletResponse resp){
        if (req.isAsyncStarted()){
            doAfter(req, resp, -1);
        }else{
            log.error("this methods can only be called at the Async servlet's thread");
        }
    }

    @NotNull
    private HttpServletResponse getResp(HttpServletResponse servletResponse) throws IOException {
        if (accessLogProps.isResponseBodyPrintFLag()){
            return new BodyWriterHttpServletResponseWrapper(servletResponse);
        }
        return servletResponse;
    }

    private HttpServletRequest getServletRequest(HttpServletRequest req) {
        String method = StringUtils.lowerCase(getMethod(req));
        if (StringUtils.equals(method,"post")){
            return new BodyReaderHttpServletRequestWrapper(req);
        }
        return req;
    }

    protected void doBefore(HttpServletRequest req, HttpServletResponse resp) {
        Map<String, String> collCookie=new HashMap<>();
        try {
            //在执行filter之前种cookie,避免resp被提前关闭,无法写入cookie
            addCookieIfNeed(req, resp, collCookie);
            addAttribute(req, COLLECT_COOKIE_MAP_KEY, collCookie);
        }catch(Exception e){
            log.error("addCookieIfNeed error");
        }
    }

    protected void doAfter(HttpServletRequest req, HttpServletResponse resp, long cost) {
        try {
            //判断是否性能压测url
            boolean isPerfUrl = isPerfTestRequest(req);
            if(!isPerfUrl && needLog(req)){
                Map<String, String> collCookie = (Map<String, String>)getAttribute(req, COLLECT_COOKIE_MAP_KEY);
                processAccessLog(req,resp,cost,collCookie);
            }
        }catch (Exception e){
            log.error("AccessLogFilter process error, message=", e);
        }
    }

    private void addCookieIfNeed(HttpServletRequest req, HttpServletResponse resp, Map<String,String> collCookie){
        List<String> parameterNames=getParameterNames(req);
        String host=getHeader(req, "host");
        String cookieHost=getCookieHost(host);

        for(String key : parameterNames){
            if(!key.startsWith("tck_")){
                continue;
            }

            String[] v=key.split("_");
            if(v.length!=3){
                continue;
            }

            String verify = SecurityUtils.encode2StringByMd5(v[1]);
            if(verify.endsWith(v[2])){
                String name=v[1];
                String value=getParameter(req, key);
                //验证通过
                String cookiekey="_coll_"+name;
                collCookie.put(cookiekey,value);
                //写入cookie

                try {
                    addCookie(resp, cookiekey, value, cookieHost, "/");
                }catch (Exception e){
                    log.error("addCookie error,cookieKey="+cookiekey+",value="+value,e);
                }
            }
        }
    }

    private void processAccessLog(HttpServletRequest req, HttpServletResponse resp,long cost,Map<String,String> collCookie) throws Exception{//NOSONAR
        //只记录get post请求
        String method = StringUtils.lowerCase(getMethod(req));
        if(!"get".equals(method) && !"post".equals(method)){
            return;
        }

        Map<String, Object> map = new HashMap<>();
        map.put("app_name", appName);//当前应用名称
        map.put("url_host", getHost(req));
        final String path = clearRequestURI(getPath(req));
        map.put("url_path", path);
        map.put("url_query", getQueryString(req));
        if (method.equals("post") && req instanceof BodyReaderHttpServletRequestWrapper){ //post 添加body
            map.put("rb",((BodyReaderHttpServletRequestWrapper)req).getBody());
        }
        putIfNotNull(map,"uid",getAttribute(req,KJJ_ACCESS_UID));
        putIfNotNull(map,"sid",getAttribute(req,KJJ_ACCESS_SID));
        map.put("http_method",method);
        final int status = getStatus(resp);
        putLocation(status,req,resp);
        map.put("rc", status);
        map.put("rt",cost);
        map.put("mip", NetUtils.getLocalIp());//记录服务器IP
        putIfNotNull(map, "ex", getExPair(req));

        //如果url中有callback参数,认为此url是一个JSONP请求,标记为POST类型,不作为PV统计
        String callback=getParameter(req, "callback");
        if(!StringUtils.isBlank(callback)){
            map.put("http_method","POST");
        }

        Map<String, String> cookieMap = getCookieMap(req);

        handleCookies(cookieMap, collCookie);

        handleParams(req, collCookie);

        //支持线程变量中取cookie进行日志打印,便于不同系统的一些定制需求,此为cookie的最高优先级获取方法
        handleLogCookie(req, collCookie);

        putIfNotEmpty(map, "cookie", collCookie);

        putUa(req, map);

        if (resp instanceof BodyWriterHttpServletResponseWrapper){
            String responseBody = new String(((BodyWriterHttpServletResponseWrapper)resp).getResponseBody());
            if (StringUtils.isNotBlank(responseBody)){
                map.put("pb",StringUtils.substring(responseBody,0,accessLogProps.getResponseBodyMaxLength()));
            }
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        map.put("time", sdf.format(new Date()));
        String referer = getHeader(req, "referer");
        putIfNotNull(map, "referer", referer);
        String ip = getIpAddr(req);
        putIfNotNull(map, "ip", ip);

        String body = JSON.toJSONString(map);
        if(body.length()>=loggerSizeFilter.getWarnThreshold()){
            map.remove("pb");
            map.remove("rb");
            log.info("access Log too large so rm request body and response body ! url = {}",path);
            body = JSON.toJSONString(map);
        }
        accessLog.info(body);
        if (accessLogProps.isPrintAccessLogIntoApplication()){
            log.info(body);
        }
    }

    private void putLocation(int status, HttpServletRequest req, HttpServletResponse resp) {
        try {
            if (status==302){
                putExPair(req,"ln",resp.getHeader("Location"));
            }
        }catch (Exception e){
            //ignore
            log.info("put location error",e);
        }

    }


    private Map<String, Object> getExPair(HttpServletRequest req){
        return (Map<String, Object>)getAttribute(req, EX_ATTRIBUTE_KEY);
    }

    public void setUid(HttpServletRequest req,Long uid){
        if(Objects.isNull(uid)){
            return;
        }
        req.setAttribute(this.KJJ_ACCESS_UID,uid);
    }

    public void setSid(HttpServletRequest req,Long sid){
        if(Objects.isNull(sid)){
            return;
        }
        req.setAttribute(this.KJJ_ACCESS_SID,sid);
    }

    private void putUa(HttpServletRequest req, Map<String, Object> map){
        String ua = StringUtils.trimToEmpty(getHeader(req, "user-agent"));
        if (ua.length() > 500) {
            ua = ua.substring(0, 499);
        }
        if (!StringUtils.isEmpty(ua)) {
            map.put("user_agent", ua);
        }
    }

    private void putIfNotNull(Map<String, Object> map, String key, Object value){
        if(value != null){
            map.put(key, value);
        }
    }

    private void putIfNotEmpty(Map<String, Object> map, String key, Map<String,String> collCookie){
        if(!collCookie.isEmpty()){
            map.put(key, collCookie);
        }
    }

    private void handleCookies(Map<String, String> cookieMap,Map<String,String> collCookie){
        for(Map.Entry<String, String> entry : cookieMap.entrySet()){
            String key = entry.getKey();
            String cookie = entry.getValue();
            if(key.startsWith("_coll_")){
                collCookie.put(key,cookie);
            }
        }
    }

    private void handleParams(HttpServletRequest req, Map<String,String> collCookie){
        //特殊处理
        String slotId=getParameter(req, "adslotId");
        if(slotId!=null){
            collCookie.put("_coll_slot",slotId);
        }
        String deviceId=getParameter(req, "deviceId");
        if(deviceId!=null){
            collCookie.put("_coll_device",deviceId);
        }
    }

    private void handleLogCookie(HttpServletRequest req, Map<String,String> collCookie){
        Map<String,String> logCookie=getLogCookie(req);
        if(logCookie!=null){
            for(Map.Entry<String, String> entry : logCookie.entrySet()){
                collCookie.put(entry.getKey(),entry.getValue());
            }
        }
    }

    private Map<String, String> getLogCookie(HttpServletRequest req) {
        return (Map<String, String>) getAttribute(req, LOG_COOKIE_ATTRIBUTE_KEY);
    }

    /**
     * requestURI中可能出现连续两个/，比如//activity//index，需要处理下转为一个/
     * @param requestURI
     * @return
     */
    private String clearRequestURI(String requestURI){
        if(StringUtils.isBlank(requestURI)){
            return requestURI;
        }

        return StringUtils.replace(requestURI, "//", "/");
    }

    /**
     * 根据host查询 通配符cookie地址
     * @param host
     * @return
     */
    private String getCookieHost(String host){
        if(host==null){
            return null;
        }
        if(host.contains(":")){
            host=host.substring(0,host.indexOf(':'));
        }
        String cookieHost=cookieHosts.get(host);
        if(cookieHost!=null){
            return cookieHost;
        }
        String hostLow=host.toLowerCase();
        if(hostLow.endsWith(".duiba.com.cn") ){
            cookieHost="duiba.com.cn";
        }else if(hostLow.endsWith(".dui88.com")){
            cookieHost="dui88.com";
        }else if(hostLow.endsWith(".duibar.com")){
            cookieHost="duibar.com";
        }else if(hostLow.endsWith(".tuia.cn")) {
            cookieHost = "tuia.cn";
        }

        if(cookieHost!=null){
            cookieHosts.put(host,cookieHost);
        }
        return cookieHost;
    }


    /**
     * 添加cookie信息到线程变量,日志优先从线程变量中获取
     * @param key
     * @param value
     */
    public void putLogCookie(HttpServletRequest req, String key,String value){
        ensureSafeCall();
        Map<String, String> map = (Map<String, String>)getAttribute(req, LOG_COOKIE_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            addAttribute(req, LOG_COOKIE_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    /**
     * 添加额外信息到线程变量,日志从线程变量获取信息进行打压输出
     * @param key
     * @param value
     */
    public void putExPair(HttpServletRequest req, String key,Object value){
        ensureSafeCall();
        Map<String, Object> map = (Map<String, Object>)getAttribute(req, EX_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            addAttribute(req, EX_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    /**
     * 添加host等覆盖信息到线程变量, 打印日志时优先从此处获取
     * @param key
     * @param value
     */
    public void putOverWritePair(HttpServletRequest req, String key,String value){
        ensureSafeCall();
        Map<String, String> map = (Map<String, String>)getAttribute(req, OVERWRITE_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            addAttribute(req, OVERWRITE_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    /**
     * 当前请求不写accesslog，该设置只对本次请求有效
     */
    public void dontWriteCurrentLog(HttpServletRequest req){
        addAttribute(req, DONT_WRITE_CURRENT_ACCESSLOG, true);
    }

    private String getPath(HttpServletRequest req){
        String path = (String)getAttribute(req, OW_PATH);
        if(StringUtils.isBlank(path)){
            path = getRequestURI(req);
        }

        return path;
    }

    /**
     * 是否需要进行日志打印
     * @return
     */
    private boolean needLog(HttpServletRequest req){
        Boolean val = (Boolean)getAttribute(req, DONT_WRITE_CURRENT_ACCESSLOG);
        if(val != null && val){
            return false;
        }
        String host=getHost(req);
        if(ModeBlack.equals(mode)){
            if(inhosts.contains(host)){
                return false;
            }
        }else if(ModeWhite.equals(mode)){
            if(!inhosts.contains(host)){
                return false;
            }
        }

        String path=clearRequestURI(getPath(req));
        if(blackPaths.contains(path)){
            return false;
        }
        for(String pattern : blackUriPatterns){
            if(this.pathMatcher.match(pattern, path)){
                return false;
            }
        }

        return true;
    }

    private String getHost(HttpServletRequest req){
        Map<String, String> map = (Map<String, String>)getAttribute(req, OVERWRITE_ATTRIBUTE_KEY);
        if(map!=null){
            String host=map.get(OW_HOST);
            if(host!=null){
                return host;
            }
        }

        return getHeader(req, "host");
    }

    /**
     * 设置path，设置后在记录path时会优先使用这里设置的值
     * @param path
     */
    public void setOverWritePath(HttpServletRequest req, String path){
        this.addAttribute(req, OW_PATH, path);
    }

    protected String getRequestURI(HttpServletRequest req) {
        return req.getRequestURI();
    }

    protected String getMethod(HttpServletRequest req) {
        return req.getMethod();
    }

    protected String getQueryString(HttpServletRequest req) {
        return req.getQueryString();
    }

    protected String getHeader(HttpServletRequest req, String key) {
        return req.getHeader(key);
    }

    protected String getIpAddr(HttpServletRequest req) {
        return HttpRequestUtils.getIpAddr(req);
    }

    protected void addAttribute(HttpServletRequest req, String key, Object value) {
        req.setAttribute(key, value);
    }

    protected Object getAttribute(HttpServletRequest req, String key) {
        return req.getAttribute(key);
    }

    protected String getParameter(HttpServletRequest req, String key) {
        return req.getParameter(key);
    }

    protected List<String> getParameterNames(HttpServletRequest req) {
        return EnumerationUtils.toList(req.getParameterNames());
    }

    protected int getStatus(HttpServletResponse resp) {
        return resp.getStatus();
    }

    protected void addCookie(HttpServletResponse resp, String key, String value, String domain, String path) {
        Cookie cookie = new Cookie(key, value);
        if(domain != null){
            cookie.setDomain(domain);
        }
        if (path != null) {
            cookie.setPath(path);
        }
        resp.addCookie(cookie);
    }

    protected boolean isPerfTestRequest(HttpServletRequest req) {
        return PerfTestUtils.isPerfTestRequest(req);
    }

    protected Map<String, String> getCookieMap(HttpServletRequest req) {
        Cookie[] cookies = req.getCookies();
        Map<String, String> cookieMap = new HashMap<>();
        if(cookies!=null){
            for (Cookie c : cookies) {
                cookieMap.put(c.getName(), c.getValue());
            }
        }
        return cookieMap;
    }

    /**
     * 确保KjjCustomAccessLogFilter的方法只能被servlet线程调用，否则抛出异常，以避免误用
     */
    protected void ensureSafeCall(){
        String threadName = StringUtils.defaultString(Thread.currentThread().getName());
        if (!threadName.startsWith("http-nio")//tomcat
                && !threadName.startsWith("XNIO")//undertow
                && !threadName.startsWith("qtp")//jetty
                && !threadName.equals("main")) {//允许main方法调用，方便测试
            log.error("KjjCustomAccessLogFilter's methods can only be called at the servlet's thread");
        }
    }

    @Override
    public void destroy() {
        //do nothing
    }
}
