package cn.com.duiba.kjy.base.customweb.web.condition;

import cn.com.duiba.kjy.base.customweb.util.CorsUtils;
import cn.com.duiba.kjy.base.customweb.web.bean.KjjHttpRequest;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.annotation.RequestMethod;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

/**
 * @author dugq
 * @date 2021/4/1 5:02 下午
 */
public class RequestMethodsRequestCondition extends AbstractRequestCondition<RequestMethodsRequestCondition> {

    /** Per HTTP method cache to return ready instances from getMatchingCondition. */
    private static final Map<String, RequestMethodsRequestCondition> requestMethodConditionCache;

    static {
        requestMethodConditionCache = new HashMap<>(RequestMethod.values().length);
        for (RequestMethod method : RequestMethod.values()) {
            requestMethodConditionCache.put(method.name(), new RequestMethodsRequestCondition(method));
        }
    }


    private final Set<RequestMethod> methods;


    /**
     * Create a new instance with the given request methods.
     * @param requestMethods 0 or more HTTP request methods;
     * if, 0 the condition will match to every request
     */
    public RequestMethodsRequestCondition(RequestMethod... requestMethods) {
        this(Arrays.asList(requestMethods));
    }

    private RequestMethodsRequestCondition(Collection<RequestMethod> requestMethods) {
        this.methods = Collections.unmodifiableSet(new LinkedHashSet<>(requestMethods));
    }


    /**
     * Returns all {@link RequestMethod RequestMethods} contained in this condition.
     */
    public Set<RequestMethod> getMethods() {
        return this.methods;
    }

    @Override
    protected Collection<RequestMethod> getContent() {
        return this.methods;
    }

    @Override
    protected String getToStringInfix() {
        return " || ";
    }

    /**
     * Returns a new instance with a union of the HTTP request methods
     * from "this" and the "other" instance.
     */
    @Override
    public RequestMethodsRequestCondition combine(RequestMethodsRequestCondition other) {
        Set<RequestMethod> set = new LinkedHashSet<>(this.methods);
        set.addAll(other.methods);
        return new RequestMethodsRequestCondition(set);
    }

    /**
     * Check if any of the HTTP request methods match the given request and
     * return an instance that contains the matching HTTP request method only.
     * @param request the current request
     * @return the same instance if the condition is empty (unless the request
     * method is HTTP OPTIONS), a new condition with the matched request method,
     * or {@code null} if there is no match or the condition is empty and the
     * request method is OPTIONS.
     */
    @Override
    @Nullable
    public boolean getMatchingCondition(KjjHttpRequest request) {
        if (CorsUtils.isPreFlightRequest(request)) {
            return true;
        }
        if (getMethods().isEmpty()) {
            return true;
        }
        return matchRequestMethod(request.getMethod());
    }

    @Nullable
    private boolean matchRequestMethod(String httpMethodValue) {
        RequestMethod requestMethod;
        try {
            requestMethod = RequestMethod.valueOf(httpMethodValue);
            if (getMethods().contains(requestMethod)) {
                return true;
            }
            if (requestMethod.equals(RequestMethod.HEAD) && getMethods().contains(RequestMethod.GET)) {
                return true;
            }
        }
        catch (IllegalArgumentException ex) {
            // Custom request method
        }
        return false;
    }

    @Override
    public int compareTo(RequestMethodsRequestCondition other, KjjHttpRequest request) {
        if (other.methods.size() != this.methods.size()) {
            return other.methods.size() - this.methods.size();
        }
        else if (this.methods.size() == 1) {
            if (this.methods.contains(RequestMethod.HEAD) && other.methods.contains(RequestMethod.GET)) {
                return -1;
            }
            else if (this.methods.contains(RequestMethod.GET) && other.methods.contains(RequestMethod.HEAD)) {
                return 1;
            }
        }
        return 0;
    }

}
