package cn.com.duiba.kjy.api.enums.crm;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * crm通用扩展表所存字段枚举
 */
public enum CrmCommonExtColumnEnum {

    /**
     * 扩展字段枚举，参数code-参数名称
     */
    BIZ_CITY(1, "bizCity"),
    TODAY_SCHEDULED_CUSTOMER(2, "todayScheduledCustomer"),
    INTENTION_TOPIC(3, "intentionTopic"),
    INTENTION_SCHEDULE(4, "intentionSchedule"),
    BUDGET(5, "budget"),
    OBJECTION(6, "objection"),
    FOLLOW_PLAN(7, "followPlan")
    ;


    private final Integer code;
    private final String paramName;

    /**
     * 扩展字段最大长度限制
     */
    private static final Integer MAX_LENGTH = 200;

    CrmCommonExtColumnEnum(Integer code, String paramName) {
        this.code = code;
        this.paramName = paramName;
    }

    public Integer getCode() {
        return code;
    }

    public String getParamName() {
        return paramName;
    }

    private static final Map<Integer, CrmCommonExtColumnEnum> CODE_MAP = new HashMap<>();
    private static final Map<String, CrmCommonExtColumnEnum> PARAM_NAME_MAP = new HashMap<>();

    static {
        for (CrmCommonExtColumnEnum item : CrmCommonExtColumnEnum.values()) {
            CODE_MAP.put(item.getCode(), item);
            PARAM_NAME_MAP.put(item.getParamName(), item);
        }
    }

    public static CrmCommonExtColumnEnum getByCode(Integer code) {
        if (code == null) {
            return null;
        }
        return CODE_MAP.get(code);
    }

    public static CrmCommonExtColumnEnum getByParamName(String paramName) {
        if (paramName == null) {
            return null;
        }
        return PARAM_NAME_MAP.get(paramName);
    }

    /**
     * 扩展字段正向转换，参数名称 => 参数对应code
     *
     * @param bizExts 需要转换的扩展字段map
     * @return 转换后的扩展字段map
     */
    public static Map<Integer,String> transBizExtsForward(Map<String,String> bizExts) {
        if (CollectionUtils.isEmpty(bizExts)) {
            return null;
        }
        Map<Integer, String> bizExtsTrans = Maps.newHashMap();
        for (String paramName : bizExts.keySet()) {
            if (Objects.nonNull(CrmCommonExtColumnEnum.getByParamName(paramName))) {
                bizExtsTrans.put(CrmCommonExtColumnEnum.getByParamName(paramName).getCode(), Optional.ofNullable(bizExts.get(paramName)).orElse(""));
            }
        }
        checkLength(bizExtsTrans);
        return bizExtsTrans;
    }

    /**
     * 扩展字段反向转换，参数对应code => 参数名称
     *
     * @param bizExts 扩展字段map
     * @return 转换后的扩展字段map
     */
    public static Map<String,String> transBizExtsReverse(Map<Integer,String> bizExts) {
        if (org.springframework.util.CollectionUtils.isEmpty(bizExts)) {
            return null;
        }

        Map<String, String> bizExtsTrans = Maps.newHashMap();
        for (Integer code : bizExts.keySet()) {
            if (Objects.nonNull(CrmCommonExtColumnEnum.getByCode(code))) {
                bizExtsTrans.put(CrmCommonExtColumnEnum.getByCode(code).getParamName(), bizExts.get(code));
            }
        }
        return bizExtsTrans;
    }

    private static void checkLength(Map<Integer,String> bizExts) {
        if (CollectionUtils.isEmpty(bizExts)) {
            return;
        }
        List<Integer> removeKeys = Lists.newArrayList();
        for (Map.Entry<Integer,String> entry : bizExts.entrySet()) {
            if (entry.getValue().length() >= MAX_LENGTH) {
                removeKeys.add(entry.getKey());
            }
            if (Objects.equals(INTENTION_TOPIC.getCode(),entry.getKey()) && !CrmCustumerIntentionEnum.containsCode(entry.getValue())) {
                removeKeys.add(entry.getKey());
            }
        }
        if (!CollectionUtils.isEmpty(removeKeys)) {
            for (Integer code : removeKeys) {
                bizExts.remove(code);
            }
        }
    }

}
