package cn.com.duiba.kjy.api.util;

import cn.com.duiba.kjy.api.constant.KjyRuntimeException;
import org.apache.commons.lang.StringUtils;

import java.util.Objects;

/**
 * 用于SellerID的序列化反序列化  反序列化的结果用于客户端展示
 * 例子:  会员ID 一共7位
 *      第一位: sellerId的长度
 *      第二位: sellerId的最后一位的平方+ 剩余空位数
 *      第三位: (前一位的计算结果)的平方 + 剩余空位数
 *      以此类推 填满剩余空位为止
 *      例子:
 *          sellerId: 0000001
 *          会员ID:   1603121
 *          拆分后算法:
 *          会员ID:   1               6                           0                   3                           1               2                1
 *                  sellerId位数     1*1+5=6(取个位)         6*6+4=40 取个位0      0*0+3=3(取个位3)           3*3+2(取个位:1)       1*1+1(取个位:2)    1*1+0(取个位:1)
 * @auther wuwenzhe
 * @date 2019-08-01 20:10
 */
public class SellerIdConverter {

    /**
     * 序列化销售ID  得到 memberId 会员ID
     * @param sellerId
     * @return
     */
    public static Long encode(Long sellerId){
        if(Objects.isNull(sellerId)){
            throw new KjyRuntimeException("销售ID为空,无法获取会员ID");
        }
        //sellerId大于999999
        if(sellerId > 999999l){
            throw new KjyRuntimeException("销售ID最大支持999999");
        }
        String sellerIdStr = String.valueOf(sellerId);
        //第一位是长度
        int length = sellerIdStr.length();
        //需要补位的长度
        int replaceLength = 6-length;
        //补位数值
        String replaceStr =  getReplaceStr(replaceLength,sellerIdStr);

        return Long.valueOf(length + replaceStr + sellerIdStr);

    }

    /**
     *
     * @param replaceLength  需要补位的长度
     * @param sellerIdStr  sellerId的值
     * @return
     */
    private static String getReplaceStr(int replaceLength, String sellerIdStr) {
        if(replaceLength==0){
            return "";
        }

        //sellerIdStr的首位 1-9
        int start = Integer.valueOf(sellerIdStr.substring(sellerIdStr.length()-1));
        StringBuilder sb = new StringBuilder();
        for(int i = replaceLength; i >0  ;i--){
            //取sellerId的第一位与需要补空的位数值相加
            int replaceNum = start * start + i;
            //获取个位重新赋给start
            start = getUnits(replaceNum);
            //最终字符串追加
            sb.append(start);
        }
        return sb.toString();

    }

    /**
     * 获取个位数
     * @param num
     * @return
     */
    private static int getUnits(int num) {
        return num>9?num%10:num;
    }


    /**
     * 反序列化会员ID 校验失败会返回-1
     * @param sourceMemberId
     * @return
     */
    public static Long decode(Long sourceMemberId){
        return validate(sourceMemberId)?innerDecode(sourceMemberId+""):-1l;
    }

    /**
     * 内部转换  用会员ID的第一位去截取指定长度的 sellId
     * @param memberId
     * @return
     */
    private static Long innerDecode(String memberId){
        int length = Integer.valueOf(memberId.substring(0,1));
        int replaceLength = 7-length;
        return Long.valueOf(memberId.substring(replaceLength,7));
    }

    /**
     * 根据会员ID 校验会员ID是否正确
     *
     * @param sourceMemberId 原会员ID
     * @return 返回校验结果
     */
    public static boolean validate(Long sourceMemberId){
        String memberId = sourceMemberId+"";
        if(StringUtils.isEmpty(memberId) || memberId.length()!=7){
            return false;
        }
        Long realSellerId = innerDecode(memberId);
        return sourceMemberId.equals(encode(realSellerId));

    }
}
