package cn.com.duiba.kjy.api.util;

/**
 * 加密规则：
 *      1.加密后字符串结构：1 + id长度（2位） + 补充位（总长度-3-id长度） + 加密后的ID
 *      2.补充位规则：
 *            a.获取基础补充字符串（加密后的ID长度超过替换长度，截取加密ID，不足，前后补0）
 *            b.将补充字符串每一位 平方，然后取余 (位置+值)%10
 *      3.ID加密规则：
 *            a.每一位数字+7%10
 *            b.相邻两位数字交换位置
 * 解密规则：
 *      1.先取id长度，根据长度截取加密后的ID
 *      2.ID解密规则：
 *            a.相邻两位数字交换位置
 *            b.数字 > 7，直接减去7，小于7，直接加3
 *
 * @author lizhi
 * @date 2019/8/16 4:55 PM
 */
public class IdConverter {

    private static final int PRE_LENGTH = 3;

    /**
     * 加密ID
     * @param id 需要加密的ID，最大支持9位数的ID
     * @return 加密后的ID（12位）
     */
    public static Long encodeId(Long id) {
        return encode(id, 12);
    }

    /**
     * 加密订单ID
     * @param id 订单ID，最大支持16位
     * @return 加密后的ID（19位）
     */
    public static Long encodeOrderId(Long id) {
        return encode(id, 19);
    }

    /**
     * 加密，总长度需要>=（加密ID长度+3）
     * @param id 要加密的ID
     * @param totalLength 加密后总长度
     * @return 加密后的ID， null=id为null时，-1=参数错误
     */
    private static Long encode(Long id, int totalLength){
        if (id == null) {
            return null;
        }
        if (id <= 0) {
            return id;
        }
        String idStr = String.valueOf(id);
        int idLength = idStr.length();
        if (idLength + PRE_LENGTH > totalLength) {
            return -1L;
        }
        String pre = getPre(idLength);
        String encodeIdStr = getEncodeIdStr(idStr);
        String replaceStr = getReplaceStr(encodeIdStr, totalLength - PRE_LENGTH - idLength);
        return Long.parseLong(pre + replaceStr + encodeIdStr);
    }

    /**
     * 解密，长度要大于3
     * @param id 要解密的ID
     * @return 解密后的ID，null=id为null时，-1=参数错误
     */
    public static Long decode(Long id) {
        if (id == null) {
            return null;
        }
        if (id <= 0) {
            return id;
        }
        String s = String.valueOf(id);
        if (s.length() <= PRE_LENGTH) {
            return -1L;
        }
        String encodeIdStr = getEncodeIdStrById(s);
        String decodeIdStr = getDecodeIdStr(encodeIdStr);
        return Long.parseLong(decodeIdStr);
    }

    /**
     * 获取加密ID的前缀
     * @param length 要加密的ID长度
     * @return 加密ID的前缀
     */
    private static String getPre(int length) {
        String lengthStr = String.valueOf(length);
        if (lengthStr.length() < 2) {
            return "10" + lengthStr;
        }
        return "1" + lengthStr;
    }

    /**
     * 获取加密ID的补充数值
     * 规则：
     *      1.获取基础补充字符串（加密后的ID长度超过替换长度，截取加密ID，不足，前后补0）
     *      2.将补充字符串每一位 平方，然后取余 (位置+值)%10
     * @param encodeIdStr 加密的ID
     * @param replaceLength 需要补充的长度
     * @return 补充数值
     */
    private static String getReplaceStr(String encodeIdStr, int replaceLength) {
        if (replaceLength <= 0) {
            return "";
        }
        int num = replaceLength - encodeIdStr.length();
        String replaceStr;
        if (num < 0) {
            replaceStr = encodeIdStr.substring(0,replaceLength);
        } else if (num == 0){
            replaceStr = encodeIdStr;
        } else {
            replaceStr = getZero(num/2) + encodeIdStr + getZero(num - num/2);
        }
        return replace(replaceStr);
    }

    private static String replace(String replaceStr) {
        char[] chars = replaceStr.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            int num = Integer.parseInt(String.valueOf(chars[i])) + i + 1;
            int pow = num * num;
            int yu = (num + i) % 10;
            if (yu == 0) {
                yu = 10;
            }
            int result = pow % yu;
            chars[i] = (char) (result + '0');
        }
        return new String(chars);
    }

    /**
     * 获取0
     * @param num 需要的0的个数
     * @return 0字符串
     */
    private static String getZero(int num) {
        if (num <= 0) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < num; i++) {
            sb.append("0");
        }
        return sb.toString();
    }

    /**
     * 截取加密的ID
     * @param s 加密ID字符串
     * @return 截取到的ID
     */
    private static String getEncodeIdStrById(String s) {
        String substring = s.substring(1, PRE_LENGTH);
        int idLength = Integer.parseInt(substring);
        return s.substring(s.length() - idLength);
    }

    /**
     * 加密ID
     * @param idStr id字符串
     * @return 加密后的ID字符串
     */
    private static String getEncodeIdStr(String idStr) {
        char[] chars = idStr.toCharArray();
        //数字转换
        convert(chars);
        //位置交换
        exchange(chars);
        return new String(chars);
    }

    /**
     * 解密ID
     * @param encodeIdStr 加密的ID字符串
     * @return 解密后的ID字符串
     */
    private static String getDecodeIdStr(String encodeIdStr) {
        char[] chars = encodeIdStr.toCharArray();
        //位置交换
        decodeExchange(chars);
        //数字转换
        decodeConvert(chars);
        return new String(chars);
    }


    private static void decodeExchange(char[] chars) {
        exchange(chars);
    }

    /**
     * 将数组中相邻两位位置交换
     * @param chars 需要位置交换的数组
     */
    private static void exchange(char[] chars) {
        for (int i = 0; i < chars.length/2; i++) {
            int index = i * 2;
            char temp = chars[index];
            chars[index] = chars[index+1];
            chars[index+1] = temp;
        }
    }

    private static void decodeConvert(char[] chars) {
        for (int i = 0; i < chars.length; i++) {
            int num = Integer.parseInt(String.valueOf(chars[i]));
            if (num >= 7) {
                num = num - 7;
            } else {
                num = num + 3;
            }
            chars[i] = (char) (num + '0');
        }
    }

    /**
     * 将每一位数字+7，然后取余10
     * @param chars 需要转换数字的数组
     */
    private static void convert(char[] chars) {
        for (int i = 0; i < chars.length; i++) {
            int num = Integer.parseInt(String.valueOf(chars[i]));
            num = (num + 7) % 10;
            chars[i] = (char) (num + '0');
        }
    }

    public static void main(String[] args) {
        for (long i = 0; i < 10000000000000000L; i++) {
            Long decode = decode(encodeOrderId(i));
            if (decode != i) {
                System.out.println(i);
                return;
            }
        }
    }

    public static void test(Long id) {
        Long encode = encodeOrderId(id);
        System.out.println(encode);
        Long decode = decode(encode);
        System.out.println(decode);
    }
}
