package cn.com.duiba.kjy.api.util;

import cn.com.duiba.kjy.api.enums.push.PushEventEnum;
import cn.com.duiba.kjy.api.enums.push.PushMessageTypeEnum;
import cn.com.duiba.wolf.utils.UrlUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author lizhi
 * @date 2019/7/16 11:08 AM
 */
public class PushEmbedUtils {

    private static final String SEPARATOR = "_";

    /**
     * 将推送参数转换成一个参数值
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送参数值
     */
    public static String getPushParamStr(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return (oaId == null ? "" : IdMakerUtil.encodingId(oaId)) +  SEPARATOR + (eventEnum == null ? "" : eventEnum.getEventType()) + SEPARATOR
                + ( messageTypeEnum == null ? "" : messageTypeEnum.getType()) + SEPARATOR + (urlOrder == null ? "" : urlOrder)
                + SEPARATOR + (dpm == null ? "" : dpm);
    }

    /**
     * 将推送参数值，转换成推送参数map
     * @param pushParamStr 推送参数值
     * @return 推送参数map
     */
    public static Map<String, String> getPushParamMapByStr(String pushParamStr) {
        if (StringUtils.isBlank(pushParamStr)) {
            return Collections.emptyMap();
        }
        Long oaId = null;
        String pushId = null;
        Integer messageType = null;
        Integer urlOrder = null;
        String dpm = null;
        String[] split = pushParamStr.split(SEPARATOR);
        for (int i = 0; i < split.length; i++) {
            String str = split[i];
            if (StringUtils.isBlank(str)) {
                continue;
            }
            if (i == 0) {
                oaId = IdMakerUtil.decodingId(str);
            }
            if (i == 1) {
                pushId = str;
            }
            if (i == 2 && NumberUtils.isNumber(str)) {
                messageType = Integer.parseInt(str);
            }
            if (i == 3 && NumberUtils.isNumber(str)) {
                urlOrder = Integer.parseInt(str);
            }
            if (i == 4) {
                dpm = str;
            }
        }
        return getPushParamMap(oaId, pushId, messageType, urlOrder, dpm);
    }

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getPushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> pushParamMap = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm);
        return UrlUtils.buildURLParams(pushParamMap);
    }

    /**
     * 获取map中的推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数（不包含dpm）
     */
    public static String getPushParam(Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils.buildURLParams(pushParamMap);
    }

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getEncodePushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return encode(getPushParam(oaId,eventEnum, messageTypeEnum, urlOrder, dpm));
    }

    /**
     * URL增加推送埋点参数
     * @param url 链接
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 带参数的URL
     */
    public static String buildPushParam(String url, Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> params = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm);
        int index = url.indexOf("?");
        if (index < 0) {
            return UrlUtils.appendParams(url, params);
        }
        Map<String, String> map = UrlUtils.uRLRequest(url);
        if (MapUtils.isEmpty(map)) {
            return UrlUtils.appendParams(url, params);
        }
        String uriUrl = map.get("uri");
        if (StringUtils.isBlank(uriUrl)) {
            return UrlUtils.appendParams(url, params);
        }
        String decode = decode(uriUrl);
        String decodePushUrl = UrlUtils.appendParams(decode, params);
        map.put("uri", encode(decodePushUrl));
        return UrlUtils.appendParams(UrlUtils.urlPage(url), map);
    }

    /**
     * URL追加map中的推送埋点参数（不包含dpm）
     * @param url 链接
     * @param parameterMap request中的所有参数
     * @return 带推送参数的URL
     */
    public static String buildPushParam(String url, Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils.appendParams(url, pushParamMap);
    }

    /**
     * 封装推送埋点参数map
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        String pushId = Objects.isNull(eventEnum) ? null : eventEnum.getEventType();
        Integer messageType = Objects.isNull(messageTypeEnum) ? null : messageTypeEnum.getType();
        return getPushParamMap(oaId,pushId, messageType, urlOrder, dpm);
    }

    /**
     * 封装推送埋点参数map
     * @param oaId 公众号ID
     * @param pushId 推送事件Id
     * @param messageType 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, String pushId, Integer messageType, Integer urlOrder, String dpm) {
        Map<String, String> params = new HashMap<>(16);
        params.put("push", "push");
        if (Objects.nonNull(oaId)) {
            params.put("oa_id", IdMakerUtil.encodingId(oaId));
        }
        if (Objects.nonNull(pushId)) {
            params.put("push_id", pushId);
        }
        if (Objects.nonNull(messageType)) {
            params.put("message_type", String.valueOf(messageType));
        }
        if (Objects.nonNull(urlOrder)) {
            params.put("url_order", String.valueOf(urlOrder));
        }
        params.put("access_source", String.valueOf(1));
        if (StringUtils.isNotBlank(dpm)) {
            params.put("dpm", dpm);
        }
        return params;
    }

    /**
     * 获取请求中推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数map
     */
    public static Map<String, String> getPushParamMap(Map<String, String[]> parameterMap) {
        if (MapUtils.isEmpty(parameterMap)) {
            return Collections.emptyMap();
        }
        Map<String, String> params = new HashMap<>(16);
        Set<String> pushKey = getPushParamMap(0L, "", 0, 0, null).keySet();
        for (String key : pushKey) {
            String[] param = parameterMap.get(key);
            if (isNotEmpty(param)) {
                params.put(key, param[0]);
            }
        }
        return params;
    }

    /**
     * 根据推送事件，获取推送点击埋点
     * @param eventEnum 推送事件
     * @return 点击埋点
     */
    public static String getDpmByPushEventEnum(PushEventEnum eventEnum) {
        if (eventEnum == null) {
            return "";
        }
        if (eventEnum == PushEventEnum.STATISTICS_DAY) {
            return "11.9.0.0";
        }
        if (eventEnum == PushEventEnum.STATISTICS_WEEK) {
            return "11.10.0.0";
        }
        if (eventEnum == PushEventEnum.SUBSCRIBE_WELCOME) {
            return "11.12.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVATION) {
            return "11.14.0.0";
        }
        if (eventEnum == PushEventEnum.VISIT_FIRST) {
            return "11.15.0.0";
        }
        if (eventEnum == PushEventEnum.VISIT_NON_FIRST) {
            return "11.16.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT) {
            return "11.17.0.0";
        }
        if (eventEnum ==  PushEventEnum.PREFERENTIAL_INFORMATION) {
            return "11.18.0.0";
        }
        if (eventEnum == PushEventEnum.PRIVATE_CHAT_ACCESS) {
            return "11.19.0.0";
        }
        if (eventEnum == PushEventEnum.PRIVATE_CHAT_SEND) {
            return "11.20.0.0";
        }
        if (eventEnum == PushEventEnum.DISTRIBUTION) {
            return "11.22.0.0";
        }
        if (eventEnum == PushEventEnum.REWARDED) {
            return "11.23.0.0";
        }
        if (eventEnum == PushEventEnum.STATISTICS_VIP_INCOME) {
            return "11.24.0.0";
        }
        if (eventEnum == PushEventEnum.INTERACTIVE_PUSH) {
            return "11.25.0.0";
        }
        if (eventEnum == PushEventEnum.SELLER_CARD_REAL_TIME_PUSH) {
            return "11.26.0.0";
        }
        if (eventEnum == PushEventEnum.FORWARD_CONTENT_PUSH) {
            return "11.27.0.0";
        }
        if (eventEnum == PushEventEnum.FORWARD_SELLER_CARD_PUSH) {
            return "11.28.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT_NON_REAL_TIME) {
            return "11.30.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT_LOTTERY) {
            return "11.31.0.0";
        }
        if (eventEnum == PushEventEnum.WEEKLY_THUMBS_UP) {
            return "11.32.0.0";
        }
        if (eventEnum == PushEventEnum.SCAN_PROM) {
            return "11.34.0.0";
        }
        return "";
    }

    /**
     * URL编码
     * @param url 链接
     * @return 编码后的链接
     */
    public static String encode(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    /**
     * URL解码
     * @param url 链接
     * @return 解码后的链接
     */
    public static String decode(String url) {
        try {
            return URLDecoder.decode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    private static boolean isNotEmpty(String[] param) {
        return Objects.nonNull(param) && param.length > 0;
    }
}
