package cn.com.duiba.kjy.api.util;

import cn.com.duiba.kjy.api.enums.push.PushEventEnum;
import cn.com.duiba.kjy.api.enums.push.PushMessageTypeEnum;
import cn.com.duiba.wolf.utils.UrlUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author lizhi
 * @date 2019/7/16 11:08 AM
 */
public class PushEmbedUtils {

    private static final String SEPARATOR = "_";

    /**
     * 将推送参数转换成一个参数值
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送参数值
     */
    public static String getPushParamStr(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return (oaId == null ? "" : IdMakerUtil.encodingIdByBase64(oaId)) +  SEPARATOR + (eventEnum == null ? "" : eventEnum.getEventType()) + SEPARATOR
                + ( messageTypeEnum == null ? "" : messageTypeEnum.getType()) + SEPARATOR + (urlOrder == null ? "" : urlOrder)
                + SEPARATOR + (dpm == null ? "" : dpm);
    }

    /**
     * 将推送参数转换成一个参数值（oaId不加密）
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送参数值
     */
    public static String getPushParamStrShort(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return (oaId == null ? "" : (oaId + 3)) +  SEPARATOR + (eventEnum == null ? "" : eventEnum.getEventType()) + SEPARATOR
                + ( messageTypeEnum == null ? "" : messageTypeEnum.getType()) + SEPARATOR + (urlOrder == null ? "" : urlOrder)
                + SEPARATOR + (dpm == null ? "" : dpm);
    }

    /**
     * 将推送参数值，转换成推送参数map
     * @param pushParamStr 推送参数值
     * @return 推送参数map
     */
    public static Map<String, String> getPushParamMapByStr(String pushParamStr) {
        if (StringUtils.isBlank(pushParamStr)) {
            return Collections.emptyMap();
        }
        Long oaId = null;
        String pushId = null;
        Integer messageType = null;
        Integer urlOrder = null;
        String dpm = null;
        String[] split = pushParamStr.split(SEPARATOR);
        for (int i = 0; i < split.length; i++) {
            String str = split[i];
            if (StringUtils.isBlank(str)) {
                continue;
            }
            if (i == 0) {
                if (NumberUtils.isNumber(str)) {
                    oaId = Long.parseLong(str) - 3;
                } else {
                    oaId = IdMakerUtil.decodingId(str);
                }
            }
            if (i == 1) {
                pushId = str;
            }
            if (i == 2 && NumberUtils.isNumber(str)) {
                messageType = Integer.parseInt(str);
            }
            if (i == 3 && NumberUtils.isNumber(str)) {
                urlOrder = Integer.parseInt(str);
            }
            if (i == 4) {
                dpm = str;
            }
        }
        return getPushParamMap(oaId, pushId, messageType, urlOrder, dpm, null);
    }

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getPushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> pushParamMap = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm, null);
        return UrlUtils.buildURLParams(pushParamMap);
    }

    /**
     * 获取map中的推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数（不包含dpm）
     */
    public static String getPushParam(Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils.buildURLParams(pushParamMap);
    }

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getEncodePushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return encode(getPushParam(oaId,eventEnum, messageTypeEnum, urlOrder, dpm));
    }

    /**
     * URL增加推送埋点参数
     * @param url 链接
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 带参数的URL
     */
    public static String buildPushParam(String url, Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return buildPushParam(url, oaId, eventEnum, messageTypeEnum, urlOrder, dpm, null);
    }

    /**
     * URL增加推送埋点参数
     * @param url 链接
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @param markValue 唯一标识，为空则不加唯一标识
     * @return 带参数的URL
     */
    public static String buildPushParam(String url, Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm, String markValue) {
        Map<String, String> params = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm, markValue);
        int index = url.indexOf("?");
        if (index < 0) {
            return UrlUtils.appendParams(url, params);
        }
        Map<String, String> map = UrlUtils.uRLRequest(url);
        if (MapUtils.isEmpty(map)) {
            return UrlUtils.appendParams(url, params);
        }
        String uriUrl = map.get("uri");
        if (StringUtils.isBlank(uriUrl)) {
            return UrlUtils.appendParams(url, params);
        }
        String decode = decode(uriUrl);
        String decodePushUrl = UrlUtils.appendParams(decode, params);
        map.put("uri", encode(decodePushUrl));
        return UrlUtils.appendParams(UrlUtils.urlPage(url), map);
    }

    /**
     * 增加推送唯一标识
     * @param page 小程序模板消息，跳转页面
     * @param markValue 唯一标识
     * @return 带参数的页面地址
     */
    public static String buildMpPushParam(String page, String markValue) {
        Map<String, String> params = new HashMap<>();
        if (markValue == null) {
            markValue = "";
        }
        params.put("unique_mark", markValue);
        return UrlUtils.appendParams(page, params);
    }

    /**
     * URL追加map中的推送埋点参数（不包含dpm）
     * @param url 链接
     * @param parameterMap request中的所有参数
     * @return 带推送参数的URL
     */
    public static String buildPushParam(String url, Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils.appendParams(url, pushParamMap);
    }

    /**
     * 获取请求中推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数map
     */
    public static Map<String, String> getPushParamMap(Map<String, String[]> parameterMap) {
        if (MapUtils.isEmpty(parameterMap)) {
            return Collections.emptyMap();
        }
        Map<String, String> params = new HashMap<>(16);
        Set<String> pushKey = getPushParamMap(0L, "", 0, 0, null, "").keySet();
        for (String key : pushKey) {
            String[] param = parameterMap.get(key);
            if (isNotEmpty(param)) {
                params.put(key, param[0]);
            }
        }
        return params;
    }

    /**
     * 根据推送事件，获取推送点击埋点
     * @param eventEnum 推送事件
     * @return 点击埋点
     */
    public static String getDpmByPushEventEnum(PushEventEnum eventEnum) {
        if (eventEnum == null) {
            return "";
        }
        if (eventEnum == PushEventEnum.STATISTICS_DAY) {
            return "11.9.0.0";
        }
        if (eventEnum == PushEventEnum.STATISTICS_WEEK) {
            return "11.10.0.0";
        }
        if (eventEnum == PushEventEnum.SUBSCRIBE_WELCOME) {
            return "11.12.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVATION) {
            return "11.14.0.0";
        }
        if (eventEnum == PushEventEnum.VISIT_FIRST) {
            return "11.15.0.0";
        }
        if (eventEnum == PushEventEnum.VISIT_NON_FIRST) {
            return "11.16.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT) {
            return "11.17.0.0";
        }
        if (eventEnum ==  PushEventEnum.PREFERENTIAL_INFORMATION) {
            return "11.18.0.0";
        }
        if (eventEnum == PushEventEnum.PRIVATE_CHAT_ACCESS) {
            return "11.19.0.0";
        }
        if (eventEnum == PushEventEnum.PRIVATE_CHAT_SEND) {
            return "11.20.0.0";
        }
        if (eventEnum == PushEventEnum.DISTRIBUTION) {
            return "11.22.0.0";
        }
        if (eventEnum == PushEventEnum.REWARDED) {
            return "11.23.0.0";
        }
        if (eventEnum == PushEventEnum.STATISTICS_VIP_INCOME) {
            return "11.24.0.0";
        }
        if (eventEnum == PushEventEnum.INTERACTIVE_PUSH) {
            return "11.25.0.0";
        }
        if (eventEnum == PushEventEnum.SELLER_CARD_REAL_TIME_PUSH) {
            return "11.26.0.0";
        }
        if (eventEnum == PushEventEnum.FORWARD_CONTENT_PUSH) {
            return "11.27.0.0";
        }
        if (eventEnum == PushEventEnum.FORWARD_SELLER_CARD_PUSH) {
            return "11.28.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT_NON_REAL_TIME) {
            return "11.30.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT_LOTTERY) {
            return "11.31.0.0";
        }
        if (eventEnum == PushEventEnum.WEEKLY_THUMBS_UP) {
            return "11.32.0.0";
        }
        if (eventEnum == PushEventEnum.SCAN_PROM) {
            return "11.34.0.0";
        }
        if (eventEnum == PushEventEnum.NATIONAL_DAY_ACTIVITY) {
            return "11.37.0.0";
        }
        //如果dpm没有特殊要求，请不要再加if，走下面的方法自动计算dpm
        return calculateDpm(eventEnum);
    }

    /**
     * 封装推送埋点参数map
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm, String markValue) {
        String pushId = Objects.isNull(eventEnum) ? null : eventEnum.getEventType();
        Integer messageType = Objects.isNull(messageTypeEnum) ? null : messageTypeEnum.getType();
        return getPushParamMap(oaId,pushId, messageType, urlOrder, dpm, markValue);
    }

    /**
     * 封装推送埋点参数map
     * @param oaId 公众号ID
     * @param pushId 推送事件Id
     * @param messageType 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, String pushId, Integer messageType, Integer urlOrder, String dpm, String markValue) {
        Map<String, String> params = new HashMap<>(16);
        params.put("push", "push");
        if (Objects.nonNull(oaId)) {
            params.put("oa_id", IdMakerUtil.encodingIdByBase64(oaId));
        }
        if (Objects.nonNull(pushId)) {
            params.put("push_id", pushId);
        }
        if (Objects.nonNull(messageType)) {
            params.put("message_type", String.valueOf(messageType));
        }
        if (Objects.nonNull(urlOrder)) {
            params.put("url_order", String.valueOf(urlOrder));
        }
        params.put("access_source", getAccessSourceByPushId(pushId));
        if (StringUtils.isNotBlank(dpm)) {
            params.put("dpm", dpm);
        }
        if (markValue != null) {
            params.put("unique_mark", markValue);
        }
        return params;
    }

    /**
     * 根据pushId获取accessSource
     * @param pushId
     * @return
     */
    private static String getAccessSourceByPushId(String pushId) {
        if (StringUtils.isBlank(pushId) || pushId.length() < 2) {
            return "1";
        }
        //根据pushId的首字母，得出数字
        String numByPushId = getNumByPushId(pushId);
        if (StringUtils.isBlank(numByPushId)) {
            return "1";
        }
        //accessSource= 10 + 首字母对应数字 + pushId的数字（不足3位补0）
        String accessSource = "10";
        accessSource += numByPushId;
        String num = pushId.substring(1);
        while (num.length() < 3) {
            num = "0" + num;
        }
        return accessSource + num;
    }

    /**
     * 根据pushId的首字母获取对应数字
     * @param pushId 推送ID
     * @return 首字母对应数字
     */
    private static String getNumByPushId(String pushId) {
        if (StringUtils.isBlank(pushId) || pushId.length() < 1) {
            return "";
        }
        String word = pushId.substring(0, 1);
        switch (word) {
            case "S":
                return "1";
            case "D":
                return  "2";
            case "H":
                return  "3";
            case "V":
                return  "4";
            case "X":
                return  "5";
            case "Z":
                return "6";
            default:
                return  "";
        }
    }

    /**
     * 根据eventEnum计算dpm
     * @param eventEnum
     * @return
     */
    private static String calculateDpm(PushEventEnum eventEnum) {
        //dpm由4位组成，用"."隔开
        //规则：11.0. + 推送ID首字母对应数字 + . + 推送ID数字
        if (eventEnum == null) {
            return "";
        }
        String pushId = eventEnum.getEventType();
        if (pushId.length() < 2) {
            return "";
        }
        String numByPushId = getNumByPushId(pushId);
        if (StringUtils.isBlank(numByPushId)) {
            return "";
        }
        return "11.0." + numByPushId + "." + pushId.substring(1);
    }

    /**
     * URL编码
     * @param url 链接
     * @return 编码后的链接
     */
    public static String encode(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    /**
     * URL解码
     * @param url 链接
     * @return 解码后的链接
     */
    public static String decode(String url) {
        try {
            return URLDecoder.decode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    private static boolean isNotEmpty(String[] param) {
        return Objects.nonNull(param) && param.length > 0;
    }

    /**
     * 给文本信息插入唯一标识
     * @param msg 文本信息
     * @param eventEnum 推送事件
     * @return 插入唯一标识后的文本
     */
    public static String insertUniqueMark(String msg, PushEventEnum eventEnum, String markValue) {
        if (StringUtils.isBlank(msg) || eventEnum == null) {
            return msg;
        }
        String pushIdKey = "push_id";
        //最终结果收集
        StringBuilder sb = new StringBuilder();
        //push_id的长度
        int pushIdKeyLength = pushIdKey.length();
        //push_id的值
        String eventType = eventEnum.getEventType();
        //push_id值的长度
        int eventTypeLength = eventType.length();
        int lastIndex = 0;
        for (; lastIndex < msg.length();) {
            //push_id开始的下标
            int pushIdIndex = msg.indexOf(pushIdKey, lastIndex);
            if (pushIdIndex < 0) {
                //没有push_id
                break;
            }
            //push_id值开始的下标
            int eventTypeIndex = msg.indexOf(eventType, pushIdIndex);
            if (eventTypeIndex < 0) {
                //没有push_id值，非推送参数
                break;
            }
            //push_id结束的下标
            int pushIdLastIndex = pushIdIndex + pushIdKeyLength;
            //push_id值结束的下标
            int eventTypeLastIndex = eventTypeIndex + eventTypeLength;
            //获取push_id和值之间的符号，用于判断是否编码过
            String equalSign = msg.substring(pushIdLastIndex, eventTypeIndex);
            //获取编码次数
            int count = encodeCount(equalSign);
            //根据编码次数，获取要插入的唯一标识
            String mark = getMark(count, markValue);
            //追加push_id之前的数据
            sb.append(msg, lastIndex, pushIdIndex);
            //追加唯一标识
            sb.append(mark);
            //追加push_id及push_id的值
            sb.append(msg, pushIdIndex, eventTypeLastIndex);
            //将处理下标移动到push_id值后
            lastIndex = eventTypeLastIndex;
        }
        sb.append(msg.substring(lastIndex));
        return sb.toString();
    }

    /**
     * 根据编码次数，获取唯一标识
     * @param encodeCount 编码次数
     * @return 唯一标识
     */
    private static String getMark(int encodeCount, String markValue) {
        String mark = "unique_mark=" + markValue + "&";
        if (encodeCount <= 0) {
            return mark;
        }
        for (int i = 0; i < encodeCount; i++) {
            mark = encode(mark);
        }
        return mark;
    }

    /**
     * 获取编码次数
     * @param equalSign 编码后的 =
     * @return 编码次数
     */
    private static int encodeCount(String equalSign) {
        int encodeCount = 0;
        if (StringUtils.isBlank(equalSign)) {
            return encodeCount;
        }
        String sign = "=";
        int maxCount = 5;
        for (int i = 0; i < maxCount; i++) {
            if (StringUtils.equals(equalSign, sign)) {
                return encodeCount;
            }
            equalSign = decode(equalSign);
            encodeCount++;
        }
        return encodeCount;
    }
}
