package cn.com.duiba.kjy.api.util;

import cn.com.duiba.kjy.api.enums.push.PushEventEnum;
import cn.com.duiba.kjy.api.enums.push.PushMessageTypeEnum;
import cn.com.duiba.wolf.utils.UrlUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @author lizhi
 * @date 2019/7/16 11:08 AM
 */
public class PushEmbedUtils {

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm
     * @return 推送埋点参数
     */
    public static String getPushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> pushParamMap = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm);
        return UrlUtils.buildURLParams(pushParamMap);
    }

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm
     * @return 推送埋点参数
     */
    public static String getEncodePushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return encode(getPushParam(oaId,eventEnum, messageTypeEnum, urlOrder, dpm));
    }

    /**
     * URL增加推送埋点参数
     * @param url 链接
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm
     * @return 带参数的URL
     */
    public static String buildPushParam(String url, Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> params = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm);
        int index = url.indexOf("?");
        if (index < 0) {
            return UrlUtils.appendParams(url, params);
        }
        Map<String, String> map = UrlUtils.uRLRequest(url);
        if (MapUtils.isEmpty(map)) {
            return UrlUtils.appendParams(url, params);
        }
        String uriUrl = map.get("uri");
        if (StringUtils.isBlank(uriUrl)) {
            return UrlUtils.appendParams(url, params);
        }
        String decode = decode(uriUrl);
        String decodePushUrl = UrlUtils.appendParams(decode, params);
        map.put("uri", encode(decodePushUrl));
        return UrlUtils.appendParams(UrlUtils.urlPage(url), map);
    }

    private static Map<String, String> getPushParamMap(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> params = new HashMap<>(16);
        params.put("push", "push");
        if (Objects.nonNull(oaId)) {
            params.put("oa_id", IdMakerUtil.encodingId(oaId));
        }
        if (Objects.nonNull(eventEnum)) {
            params.put("push_id", eventEnum.getEventType());
        }
        if (Objects.nonNull(messageTypeEnum)) {
            params.put("message_type", String.valueOf(messageTypeEnum.getType()));
        }
        if (Objects.nonNull(urlOrder)) {
            params.put("url_order", String.valueOf(urlOrder));
        }
        params.put("access_source", String.valueOf(1));
        if (StringUtils.isNotBlank(dpm)) {
            params.put("dpm", dpm);
        }
        return params;
    }

    private static String encode(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    private static String decode(String url) {
        try {
            return URLDecoder.decode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }
}
