package cn.com.duiba.kjy.livecenter.api.util;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

/**
 * Created by dugq on 2020-06-03.
 */
@Slf4j
public class LotteryCodeUtil {
    /**
     * 2进制 10101010101010010101010101 (26位)
     * 根据 2的n次方结果位数小于9 得出的最大n值 为 26
     * 为了让数字乱序一点，不是从1-9按序排列。所以用一个key进行一次异或操作
     */
    private static final int ENCODE_NUMBER_AS_10_STRING_KEY = 44737877;

    /**
     * 一共只有8位数字，所能承载的数字是有限的。
     * 26位进制 支持的最大值 （2的26次方-1）
     * 超过max以后，将不能再还原了。（加密时对最大值进行取模运算。）
     */
    private static final int ENCODE_NUMBER_AS_10_STRING_MAX = 67108863;
    /**
     * 私有化
     */
    private LotteryCodeUtil(){ }

    /**
     * 根据输入数字，计算出一个N****K****的code码
     *  超过 {@value ENCODE_NUMBER_AS_10_STRING_MAX} 后归0重新开始
     * @param value 任意数字
     * @return code码
     */
    public static String encodeNumberAs10String(long value){

        if (value>ENCODE_NUMBER_AS_10_STRING_MAX){  // 超过最大值的数字：取模（利用位运算，比%效率高）
            //把 大于 16777216的数字 进行 i = i % key 计算
            value = value & 2 ^ 26;
        }

        //用异或运算调整下26位二进制数字  让数字混乱，看不出规律
        value = value ^ ENCODE_NUMBER_AS_10_STRING_KEY;

        // 把二进制26位数字 转化位8位十进制字符串，不足8位的，在前面补0
        String val = StringUtils.leftPad(String.valueOf(value),8,"0");
        //把code 装成 N****K****的 形式
        return "N" + val.substring(0,4) + "K" + val.substring(4);
    }

    /**
     * encode的逆运算
     * 只能正确还原 67108863 以下的数字
     * @param value
     * @return
     */
    public static Long decode10StringAsNumber(String value){
        if (StringUtils.isBlank(value) || value.length()!=10){
            return null;
        }
        //去掉N和K。 截取效率优于replace
        value = value.substring(1,5)+value.substring(6);
        try {
            Long number = Long.valueOf(value);
            //利用key还原数字
            return number ^ENCODE_NUMBER_AS_10_STRING_KEY;
        }catch (NumberFormatException e){ //转换错误说明code非法
            return null;
        }
    }





    public static void main(String[] args) {
        Long aLong = decode10StringAsNumber("N4465K9557");
        System.out.println(aLong);
        String encode = encodeNumberAs10String(19975803);
       log.info(encode);
        test1();
        log.info(Integer.valueOf("10101010101010010101010101",2)+"");
        log.info(Integer.valueOf("11111111111111111111111111",2)+"");
        long pow = (long) Math.pow(2, 26);
        log.info(pow+"");
        int length = String.valueOf(pow).length();
        log.info(length+"");
        for (int i = 0 ; i < 1000; i++){
            log.info(i+"="+encodeNumberAs10String(i));
        }
    }

    private static void test1() {
        long start = System.currentTimeMillis();
        for (int i = 0 ; i < 100000000; i++){
            String encode = encodeNumberAs10String(i);
            Long number = decode10StringAsNumber(encode);
            if (i!=number && (i & 2 ^ 26) != number){
                log.info(i+"");
            }
        }
        long end = System.currentTimeMillis();
        log.info(end - start+"");
    }

}
