package cn.com.duiba.kjy.livecenter.api.util;

import cn.com.duiba.kjy.livecenter.api.dto.company.LiveCompanyDto;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.checkerframework.checker.nullness.qual.NonNull;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 公司等级判断工具
 * @author lizhi
 * @date 2020/12/8 2:08 下午
 */
@Slf4j
public class LiveCompanyLevelUtil {

    private LiveCompanyLevelUtil() {}

    /**
     * 获取最低级的公司
     * @param companyDtoList 公司集合
     * @return 最低级的公司
     */
    @Nullable
    public static LiveCompanyDto getLowerLevelLiveCompanyDto(List<LiveCompanyDto> companyDtoList) {
        if (CollectionUtils.isEmpty(companyDtoList)) {
            return null;
        }
        if (companyDtoList.size() == 1) {
            return companyDtoList.get(0);
        }
        //排序
        List<LiveCompanyDto> list = sortByLevel(companyDtoList);
        return getLast(list);
    }

    /**
     * 是否有重复等级的公司
     * @param companyDtoList 公司集合
     * @return true-等级有重复的，false-等级没有重复的
     */
    public static boolean hasDistinctLevel(List<LiveCompanyDto> companyDtoList) {
        if (CollectionUtils.isEmpty(companyDtoList)) {
            return false;
        }
        Set<Integer> levelSet = new HashSet<>();
        for (LiveCompanyDto companyDto : companyDtoList) {
            if (!levelSet.add(companyDto.getCompanyLevel())) {
                return true;
            }
        }
        return false;
    }

    /**
     * 获取重复的公司
     * @param companyDtoList 公司集合
     * @param liveCompanyDto 当前公司
     * @return 第一个与当前公司等级重复的公司
     */
    public static LiveCompanyDto getDistinctLevelCompany(List<LiveCompanyDto> companyDtoList, @NonNull LiveCompanyDto liveCompanyDto) {
        if (CollectionUtils.isEmpty(companyDtoList)) {
            return null;
        }
        Set<Integer> levelSet = new HashSet<>();
        levelSet.add(liveCompanyDto.getCompanyLevel());
        for (LiveCompanyDto companyDto : companyDtoList) {
            if (!levelSet.add(companyDto.getCompanyLevel())) {
                return companyDto;
            }
        }
        return null;
    }

    /**
     * 公司是否在一条从属关系线上
     * @param companyDtoList 公司集合
     * @return true-在一条线上，false-不在一条线上
     */
    public static boolean isLineCompany(List<LiveCompanyDto> companyDtoList) {
        if (CollectionUtils.isEmpty(companyDtoList)) {
            //没有公司
            return false;
        }
        if (companyDtoList.size() == 1) {
            //只有一个公司，肯定是同一条线上的
            return true;
        }
        if (hasDistinctLevel(companyDtoList)) {
            //有重复的等级，不可能在一条线上
            return false;
        }
        //最低级公司
        LiveCompanyDto lowerLevelCompany = getLowerLevelLiveCompanyDto(companyDtoList);
        if (lowerLevelCompany == null) {
            return false;
        }
        //最低级公司的上级ID集合
        String higherIds = lowerLevelCompany.getHigherIds();
        if (StringUtils.isBlank(higherIds)) {
            //都是总公司，肯定不在一条线路上
            return false;
        }
        String[] higherIdStr = higherIds.split(",");
        List<Long> higherIdList = Arrays.stream(higherIdStr).map(Long::parseLong).collect(Collectors.toList());

        for (LiveCompanyDto liveCompanyDto : companyDtoList) {
            if (Objects.equals(liveCompanyDto.getId(), lowerLevelCompany.getId())) {
                //当前公司是级别最低的公司
                continue;
            }
            //不是级别最低的公司
            if (!higherIdList.contains(liveCompanyDto.getId())) {
                //不在最低级公司的链路上
                return false;
            }
        }
        return true;
    }

    /**
     * 获取集合中最后一个元素
     * @param list 集合
     * @param <T> 元素类型
     * @return 最后一个元素
     */
    public static <T> T getLast(List<T> list) {
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        return list.get(list.size() - 1);
    }

    /**
     * 根据公司等级正序排列（总公司-分公司-中心支公司）
     * @param companyDtoList 公司集合
     * @return 排序后的公司集合
     */
    public static List<LiveCompanyDto> sortByLevel(List<LiveCompanyDto> companyDtoList) {
        if (CollectionUtils.isEmpty(companyDtoList)) {
            return companyDtoList;
        }
        return companyDtoList.stream().sorted(Comparator.comparing(LiveCompanyDto::getCompanyLevel)).collect(Collectors.toList());
    }

    /**
     * 获取同级别公司
     * @param companyDtoList 所有公司列表
     * @param companyDto 当前公司
     * @return 同级别
     */
    public static LiveCompanyDto getSameLevelCompany(List<LiveCompanyDto> companyDtoList, LiveCompanyDto companyDto) {
        Map<Integer, List<LiveCompanyDto>> levelMap = companyDtoList.stream().collect(Collectors.groupingBy(LiveCompanyDto::getCompanyLevel));
        List<LiveCompanyDto> liveCompanyDtoList = levelMap.get(companyDto.getCompanyLevel());
        if (CollectionUtils.isEmpty(liveCompanyDtoList)) {
            return null;
        }
        if (liveCompanyDtoList.size() > 1) {
            log.error("liveActivate, has save level company");
        }
        return liveCompanyDtoList.get(0);
    }
}
