package cn.com.duiba.linglong.client.service.logger;

import cn.com.duiba.linglong.client.domain.dto.JobKey;
import cn.com.duiba.linglong.client.domain.responce.FetchJobLogResult;
import cn.com.duiba.wolf.utils.UUIDUtils;
import com.google.common.collect.Sets;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.map.IMap;
import com.hazelcast.multimap.MultiMap;
import org.apache.commons.lang.StringUtils;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Collection;
import java.util.Objects;
import java.util.Set;

/**
 * @author liuyao
 */
public class JobLogSessionService {

    private final static long SESSION_TIMEOUT = 10000L;

    @Resource
    private HazelcastInstance linglongHazelcastClient;
    /**
     * 日志带消费容器
     */
    private MultiMap<String,JobLog> jobLogMultimap;
    /**
     * jobKey->sessionId 维护会话和日志的订阅关系
     */
    private MultiMap<String,String> sessionMultimap;
    /**
     * 会话的时间戳
     */
    private IMap<String,Long> sessionTimeMap;
    /**
     * 会话和任务的逆向关系
     */
    private IMap<String,String> sessionBiMap;

    @PostConstruct
    public void init(){
        jobLogMultimap = linglongHazelcastClient.getMultiMap("JobLogSessionService.jobLogMultimap");
        sessionMultimap = linglongHazelcastClient.getMultiMap("JobLogSessionService.sessionMultimap");
        sessionTimeMap = linglongHazelcastClient.getMap("JobLogSessionService.sessionTimeMap");
        sessionBiMap = linglongHazelcastClient.getMap("JobLogSessionService.sessionBiMap");
    }

    public String createSession(JobKey jobKey) {
        String sessionId = UUIDUtils.createUUID();
        sessionTimeMap.put(sessionId,System.currentTimeMillis());
        sessionMultimap.put(jobKey.key(),sessionId);
        sessionBiMap.put(sessionId,jobKey.key());
        return sessionId;
    }

    public FetchJobLogResult fetchJobLog(String sessionId){
        FetchJobLogResult result = new FetchJobLogResult();
        boolean hasSession = sessionTimeMap.containsKey(sessionId);
        if(hasSession){
            sessionTimeMap.put(sessionId,System.currentTimeMillis());
        }
        result.setHasSession(hasSession);
        result.setLogs(jobLogMultimap.remove(sessionId));
        return result;
    }

    public void closeSession(String sessionId){
        sessionTimeMap.remove(sessionId);
        jobLogMultimap.remove(sessionId);

        String jobKey = sessionBiMap.remove(sessionId);
        if(StringUtils.isNotBlank(jobKey)){
            sessionMultimap.remove(jobKey,sessionId);
        }
    }

    public void scanSession(){
        Set<String> sessionIdSet = Sets.newHashSet(sessionBiMap.keySet());
        for(String sessionId:sessionIdSet){
            Long time = sessionTimeMap.get(sessionId);
            if(Objects.isNull(time) || time-System.currentTimeMillis()> SESSION_TIMEOUT){
                closeSession(sessionId);
            }
        }
    }

    void acceptLog(JobKey jobKey, JobLog jobLog) {
        Collection<String> sessionIds = sessionMultimap.get(jobKey.key());
        for(String sessionId:sessionIds){
            if(!sessionTimeMap.containsKey(sessionId)){
                continue;
            }
            jobLogMultimap.put(sessionId,jobLog);
        }
    }

    public void releaseLog(JobKey jobKey) {
        Collection<String> sessionIds = sessionMultimap.get(jobKey.key());
        for(String sessionId:sessionIds){
            sessionTimeMap.remove(sessionId);
        }
        sessionMultimap.remove(jobKey.key());
    }
}
