package cn.com.duiba.linglong.client.job.render;

import cn.com.duiba.linglong.client.job.jobs.JobContext;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;

import javax.annotation.Resource;
import java.io.StringWriter;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author liuyao
 */
@Slf4j
public class ConfigRender{

    private static final String INSTANTIATING = "$$INSTANTIATING";

    private static final Pattern PATTERN = Pattern.compile("\\$\\{(.+?)\\}");

    public static final Set<String> TIME_KEY_PREFIX = ImmutableSet.of("now","zdt");

    @Resource
    private VelocityEngine velocityEngine;

    private final Map<String,String> cache = Maps.newHashMap();

    private final Stack<String> stack = new Stack<>();

    private Properties repository;

    private final VelocityContext velocityContext = new VelocityContext();

    public String getRenderConfig(String key){
        if (cache.containsKey(key)) {
            if (StringUtils.equals(cache.get(key),INSTANTIATING)) {
                List<String> circularKeys = Lists.newArrayList(stack);
                circularKeys.add(key);
                Joiner joiner = Joiner.on("->").skipNulls();
                throw new RuntimeException("配置循环依赖:"+joiner.join(circularKeys));
            }
            return cache.get(key);
        }
        try {
            stack.push(key);
            //1.预先设置正在实例化的占位
            cache.put(key,INSTANTIATING);

            //2.查询依赖的key
            String sourceValue = getRepositoryValue(key);
            Set<String> dependencies = analyseScript(sourceValue);
            Properties dependencieKeyValues = new Properties();
            for(String preKey:dependencies){
                dependencieKeyValues.put(preKey,getRenderConfig(preKey));
            }
            //3.递归的结果替换占位符
            String renderValue = replaceArgs(sourceValue,dependencieKeyValues);
            cache.put(key,renderValue);
            //4.返回实例化的结果
            return cache.get(key);
        } catch (Exception e) {
            if (Objects.equals(cache.get(key),INSTANTIATING)){
                cache.remove(key);
            }
            throw e;
        }finally {
            stack.pop();
        }
    }

    private String getRepositoryValue(String key){
        if(isDateToolKey(key)){
            StringWriter writer = new StringWriter();
            String inString = "$!{" + key + "}";
            try {
                velocityEngine.evaluate(velocityContext, writer, "",inString);
            } catch (Exception e) {
                log.debug("配置渲染失败", e);
            }
            return writer.toString();
        }
        return repository.getProperty(key,"");
    }

    private boolean isDateToolKey(String key){
        for(String item:TIME_KEY_PREFIX){
            if(StringUtils.equals(key,item) || StringUtils.startsWith(key,item+".")){
                return true;
            }
        }
        return false;
    }



    public void setRepository(JobContext jobContext) {
        this.repository = jobContext;
        Objects.requireNonNull(jobContext.getTime());
        //注入时间工具
        DateTool dateTool = new DateTool(jobContext.getTime());
        for(String key:TIME_KEY_PREFIX){
            velocityContext.put(key, dateTool);
        }
    }

    public static Set<String> analyseScript(String script) {
        if (StringUtils.isBlank(script)) {
            return Collections.emptySet();
        }
        Set<String> keys = Sets.newHashSet();
        Matcher matcher = PATTERN.matcher(script);
        while (matcher.find()) {
            keys.add(matcher.group(1));
        }
        return keys;
    }

    /**
     * 替换模板变量
     */
    public static String replaceArgs(String template, Properties data) {
        if(StringUtils.isBlank(template)){
            return "";
        }
        StringBuffer sb = new StringBuffer();
        try {
            Matcher matcher = PATTERN.matcher(template);
            while (matcher.find()) {
                String name = matcher.group(1);
                String value = "";
                if (data.containsKey(name)) {
                    value = data.getProperty(name,"");
                }
                if (StringUtils.isNotBlank(value)) {
                    value = value.replaceAll("\\\\", "\\\\\\\\");
                    matcher.appendReplacement(sb, value);
                }
            }
            matcher.appendTail(sb);
        } catch (Exception e) {
            log.error("渲染失败", e);
        }
        return sb.toString();
    }

    public static void main(String[] args) {
        String sss = "dsds:${@now}:dsdsds";
        Properties properties = new Properties();
        System.out.println(replaceArgs(sss,properties));
    }

}
