package cn.com.duiba.linglong.client.job.consumer;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.linglong.client.constant.JobInvokeType;
import cn.com.duiba.linglong.client.domain.dto.JobKey;
import cn.com.duiba.linglong.client.domain.params.JobInvokerParams;
import cn.com.duiba.linglong.client.job.jobs.WorkerScheduleJobManager;
import cn.com.duiba.linglong.client.remoteservice.RemoteWorkerStatusService;
import cn.com.duiba.linglong.client.service.channel.JobInvoker;
import cn.com.duibaboot.ext.autoconfigure.httpclient.ssre.CanAccessInsideNetwork;
import com.google.common.util.concurrent.AbstractExecutionThreadService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.event.EventListener;
import org.springframework.web.client.RestTemplate;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 任务消费者
 * @author liuyao
 */
@Slf4j
public class WorkerJobConsumer extends AbstractExecutionThreadService {

    @Resource
    private List<JobConsumerAssert> jobConsumerAsserts;
    @Resource
    private WorkerScheduleJobManager workerScheduleJobManager;
    @Resource
    private RemoteWorkerStatusService remoteWorkerStatusService;
    @Resource
    @CanAccessInsideNetwork
    private RestTemplate linglongFetchJobRestTemplate;

    @Value("${spring.application.name}")
    private String appName;
    @Value("${server.port}")
    private int port;
    /**
     * 是否和调度器保持联通状态
     */
    private final AtomicBoolean isLink = new AtomicBoolean(false);

    private String workerId;

    private Thread heartbeat;

    @PostConstruct
    public void init(){
        workerId = NetUtils.getLocalIp()+":"+port;
    }

    @EventListener(MainContextRefreshedEvent.class)
    public void startConsumer(){
        this.startAsync();
        WorkerJobConsumer consumer = this;
        //noinspection AlibabaAvoidManuallyCreateThread
        heartbeat = new Thread(() -> {
            consumer.awaitRunning();
            while (isRunning()){
                try {
                    remoteWorkerStatusService.heartbeat(appName,workerId);
                    if(isLink.compareAndSet(false,true)){
                        log.info("调度器通信建立完成");
                    }
                    Thread.sleep(5000L);
                }catch (InterruptedException e) {
                    break;
                }catch (Exception e){
                    log.error("心跳检查失败",e);
                    isLink.set(false);
                    try {
                        Thread.sleep(5000L);
                    } catch (InterruptedException interruptedException) {
                        break;
                    }
                }
            }
        });
        heartbeat.setDaemon(true);
        heartbeat.setName("linglong.client.heartbeat");
        heartbeat.setPriority(Thread.MAX_PRIORITY);
        heartbeat.start();
    }

    @PreDestroy
    public void preDestroy(){
        if(isRunning()){
            remoteWorkerStatusService.stopWorker(appName,workerId);
            this.stopAsync();
        }
    }

    @Override
    protected void triggerShutdown() {
        //终断心跳
        heartbeat.interrupt();
    }

    @Override
    protected void startUp() {
        log.info("Worker job fetch starting.");
    }

    @Override
    public void run(){
        while (isRunning()){
            try {
                if(!canRunJob()){
                    int retryTime = 3;
                    for(int i = 0; i< retryTime; i++){
                        if(!isRunning()){
                            return;
                        }
                        Thread.sleep(1000);
                    }
                    continue;
                }
                if(!isLink.get()){
                    Thread.sleep(3000);
                    continue;
                }
                JobInvokerParams params = new JobInvokerParams();
                params.setAppId(appName);
                JobInvoker invoker = linglongFetchJobRestTemplate.postForObject("http://linglong/worker/fetch/requestJobInvoker",params,JobInvoker.class);
                if(Objects.isNull(invoker)){
                    continue;
                }
                try{
                    JobKey jobKey = new JobKey(JobInvokeType.ACTION,invoker.getHistoryId());
                    workerScheduleJobManager.submitScheduleJob(jobKey,invoker.getJobLevel());
                }catch (Exception e){
                    log.error("任务运行失败",e);
                }
            } catch (InterruptedException e) {
                break;
            }catch (Exception e){
                log.error("任务消费异常",e);
            }
        }
        if(!Thread.currentThread().isInterrupted()){
            Thread.currentThread().interrupt();
        }
    }

    private boolean canRunJob() {
        for(JobConsumerAssert consumerAssert:jobConsumerAsserts){
            if(!consumerAssert.canConsumer()){
                return false;
            }
        }
        return true;
    }

}
