package cn.com.duiba.live.clue.service.api.remoteservice.mall.order;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.live.clue.service.api.dto.mall.order.*;
import cn.com.duiba.live.clue.service.api.param.mall.order.*;
import org.apache.commons.lang3.tuple.Pair;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 商城-订单相关
 *
 * @author xuzhigang
 * @date 2022/8/10 11:42 上午
 **/
@AdvancedFeignClient
public interface RemoteMallOrderService {

    /**
     * 创建订单
     *
     * @param orderDto 订单信息
     * @param extraDto 订单扩展信息
     * @return orderId
     */
    Long creatOrder(MallOrderDto orderDto, MallOrderExtraDto extraDto);

    /**
     * 更新订单
     * @param orderDto 订单信息
     * @param extraDto 订单扩展信息
     * @return orderId
     */
    Long updateOrder(MallOrderDto orderDto, MallOrderExtraDto extraDto);

    /**
     * 批量保存订单 用户维度
     *
     * @param liveUserId2OrderMap key: liveUserId value: order
     * @return key: liveUserId value: orderId
     */
    Map<Long, Long> batchSaveOrder(Map<Long, MallOrderDetailDto> liveUserId2OrderMap);

    /**
     * 根据id查询订单
     * @param id 主键id
     * @return 订单信息
     */
    MallOrderDto selectById(Long id);

    /**
     * 根据订单id集合批量查询
     *
     * @param ids id集合
     * @return 订单集合
     */
    List<MallOrderDto> findByIds(List<Long> ids);

    /**
     * 根据订单id查询订单详情
     *
     * @param orderId 订单id
     * @return 订单详情
     */
    MallOrderDetailDto findOrderDetail(Long orderId);

    /**
     * 根据订单编号查询订单详情
     *
     * @param orderNo 订单编号
     * @return 订单详情
     */
    MallOrderDetailDto findOrderDetailByOrderNo(String orderNo);

    /**
     * 根据订单id集合查询订单列表
     *
     * @param ids 订单id集合
     * @return 订单列表
     */
    List<MallOrderListDto> findOrderListByIds(List<Long> ids);

    /**
     * 查询订单列表
     *
     * @param param 查询订单列表参数
     * @return 订单列表
     */
    List<MallOrderListDto> findOrderList(MallOrderSearchParam param);

    /**
     * 查询订单列表管理后台专用 es 检索版本
     * @param param 查询订单列表参数
     * @return 订单列表
     */
    Pair<Long,List<MallOrderListMngDto>> findOrderList4MngEs(EsMallOrderSearch4MngParam param);

    /**
     * 查询订单列表管理后台专用
     * @param param 查询订单列表参数
     * @return 订单列表
     */
    List<MallOrderListMngDto> findOrderList4Mng(MallOrderSearch4MngParam param);

    long selectCount4Mng(MallOrderSearch4MngParam param);

    /**
     * 根据订单号查询
     * @param orderNo 订单号
     * @return 订单信息
     */
    MallOrderDto selectByOrderNo(String orderNo);

    List<MallOrderDto> batchSelectByOrderNo(List<String> orderNo);


    List<MallOrderDetailDto> batchSelectDetailByOrderNo(List<String> orderNo);

    /**
     * 批量查询订单详情
     *
     * @param orderIds 订单id集合
     * @return 订单详情集合
     */
    List<MallOrderDetailDto> batchSelectDetailByIds(List<Long> orderIds);

    /**
     * 更新
     *
     * @param entity 订单数据
     * @return 执行结果
     */
    int update(MallOrderDto entity);

    /**
     * 新增
     * @param en 订单数据
     * @return 执行结果
     */
    int insert(MallOrderDto en);

    /**
     * 取消订单
     *
     * @param id id
     * @param cancelOrderReason 取消订单原因
     * @return 执行结果
     */
    int cancelOrder(Long id, String cancelOrderReason);

    /**
     * 超时未支付
     *
     * @param id 订单id
     * @return 执行结果
     */
    int timeoutCancel(Long id);

    /**
     * 超时未下单 订单状态13(待下单)->14(超时未下单)
     *
     * @param id 订单id
     * @return 执行结果
     */
    int timeoutNotPlaceOrder(Long id);

    /**
     * 确认订单
     *
     * @param id 订单id
     * @return 执行结果
     */
    int confirmOrder(Long id);

    /**
     * 更新自动确认收货时间
     * @param id 订单id
     * @param autoReceiveTime 自动确认收货时间
     * @return 执行结果
     */
    int updateAutoReceiveTime(Long id, Date autoReceiveTime);

    /**
     * 延迟自动确认收货时间（访客端用）
     * 和{@link #updateAutoReceiveTime}的区别是是否更新延迟标记
     *
     * @param id 订单id
     * @param autoReceiveTime 自动确认收货时间
     * @return 执行结果
     */
    int delayAutoReceiveTime(Long id, Date autoReceiveTime);

    /**
     * 下单结果更新
     *
     * @param id 主键id
     * @param orderStatus 订单状态
     * @param prepayId 预付单号
     * @return 执行结果
     */
    int updateCreateOrderResult(Long id, Integer orderStatus, String prepayId);

    /**
     * 支付结果更新
     * @param id 主键id
     * @param orderStatus 订单状态
     * @param paidTime 支付时间
     * @param transactionId 支付单号
     * @return 执行结果
     */
    int updatePayResult(Long id, Integer orderStatus, Date paidTime, String transactionId);

    /**
     * 支付结果更新
     *
     * @param param 支付结果
     * @return 执行结果
     */
    int updatePayResultByParam(MallPayResultParam param);

    /**
     * 下单结果更新
     *
     * @param param 结果更新参数
     * @return 执行结果
     */
    int updateCreateOrderResultByParam(MallCreateOrderResultParam param);

    /**
     * 进入售后状态
     *
     * @param id 订单Id
     * @return 执行结果
     */
    @Deprecated
    int updateAfterSale(Long id);

    /**
     * 根据订单id查询商品信息
     *
     * @param orderId 订单id
     * @return 商品信息
     */
    MallOrderGoodsSnapDto findGoodsInfoByOrderId(Long orderId);

    /**
     * 根据订单id修改收货地址
     *
     * @param orderId orderId
     * @param dto 收货地址
     * @return 执行结果
     */
    int updateAddressInfo(Long orderId, MallOrderAddressInfoDto dto);

    /**
     * 统计订单各个状态的数量
     *
     * @param liveUserId 直播用户id
     * @return 订单状态数统计
     */
    List<MallOrderStatusNumDto> statOrderStatusNum(Long liveUserId);

    /**
     * 查询用户指定业务类型的订单数量(spu维度去重)
     *
     * @param bizId  业务id
     * @param bizType  业务类型
     * @param liveUserId 用户id
     * @return 订单数量
     */
    Long countByBiz(Long bizId, Integer bizType, Long liveUserId);

    /**
     * 查询用户指定业务类型的订单数量(spu维度去重)
     *
     * @param bizId  业务id
     * @param bizType  业务类型
     * @param liveUserIds 用户id
     * @return k:userId v:orderNumList
     */
    Map<Long, List<Long>> countByBizMap(Long bizId, Integer bizType, List<Long> liveUserIds);

    /**
     * 查询用户指定业务类型的订单数量
     *
     * @param bizId  业务id
     * @param bizType  业务类型
     * @param liveUserId 用户id
     * @return 订单列表
     */
    List<MallOrderDto> listByBiz(Long bizId, Integer bizType, Long liveUserId);

    /**
     * 更新审核状态
     * @param auditState {@link cn.com.duiba.live.clue.service.api.enums.mall.order.MallOrderAuditStateEnum }
     */
    int updateAuditState(Long id, Integer auditState, String reason);

    int shipping(MallOrderUpdateShippingParam param);

    int updateShipping(MallOrderUpdateShippingParam param);

    Map<Long,Boolean> batchShipping(List<MallOrderUpdateShippingParam> params);

    /**
     * 虚拟商品批量发货
     * @param params 商品发货信息
     * @return 订单-发货结果
     */
    Map<Long,Boolean> batchShippingVirtual(List<VirtualShippingParam> params);

    /**
     * 更新虚拟商品发货信息
     */
    int updateVirtualShippingInfo(VirtualShippingParam param);

    /**
     * 批量审核
     */
    Map<Long,Boolean> batchUpdateAuditState(List<MallOrderAuditParam> params);

    /**
     * 更新物流信息
     *
     * @param orderId 订单id
     * @param expressState 物流状态
     * @param expressInfo 物流明细
     * @return 更新结果
     */
    boolean updateExpressInfo(Long orderId, Integer expressState, List<MallOrderExpressInfoDto> expressInfo);

    /**
     * 查询库存回退标识
     *
     * @param orderId 订单id
     * @return 0-未回退 1-已回退
     */
    Integer selectStockRollbackByOrderId(Long orderId);

    /**
     * 更新库存回退状态
     *
     * @param orderId 订单id
     * @return 更新结果
     */
    Integer updateStockRollbackMark(Long orderId);

    /**
     * 根据订单id查询物流信息
     *
     * @param orderId 订单id
     * @return 物流信息
     */
    List<MallOrderExpressInfoDto> selectExpressInfoByOrderId(Long orderId);

    /**
     * 刷数据脚本全部数据
     */
    @Deprecated
    void esScript();
}
