package cn.com.duiba.live.normal.service.api.remoteservice.follow;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.live.normal.service.api.dto.follow.AgentFollowChanceDto;
import cn.com.duiba.live.normal.service.api.param.follow.AgentChanceListParam;

import java.util.List;

/**
 * 代理人跟进建议接口
 * @author 吴文哲
 * @date 2021/7/25 10:25 下午
 **/
@AdvancedFeignClient
public interface RemoteAgentFollowChanceApiService {

    /**
     * 根据主键查询
     * @param id 主键
     * @return 代理人跟进机会
     */
    AgentFollowChanceDto selectById(Long id);


    /**
     * 根据条件查出一条跟进建议,如果找到多条,返回最近的一条
     *
     * @param liveId        直播间id
     * @param liveVisitorId 访客id
     * @param chanceType    跟进机会类型
     * @param bizConfId     业务配置id
     * @return 代理人跟进建议
     * @see cn.com.duiba.live.normal.service.api.enums.follow.FollowChanceTypeEnum
     */
    AgentFollowChanceDto selectOne(Long liveId, Long liveVisitorId, Integer chanceType, Long bizConfId);

    /**
     * 根据条件批量查询跟进建议列表
     *
     * @param liveId         直播间id
     * @param liveVisitorIds 访客用户id列表
     * @param chanceType     跟进机会类型
     * @return 代理人跟进机会列表
     */
    List<AgentFollowChanceDto> selectByLiveIdAndLiveVisitorIdsAndType(Long liveId, List<Long> liveVisitorIds, Integer chanceType);

    /**
     * 新增代理人跟进建议记录
     *
     * @param dto 跟进建议记录
     * @return 返回主键
     */
    Long saveAgentFollowChance(AgentFollowChanceDto dto);

    /**
     * 更新同意标记
     *
     * @param id        主键
     * @param agreeFlag 是否同意标记
     * @return 影响行数
     * @see cn.com.duiba.live.normal.service.api.enums.BooleanEnum
     */
    int updateAgreeFlag(Long id, Integer agreeFlag);

    /**
     * 批量更新同意标记
     * @param updateList 更新列表
     * @return 影响行数
     */
    int batchUpdateAgreeFlag(List<AgentFollowChanceDto> updateList);

    /**
     * 批量保存跟进建议
     *
     * @param followList 跟进建议列表
     * @return 主键id列表
     */
    List<Long> batchSaveFollowChance(List<AgentFollowChanceDto> followList);

    /**
     * 根据访客id、直播id查询跟进机会
     *
     * @param liveVisitorId  访客id
     * @param liveId 直播id
     * @param chanceType 跟进机会类型
     * @return 跟进机会
     */
    AgentFollowChanceDto findByLiveAndVisitorId(Long liveVisitorId, Long liveId, Integer chanceType);

    /**
     * 分页查询代理人+访客id查询所有跟进机会
     * 按跟进机会获取时间倒序排列
     *
     * @param param 分页查询条件
     * @return 跟进机会列表
     */
    List<AgentFollowChanceDto> findListByAgentAndVisitorId(AgentChanceListParam param);

    /**
     * 分页查询直播访客跟进机会列表
     *
     * @param liveId  直播id
     * @param liveVisitorId 直播访客id
     * @param offset  偏移量
     * @param pageSize 分页大小
     * @return 跟进机会列表
     */
    List<AgentFollowChanceDto> findListByLiveAndVisitorId(Long liveId, Long liveVisitorId, int offset, int pageSize);

    /**
     * 批量查询某场直播中客户是否存在跟进机会
     *
     * @param liveId  直播id
     * @param liveVisitorIds 直播访客id集合
     * @return 存在跟进机会的访客
     */
    List<Long> batchExistFollowChance(Long liveId, List<Long> liveVisitorIds);

    /**
     * 某场直播中客户是否存在跟进机会
     *
     * @param liveId  直播id
     * @param liveVisitorId 直播访客id
     * @return true=是，false=否
     */
    boolean existFollowChance(Long liveId, Long liveVisitorId);
}
