package cn.com.duiba.live.normal.service.api.remoteservice.oto.cust;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.live.normal.service.api.dto.common.PhoneLocationDto;
import cn.com.duiba.live.normal.service.api.dto.oto.cust.OtoCustomerDto;
import cn.com.duiba.live.normal.service.api.dto.oto.cust.OtoCustomerPageDto;
import cn.com.duiba.live.normal.service.api.param.oto.cust.OtoCustPageListParam;
import cn.com.duiba.live.normal.service.api.param.oto.cust.OtoCustomerEditParam;
import cn.com.duiba.live.normal.service.api.param.oto.cust.OtoCustomerSearchParam;
import cn.com.duiba.live.normal.service.api.util.PageResult;

import java.util.List;

/**
 * @author xiongkai
 * @version 1.0
 * @date 2022-01-10 13:44
 * 客户 rpc service
 */
@AdvancedFeignClient
public interface RemoteOtoCustService {

    /**
     * 根据客户id查询客户信息
     * @param custId 客户id
     * @return 客户信息
     */
    OtoCustomerDto findById(Long custId);

    /**
     * 根据客户id批量查询客户信息
     * @param custIds 客户id
     * @return 客户信息
     */
    List<OtoCustomerDto> listByIds(List<Long> custIds);

    /**
     * 根据客户id批量查询客户信息（过滤掉当前代理人未跟进的）
     * @param custIds 客户id
     * @param otoSid 销售id
     * @return 客户信息
     */
    List<OtoCustomerDto> listByIdsAndSid(List<Long> custIds, Long otoSid);

    /**
     * 模糊搜索客户名
     * @param custName 客户名
     * @param otoSid 销售id
     * @return 客户名列表
     */
    List<String> fuzzyCustName(String custName, Long otoSid);

    /**
     * 根据客户搜索条件获取客户列表
     * @param param 客户搜索条件
     * @return 客户列表
     */
    PageResult<OtoCustomerPageDto> searchByParam(OtoCustomerSearchParam param);

    /**
     * 给客户打星标/取消星标
     * @param custId 客户id
     * @param otoSid 销售id
     * @return 操作结果
     */
    boolean starCust(Long custId, Long otoSid);

    /**
     * 修改客户价值
     * @param custId 客户id
     * @param otoSid 销售id
     * @param custValue 客户价值
     * @see cn.com.duiba.live.normal.service.api.enums.oto.cust.ext.OtoCustValueEnum
     * @return 标记结果
     */
    boolean assessCust(Long custId, Long otoSid, Integer custValue);

    /**
     * 批量修改客户价值
     *
     * @param custIds  客户id集合
     * @param custValue 客户价值
     * @return 修改结果
     */
    boolean batchAssessCust(List<Long> custIds, Integer custValue);

    /**
     * 给客户打标签
     * @param custId 客户id
     * @param otoSid 销售id
     * @param tagIds 标签id数组
     * @return 客户打标签
     */
    boolean tagCust(Long custId, Long otoSid, List<Long> tagIds);

    /**
     * 编辑客户详情资料页
     * @param param 表单参数
     * @param otoSid 销售id
     * @return 客户详情资料页
     */
    boolean editCust(OtoCustomerEditParam param, Long otoSid);

    /**
     * 新增客户
     *
     * @param dto 客户信息
     * @return 新增结果
     */
    boolean addCust(OtoCustomerDto dto);

    /**
     * 更新客户信息
     *
     * @param dto 客户信息
     * @return 更新结果
     */
    boolean updateCust(OtoCustomerDto dto);

    /**
     * 根据手机号查询客户
     *
     * @param phoneNum 手机号
     * @return 客户信息
     */
    OtoCustomerDto findByPhone(String phoneNum);

    /**
     * 客户管理-客户分页列表
     *
     * @param param 查询参数
     * @return 分页查询结果
     */
    PageResult<OtoCustomerDto> custPageList(OtoCustPageListParam param);

    /**
     * 瀑布流查询，按照id倒序
     *
     * @param lastMaxId 上次查询的最大id
     * @param pageSize 查询数量
     * @return 查询结果
     */
    List<OtoCustomerDto> findWithWaterFall(Long lastMaxId, Long stopId, Integer pageSize);

    /**
     * 根据主键批量更新归属地
     *
     * @param locationDtos 归属地bean
     * @return 更新成功数量
     */
    int batchUpdateLocation(List<PhoneLocationDto> locationDtos);
}
