package cn.com.duiba.live.normal.service.api.enums.oto.cust;

import cn.com.duiba.live.normal.service.api.util.NumberUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.Getter;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 客户所在资源池枚举
 *
 * 资源池流转过程：
 *
 *                                                                           <------销售认领---------｜
 *                                                                           ｜                     ｜
 * 客户填写表单 -----> 「预备池」 ---定时任务待认领--> 「投放池」 ---销售认领--> 「跟进池」---系统回收----> 「回收池」
 *                                                                                    ｜
*                                                                                    ｜
*                                                                                    -------> 「垃圾池」
* @author ZhangJun
* @date 2022/2/14 16:27
*/
@Getter
@AllArgsConstructor
public enum OtoCustPoolEnum {

    WAIT_POOL(1, 0, "预备池"),
    FRESH_POOL(2, 0, "投放池"),
    FOLLOW_POOL(3, 1, "跟进池"),
    RECYCLE_POOL(4, 2, "回收池"),
    GARBAGE_POOL(5, 3, "垃圾池")
    ;

    /**
     * 客户资源池类型
     */
    private Integer poolType;

    /**
     * 客户状态
     * {@link OtoCustStatusEnum}
     */
    private Integer custStatus;

    /**
     * 资源池描述
     */
    private String desc;

    /**
     * 资源池枚举map
     */
    private static final Map<Integer, OtoCustPoolEnum> ENUM_MAP = Collections.unmodifiableMap(Arrays.stream(OtoCustPoolEnum.values()).collect(Collectors.toMap(OtoCustPoolEnum::getPoolType, Function.identity())));

    /**
     * 可进行客户认领操作的资源池
     */
    private static final List<OtoCustPoolEnum> CLAIM_POOLS = Collections.unmodifiableList(Lists.newArrayList(FRESH_POOL, RECYCLE_POOL));


    /**
     * 可重新激活的资源池
     */
    private static final List<OtoCustPoolEnum> ACTIVATE_POOLS = Collections.unmodifiableList(Lists.newArrayList(RECYCLE_POOL, GARBAGE_POOL));

    /**
     * 客户所处资源池与客户状态的映射关系
     * {@link OtoCustStatusEnum}
     */
    private static final Map<Integer, List<Integer>> POOL_STATUS_MAP = Maps.newHashMap();

    static {
        POOL_STATUS_MAP.put(OtoCustStatusEnum.WAIT_ASSIGN.getCustStatus(), Lists.newArrayList(WAIT_POOL.poolType, FRESH_POOL.getPoolType()));
        POOL_STATUS_MAP.put(OtoCustStatusEnum.ASSIGNED.getCustStatus(), Lists.newArrayList(FOLLOW_POOL.poolType));
        POOL_STATUS_MAP.put(OtoCustStatusEnum.RECYCLED.getCustStatus(), Lists.newArrayList(RECYCLE_POOL.poolType));
        POOL_STATUS_MAP.put(OtoCustStatusEnum.GARBAGE_POOL.getCustStatus(), Lists.newArrayList(GARBAGE_POOL.getPoolType()));
    }

    /**
     * 可进行客户认领的资源池类型
     */
    public static List<Integer> claimPools() {
        return CLAIM_POOLS.stream().map(OtoCustPoolEnum::getPoolType).collect(Collectors.toList());
    }

    /**
     * 判断客户所处的资源池是否可认领
     */
    public static boolean canClaim(Integer poolType) {
        if (NumberUtil.isNullOrLteZero(poolType)) {
            return false;
        }
        return claimPools().contains(poolType);
    }

    public static OtoCustPoolEnum getByPoolType(Integer poolType) {
        if (Objects.isNull(poolType)) {
            return null;
        }
        return ENUM_MAP.get(poolType);
    }

    /**
     * 根据客户状态转换对应的资源池类型
     */
    public static List<Integer> convertCustStatus2PoolTypes(Integer custStatus) {
        return Collections.unmodifiableList(Arrays.stream(OtoCustPoolEnum.values()).filter(item -> Objects.equals(item.getCustStatus(), custStatus)).map(OtoCustPoolEnum::getPoolType).collect(Collectors.toList()));
    }


    /**
     * 判断客户是否可重新激活
     */
    public static boolean canActivate(Integer poolType) {
        if (NumberUtil.isNullOrLteZero(poolType)) {
            return false;
        }
        return ACTIVATE_POOLS.stream().anyMatch(otoCustPoolEnum -> otoCustPoolEnum.getPoolType().equals(poolType));
    }
}
