package cn.com.duiba.live.statistics.service.api.enums.customer;

import cn.com.duiba.live.statistics.service.api.util.NumberUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 代理人客户基础信息扩展infoName枚举
 * <a href="http://cf.dui88.com/pages/viewpage.action?pageId=111219003">参考资料</a>
 *
 * @author ZhangJun
 * @date 2021/6/30 4:04 下午
 */
@Slf4j
@Getter
@AllArgsConstructor
public enum AgentCustomerExtEnum {

    /**
     * 性别，null/0:未知，1-男，2-女
     *
     * @see SexEnum
     */
    SEX(1, "性别"),

    /**
     * 年龄，null：未知，0：0～10岁，1：11～20岁，以此类推
     *
     * @see AgeEnum
     */
    AGE(2, "年龄"),

    /**
     * 职业，直接取值
     */
    JOB(3, "职业"),

    /**
     * 个人年收入，null：未知，0：0～10万，1：1～10万，以此类推
     *
     * @see IncomeEnum
     */
    INCOME(4, "收入"),

    /**
     * 所在地址，直接取值
     */
    LOCATION(5, "地址"),

    /**
     * 爱好，直接取值，英文逗号分隔
     */
    HOBBY(6, "爱好"),

    /**
     * 保险意识强弱，null：未知，0：弱，1：较弱，2：中，3：较强，4：强
     *
     * @see InsuranceAwarenessEnum
     */
    INSURANCE_AWARENESS(7, "保险意识"),

    /**
     * 持保情况，null：未知，0：未持保，1：已持保
     *
     * @see HasInsuranceEnum
     */
    HAS_INSURANCE(8, "持保情况"),

    /**
     * 婚姻状况，null：未知，0：未婚，1：已婚
     *
     * @see MarryEnum
     */
    MARRIAGE_STATUS(9, "婚姻状况"),

    /**
     * 是否有子女，null：未知，0：无，1：有
     *
     * @see HasChildrenEnum
     */
    HAS_CHILDREN(10, "是否有子女"),

    /**
     * 家庭成员人数，直接取值
     */
    FAMILY_MEMBER(11, "家庭成员人数"),

    /**
     * 家庭年收入，null：未知，0：0～10万，1：1～10万，以此类推
     *
     * @see IncomeEnum
     */
    FAMILY_INCOME(12, "家庭年收入"),
    ;


    /**
     * 扩展字段值
     */
    private final Integer infoName;

    /**
     * 扩展字段含义
     */
    private final String desc;

    private static final Map<Integer, AgentCustomerExtEnum> ENUM_MAP = Maps.newHashMap();

    private static final List<Integer> INFO_NAME_LIST = Lists.newArrayList();

    /**
     * 不需要校验枚举值的类型
     */
    private static final List<Integer> NON_ENUM_TYPE_VALUE = Lists.newArrayList();

    static {
        for (AgentCustomerExtEnum item : values()) {
            ENUM_MAP.put(item.getInfoName(), item);
            INFO_NAME_LIST.add(item.infoName);
        }

        NON_ENUM_TYPE_VALUE.add(JOB.infoName);
        NON_ENUM_TYPE_VALUE.add(LOCATION.infoName);
        NON_ENUM_TYPE_VALUE.add(HOBBY.infoName);
    }

    /**
     * 是否存在该枚举字段
     */
    public static boolean existEnum(Integer infoName) {
        if (NumberUtil.isNullOrLteZero(infoName)) {
            return false;
        }
        AgentCustomerExtEnum item = ENUM_MAP.get(infoName);
        return Objects.nonNull(item);
    }

    /**
     * 获取枚举的所有infoName
     */
    public static List<Integer> getInfoNameList() {
        return Collections.unmodifiableList(INFO_NAME_LIST);
    }

    /**
     * 检查枚举值是否合法
     */
    public static boolean checkExtValue(Integer infoName, String infoValue) {
        AgentCustomerExtEnum enumType = ENUM_MAP.get(infoName);
        if (Objects.isNull(enumType)) {
            log.warn("info name is illegal, not exist this enum type, infoName={}, infoValue={}", infoName, infoValue);
            return false;
        }

        // 允许空值
        if (StringUtils.isBlank(infoValue)) {
            return true;
        }
        // 不校验非枚举类型的值
        if (NON_ENUM_TYPE_VALUE.contains(infoName)) {
            return true;
        }

        int value;
        try {
            value = Integer.parseInt(infoValue);
        }catch (Exception e) {
            log.warn("infoValue is illegal, infoValue should be int, infoValue={}", infoValue);
            return false;
        }

        boolean flag;
        switch (enumType) {
            case SEX:
                flag = SexEnum.existEnum(value);
                break;
            case AGE:
                flag = AgeEnum.existEnum(value);
                break;
            case INCOME:
                flag = IncomeEnum.existEnum(value);
                break;
            case INSURANCE_AWARENESS:
                flag = InsuranceAwarenessEnum.existEnum(value);
                break;
            case HAS_INSURANCE:
                flag = HasInsuranceEnum.existEnum(value);
                break;
            default:
                flag = false;
        }

        return flag;
    }

}
