package cn.com.duiba.live.statistics.service.api.remoteservice.agent;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.live.statistics.service.api.dto.agent.AgentPresentStaticsBean;
import cn.com.duiba.live.statistics.service.api.dto.agent.AgentStaticsBean;
import cn.com.duiba.live.statistics.service.api.dto.agent.LiveStatisticAgentBean;
import cn.com.duiba.live.statistics.service.api.dto.agent.LiveStatisticsAgentDto;
import cn.com.duiba.live.statistics.service.api.dto.agent.LiveStatisticsAgentInviteDto;
import cn.com.duiba.live.statistics.service.api.dto.agent.LiveStatisticsAgentSimpleDto;
import cn.com.duiba.live.statistics.service.api.dto.live.LiveBean;
import cn.com.duiba.live.statistics.service.api.param.agent.LiveStatisticsAgentInviteParam;
import cn.com.duiba.live.statistics.service.api.param.agent.LiveStatisticsAgentReadNumParam;
import cn.com.duiba.live.statistics.service.api.param.agent.LiveStatisticsAgentSearchParam;
import org.apache.commons.lang3.tuple.Pair;

import java.util.List;
import java.util.Set;

/**
 * @author 吴文哲
 * @date 2021/1/4 4:17 下午
 **/
@AdvancedFeignClient
public interface RemoteLiveStatisticsAgentApiService {

    /**
     * 根据直播间id 代理人id 获取统计数据
     *
     * @param liveId
     * @param agentId
     * @return
     */
    LiveStatisticsAgentDto getByLiveIdAndAgentId(Long liveId, Long agentId);


    /**
     * 根据条件获取对应直播间 代理人纬度统计数据
     * @param param
     * @return
     */
    List<LiveStatisticsAgentDto> getListByCondition(LiveStatisticsAgentSearchParam param);

    /**
     * 获取代理人排名
     *      * 不同直播间状态 统计数据不同
     * @param liveId
     * @param agentId
     * @return
     */
    Integer getAgentRanking(Long liveId, Long agentId);

    /**
     * 获取代理人队内排名
     * 不同直播间状态 统计数据不同
     *
     * @param liveId 直播id
     * @param agentId 代理人id
     * @return 队内排名
     */
    Integer getAgentTeamRanking(Long liveId, Long agentId);

    /**
     * 根据条件统计
     * @param liveId 直播间id
     * @param companyId 公司id
     * @return left:邀请数量 right:到场数量
     */
    Pair<Integer, Integer> getCompanyInviteAndPresentSum(Long liveId, Long companyId);

    /**
     * 根据直播IDs 和 代理人ID查询
     * @param agentId 代理人ID
     * @param liveIds 直播IDs
     * @return 代理人统计数据
     */
    List<LiveStatisticsAgentSimpleDto> selectByAgentIdAndLiveIds(Long agentId, List<Long> liveIds);


    /**
     * 根据直播ID 和 代理人IDs查询
     * @param liveId 直播ID
     * @param agentIds 代理ids
     * @return 代理人统计数据
     */
    List<LiveStatisticsAgentSimpleDto> selectByLiveIdAndAgentIds(Long liveId, List<Long> agentIds);



    /**
     * 根据直播ID 和 代理人ID查询
     * @param agentId 代理人ID
     * @param liveId 直播ID
     * @return 代理人统计数据
     */
    LiveStatisticsAgentSimpleDto selectByAgentIdAndLiveId(Long agentId, Long liveId);

    /**
     * 获取代理人最近一场直播线索
     *
     * @param agentId 代理人id
     * @param notOverLiveIds 尚未结束的直播间id列表
     * @return 最近一场直播线索
     */
    LiveStatisticsAgentDto findAgentLastLiveClue(Long agentId,List<Long> notOverLiveIds);


    /***----- livecenter 用到的接口-------***/

    /**
     * 根据条件查询数量
     * @param param 查询条件
     * @return 符合条件的数量
     */
    Long selectCount(LiveStatisticsAgentSearchParam param);

    /**
     * 插入新的统计记录
     * @param entity 记录
     * @return 新增数量
     */
    int insert(LiveStatisticsAgentDto entity);

    /**
     * 批量初始化代理人 在多个直播间的统计数据
     * @param companyId 公司ID
     * @param teamId 团队ID
     * @param agentId 代理人ID
     * @param liveIds 直播列表
     * @return 新增数量
     */
    int batchInsertOneAgentMoreLives(Long companyId, Long teamId, Long agentId, List<Long> liveIds);


    /**
     * 根据团队ID删除统计数据（解散团队）
     * @param teamId 团队ID
     * @param liveIds 未结束的直播列表。
     * @return 删除数量
     */
    int deleteByTeamId(Long teamId,List<Long> liveIds);

    /**
     * 根据代理人ID删除统计数据（个人退出团队）
     * @param agentId 代理人ID
     * @param liveIds 未结束的直播列表。
     * @return 删除数量
     */
    int deleteByAgentId(Long agentId, List<Long> liveIds);

    /**
     * 根据代理人id和直播id批量删除统计数据（批量退出公司）
     *
     * @param agentIds 代理人id集合
     * @param liveIds 直播公司id集合
     * @return 删除数量
     */
    int deleteByAgentIdsAndLiveIds(List<Long> agentIds, List<Long> liveIds);

    /**
     * 原子性增加某场直播某个代理人的邀请数量
     * @param liveId 直播ID
     * @param agentId 代理人id
     * @return >1 成功 =0 记录不存在
     */
    int atomicIncreaseInvitationNum(Long liveId, Long agentId);

    /**
     * 原子性增加某场直播某个代理人邀请人的到场人数
     * @param liveId 直播ID
     * @param agentId 代理人id
     * @return
     */
    int atomicIncreasePresentNum(Long liveId, Long  agentId);


    /**
     * 根据直播ID 和 代理人IDs 意义：排除掉已经存在的统计记录
     * @param liveId 直播ID
     * @param agentIds 代理ids
     * @return 已经存在直播IDs
     */
    Set<Long> selectAgentIdsByLiveIdAndAgentIds(Long liveId,Set<Long> agentIds);

    /**
     * 批量更新
     * @param beans
     * @return
     */
    Integer batchUpdateRanking(List<LiveStatisticAgentBean> beans);

    /**
     * 根据直播间id 代理人Id
     * 查询统计信息
     * @param liveId 直播间id
     * @param agentId 代理人id
     * @return
     */
    LiveStatisticsAgentDto selectByLiveIdAndAgentId( Long liveId,  Long agentId);

    /**
     * 增量更新统计数据
     * @param agentStaticsBeanList
     * @return 影响行数
     * 改造成队列削峰后，不要依赖返回的影响行数做业务处理。
     */
    int batchUpdate(List<AgentStaticsBean> agentStaticsBeanList,  Long liveId);


    /**
     * 根据条件查询，不分页
     * @param param 查询条件
     * @return 直播统计-代理人维度统计信息集合
     */
    List<LiveStatisticsAgentDto> selectList(LiveStatisticsAgentSearchParam param);


    /**
     * 全量同步代理人到场数
     * @param liveId
     * @param list
     * @return
     */
    int batchUpdatePresentNum(Long liveId, List<AgentPresentStaticsBean> list);



    /**
     * 确定代理人统计信息排名
     * @param liveEntity 直播对象
     * @param rank 是否进行排名
     * @return
     */
    Integer statisticAgent(LiveBean liveEntity, boolean rank);

    /**
     * 根据直播ID查询所有的代理人统计数据
     * @param liveId
     * @return
     */
    List<LiveStatisticsAgentDto> findByLiveId(Long liveId);

    /**
     * 根据公司id瀑布流查询代理人统计数据
     *
     * @param liveId 直播id
     * @param startId 起始id
     * @param pageSize 页码
     * @return 统计数据
     */
    List<LiveStatisticsAgentDto> findByLiveIdAndStartId(Long liveId, Long startId, Integer pageSize);

    /**
     * 查询代理人参与过的有线索的直播
     * @param agentId
     * @return
     */
    List<LiveStatisticsAgentDto> findAgentLiveRecord(Long agentId);



    /**
     * 读取线索数量+1
     *
     * @param liveId 直播id
     * @param agentId 代理人id
     * @return 执行结果
     */
    int increaseReadNum(Long liveId, Long agentId);

    /**
     * 更新读取线索数
     * @param param 线索数
     * @return 更新结果
     */
    int updateReadNum(LiveStatisticsAgentReadNumParam param);

    /**
     * 批量更新读取线索数
     * @param params 线索数
     * @return 更新结果
     */
    int batchUpdateReadNum(List<LiveStatisticsAgentReadNumParam> params);

    /**
     * 根据直播IDs 和 代理人ID查询邀约数据
     * @param agentId 代理人ID
     * @param liveIds 直播IDs
     * @return 代理人邀约数据
     */
    List<LiveStatisticsAgentInviteDto> selectInviteData(Long agentId, List<Long> liveIds);

    /**
     * 批量更新代理人数据
     * @param params 邀请数据
     * @return 操作成功数
     */
    int batchUpdateInviteData(List<LiveStatisticsAgentInviteParam> params);
}
