package cn.com.duiba.local.autoconfigure.data.redis;

import com.alibaba.fastjson.support.spring.GenericFastJsonRedisSerializer;
import org.redisson.RedissonMultiLock;
import org.redisson.RedissonRedLock;
import org.redisson.api.RedissonClient;
import org.redisson.spring.starter.RedissonAutoConfiguration;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;

import java.util.concurrent.locks.ReadWriteLock;

/**
 * redis自动装配<br>
 * 仅支持lettuce连接池
 * <pre>
 * 分布式锁请务必使用Redisson，一般场景都用可重入锁{@link RedissonClient#getLock(java.lang.String)}，其他场景还包括
 * 公平锁{@link RedissonClient#getFairLock(java.lang.String)}、红锁{@link RedissonRedLock}、联锁{@link RedissonMultiLock}、
 * 读写锁{@link ReadWriteLock}等，请根据特定场景使用不同的锁
 * 分布式锁e.g.
 * RLock lock = redissonClient.getLock("your lock name");
 * try {
 *     if (lock.tryLock(0L, 60L, TimeUnit.SECONDS)) {
 *         // Do Business Something
 *     } else {
 *         // Get Lock Fail
 *     }
 * } catch (InterruptedException e) {
 *     // Lock Exception
 * } finally {
 *     // ⚠️important 释放锁之前必须加上这个判断，保证是加锁的线程去释放锁
 *     if (lock.isHeldByCurrentThread()) {
 *         lock.unlock();
 *     }
 * }
 * </pre>
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.1
 **/
@Configuration(proxyBeanMethods = false)
@AutoConfigureBefore({org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration.class,
        RedissonAutoConfiguration.class})
@ConditionalOnClass({RedisOperations.class})
@EnableConfigurationProperties(RedisProperties.class)
@Import({LettuceConnectionConfiguration.class})
public class RedisAutoConfiguration {
    @Bean(name = "redisTemplate")
    @ConditionalOnMissingBean(name = "redisTemplate")
    public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<Object, Object> template = new RedisTemplate<>();
        template.setConnectionFactory(redisConnectionFactory);
        // 使用fastJson序列化
        template.setDefaultSerializer(new GenericFastJsonRedisSerializer());
        return template;
    }

    @Bean(name = "stringRedisTemplate")
    @ConditionalOnMissingBean(name = "stringRedisTemplate")
    public StringRedisTemplate stringRedisTemplate(RedisConnectionFactory redisConnectionFactory) {
        StringRedisTemplate template = new StringRedisTemplate();
        template.setConnectionFactory(redisConnectionFactory);
        return template;
    }

    @Bean
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    public Hessian2RedisDeserializeMvcEndpoint hessian2RedisDeserializeMvcEndpoint() {

        return new Hessian2RedisDeserializeMvcEndpoint();
    }
}