package cn.com.duiba.local.autoconfigure.mq.rocket;

import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.MessageListener;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly;
import org.apache.rocketmq.client.exception.MQClientException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.util.Assert;

import javax.annotation.Nonnull;

/**
 * Rocket MQ 消费者事件监听
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.1
 **/
@Slf4j
@ConditionalOnProperty(prefix = RocketMqProperties.ROCKET_PREFIX, name = "consumer.enable", havingValue = "true")
public class RocketMqConsumerListener implements ApplicationListener<ContextRefreshedEvent> {
    public static final String ROCKET_MQ_LISTENER_BEAN_NAME = "duibaRocketMqMessageListener";

    private boolean started = false;

    @Override
    public synchronized void onApplicationEvent(@Nonnull ContextRefreshedEvent event) {
        if (started) {
            log.warn("rocketmq consumer have already started, please don't call start again");
            return;
        }
        ApplicationContext applicationContext = event.getApplicationContext();
        DefaultMQPushConsumer consumer = (DefaultMQPushConsumer) applicationContext.getBean("rocketMqConsumer");
        MessageListener rocketMqMessageListener =
                (MessageListener) applicationContext.getBean(ROCKET_MQ_LISTENER_BEAN_NAME);
        Assert.notNull(rocketMqMessageListener, "bean id:[" + ROCKET_MQ_LISTENER_BEAN_NAME + "] class:[org.apache" +
                ".rocketmq.client.consumer.listener.MessageListenerConcurrently] is not exist in spring's context," +
                "请声明，否则不会启用rocketmq消费！");
        if (rocketMqMessageListener instanceof MessageListenerConcurrently) {
            consumer.registerMessageListener((MessageListenerConcurrently) rocketMqMessageListener);
        } else if (rocketMqMessageListener instanceof MessageListenerOrderly) {
            consumer.registerMessageListener((MessageListenerOrderly) rocketMqMessageListener);
        }
        try {
            consumer.start();
        } catch (MQClientException e) {
            throw new RuntimeException(e);
        }
        started = true;
    }
}