package cn.com.duiba.nezha.alg.alg.adx.rcmd;

import cn.com.duiba.nezha.alg.alg.base.MathBase;
import cn.com.duiba.nezha.alg.alg.base.Roulette;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxIdeaParamsDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxIdeaStatDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.AdxResourceRcmdDo;
import cn.com.duiba.nezha.alg.common.enums.DateStyle;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.LocalDateUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;

import java.util.*;

public class AdxIdeaRcmder {

    private static long ADX_MULTIPLIER = 10000000;

    /**
     * ctr权重分配
     */
    static double[] ctrBucket = {0.7, 0.8, 0.85, 0.9, 0.97, 1};
    static double[] ctrWeight = {1, 2, 5, 10, 20, 100};

    /**
     * pcpm权重分配1 top1>0
     */
    static double[] pCpmBucket = {0.7, 0.8, 0.85, 0.9, 0.97, 1};
    static double[] pCpmWeight = {1, 2, 5, 10, 20, 100};

    /**
     * pcpm权重分配1 top1<0
     */
    static double[] pCpmBucket2 = {1, 1.1, 1.5, 2};
    static double[] pCpmWeight2 = {100, 20, 5, 1};


    /**
     * 更新间隔权重分配
     */
    static double[] updateDelayBucket = {60, 60 * 6, 60 * 24, 60 * 24 * 3};
    static double[] updateDelayWeight = {0.3, 0.5, 0.7, 1.0};

    /**
     * 定时任务 流量分配 main1
     *
     * @param ideaList              资源下，算法组创意集合（开启）
     * @param adxMinStatDoMap       创意统计数据 20分钟
     * @param adxDayStatDoMap       创意统计数据 3天
     * @param lastAdxResourceRcmdDo 上一次输出对象（资源位创意推荐对象）
     * @return
     */
    public static AdxResourceRcmdDo trafficAllocation(List<Long> ideaList,
                                                      Map<Long, AdxIdeaStatDo> adxMinStatDoMap,
                                                      Map<Long, AdxIdeaStatDo> adxDayStatDoMap,
                                                      AdxResourceRcmdDo lastAdxResourceRcmdDo) {

        AdxResourceRcmdDo ret = lastAdxResourceRcmdDo;


        //
        if (AssertUtil.isEmpty(ideaList)) {
            return ret;
        }

        //0 初始化
        if (ret == null) {
            ret = new AdxResourceRcmdDo();
        }

        if (ret.getIdeaParamsMap() == null) {
            ret.setIdeaParamsMap(new HashMap<>());
        }


        //1 数据处理：置信、ctr等指标计算
        if (AssertUtil.isNotEmpty(adxMinStatDoMap)) {
            adxMinStatDoMap.forEach((ideaId, adxStatDo) -> ideaStatCompute(adxStatDo));
        }

        if (AssertUtil.isNotEmpty(adxDayStatDoMap)) {
            adxDayStatDoMap.forEach((ideaId, adxStatDo) -> ideaStatCompute(adxStatDo));
        }


        //2 最优指标筛选
        AdxIdeaStatDo bestAdxMinIdeaStatDo = getBestCtrAndProfitEcpm(ideaList, adxMinStatDoMap);
        AdxIdeaStatDo bestAdxDayIdeaStatDo = getBestCtrAndProfitEcpm(ideaList, adxDayStatDoMap);

        //3 退出更新：实时数据均不置信（暂停投放 或 数据量级小）
        if (bestAdxMinIdeaStatDo == null || bestAdxMinIdeaStatDo.getCtr() == null) {

            return ret;
        }

        //4 新权重分配 && 创意退出w=0（效果差）

        Map<Long, Double> newWeightMap = new HashMap<>();
        Set<Long> fusedWeightSet = new HashSet<>();

        Double oldWeightSum = 0.0;
        Double newWeightSum = 0.0;
        for (int i = 0, size = ideaList.size(); i < size; i++) {

            // 4.0 初始化
            Long ideaId = ideaList.get(i);
            AdxIdeaParamsDo adxIdeaParamsDo = ret.getIdeaParamsMap().get(ideaId);
            if (adxIdeaParamsDo == null) {
                adxIdeaParamsDo = new AdxIdeaParamsDo();
                adxIdeaParamsDo.setIdeaId(ideaId);
                ret.getIdeaParamsMap().put(ideaId, adxIdeaParamsDo);
            }
            // 数据准备
            AdxIdeaStatDo adxDayIdeaStatDo = adxDayStatDoMap.get(ideaId);
            AdxIdeaStatDo adxMinIdeaStatDo = adxMinStatDoMap.get(ideaId);

            // 4.1 权重分配

            // 历史、实时权重分配
            // 融合权重-指标维度（CTR、pECPM）
            Double dayWeight = getIdeaWeight(adxDayIdeaStatDo, bestAdxDayIdeaStatDo);
            Double minWeight = getIdeaWeight(adxMinIdeaStatDo, bestAdxMinIdeaStatDo);

            // 4.2 权重融合-时间维度
            Double currentMergeRatio = 0.8;
            Double newWeight = getMergeWeight(minWeight, dayWeight, adxIdeaParamsDo.getCurrentTimes(), size + 0L, currentMergeRatio);

            newWeightMap.put(ideaId, newWeight);

            // 累计权重
            newWeightSum += newWeight;
            oldWeightSum += DataUtil.getValueOrDefault(adxIdeaParamsDo.getWeight(), 0.0);

            // 4.3 熔断判断
            if (dayWeight != null && dayWeight <= 1 && adxDayIdeaStatDo.getExp() > 5000) {
                // 加入熔断集合，数据不为空&&权重过低&&超过5000次曝光
                fusedWeightSet.add(ideaId);
            }

        }

        //5 更新 && 归一化
        //公式 w=  lr * w_old+(1-lr) w_new

        for (int i = 0, size = ideaList.size(); i < size; i++) {
            Long ideaId = ideaList.get(i);
            AdxIdeaParamsDo adxIdeaParamsDo = ret.getIdeaParamsMap().get(ideaId);


            if (fusedWeightSet.contains(ideaId)) {
                // 熔断
                adxIdeaParamsDo.setWeight(0.0);

                adxIdeaParamsDo.setCurrentTimes(0L);

            } else {
                Double finalWeight = getUpdateWeight(adxIdeaParamsDo.getWeight(),
                        oldWeightSum,
                        newWeightMap.get(ideaId),
                        newWeightSum,
                        adxIdeaParamsDo.getLastUpdateTime());

                adxIdeaParamsDo.setWeight(finalWeight);

                adxIdeaParamsDo.setCurrentTimes(DataUtil.addLong(adxIdeaParamsDo.getCurrentTimes(), 1L));
                adxIdeaParamsDo.setLastUpdateTime(LocalDateUtil.getCurrentLocalDateTime(DateStyle.YYYYMMDDHHMMSS));
            }


//            System.out.println("ideaid="+ideaId+",oldWeight=" + adxIdeaParamsDo.getWeight()+",newWeight="+newWeightMap.get(ideaId)+",finalWeight="+adxIdeaParamsDo.getWeight());
        }


        return ret;
    }

    /**
     * 创意推荐 main2
     *
     * @param validIdeaList     有效创意集合
     * @param adxResourceRcmdDo 资源位创意推荐对象
     * @return 创意
     */
    public static Long ideaRcmd(List<Long> validIdeaList, AdxResourceRcmdDo adxResourceRcmdDo) {


        Long ret = null;

        if (AssertUtil.isEmpty(validIdeaList)) {
            return ret;
        }

        //1 数据处理
        if (adxResourceRcmdDo == null) {
            adxResourceRcmdDo = new AdxResourceRcmdDo();
            adxResourceRcmdDo.setIdeaParamsMap(new HashMap<>());
        }

        //2 轮盘赌，随机推荐创意
        Map<Long, Double> map = new HashMap<>();

        for (int i = 0, size = validIdeaList.size(); i < size; i++) {
            Long ideaId = validIdeaList.get(i);
            // 数据准备
            AdxIdeaParamsDo adxIdeaParamsDo = adxResourceRcmdDo.getIdeaParamsMap().get(ideaId);
            // 权重
            if (adxIdeaParamsDo == null || adxIdeaParamsDo.getWeight() == null) {
                // 默认权重
                map.put(ideaId, 1.0);
            } else {
                //
                map.put(ideaId, adxIdeaParamsDo.getWeight());
            }
        }

        ret = Roulette.doubleMap(map);

        //3 返回创意

        if (ret == null) {
            ret = validIdeaList.get(0);
        }
        return ret;
    }


    /**
     * 权重更新
     * 对长时间未更新特征，加大学习步长
     *
     * @param oldWeight
     * @param newWeight
     * @param lastUpdateTime
     * @return
     */
    public static Double getUpdateWeight(Double oldWeight, Double oldWeightSum, Double newWeight, Double newWeightSum, String lastUpdateTime) {

        Double ret = null;


        if (oldWeight != null) {
            // 有旧模型融合

            Long delayMinutes = LocalDateUtil.getIntervalMinutes(lastUpdateTime, DateStyle.YYYYMMDDHHMMSS);
            Double learnRatio = MathBase.getConfidenceWeight(delayMinutes, updateDelayBucket, updateDelayWeight, 0.3);

            Double newWeight2 = newWeight * 100 / newWeightSum;
            Double oldWeight2 = oldWeight * 100 / oldWeightSum;

            ret = (1 - learnRatio) * oldWeight2 + learnRatio * newWeight2;
        } else {
            //无旧模型，赋值
            ret = newWeight * 100 / newWeightSum;
        }


        return MathUtil.formatDouble(ret, 2);
    }


    /**
     * 权重融合
     *
     * @param minWeight
     * @param dayWeight
     * @return
     */
    public static Double getMergeWeight(Double minWeight, Double dayWeight, Long currentTimes, Long size, Double currentWeightRato) {
        Double ret = null;

        // 历史无数据 && 实时有数据，实时为准
        if (dayWeight == null && minWeight != null) {
            ret = minWeight;
        }

        // 历史有数据 && 实时无数据，历史位准
        if (dayWeight != null && minWeight == null) {
            ret = dayWeight;
        }

        // 历史无数据 && 实时无数据，默认
        if (dayWeight == null && minWeight == null) {
            if (currentTimes == null || currentTimes <= 5) {
                ret = 100.0 / size;
            } else {
                ret = 1.0;
            }

        }
        // 历史有数据 && 实时有数据，融合（时间维度）
        if (dayWeight != null && minWeight != null) {

            ret = (1 - currentWeightRato) * dayWeight + currentWeightRato * minWeight;
        }
//            System.out.println("dayWeight=" + dayWeight + "，minWeight=" + minWeight + ",mergeWeight=" + ret);
        return ret;
    }

    /**
     * 权重分配，根据CTR、PCPM
     *
     * @param adxIdeaStatDo
     * @param adxBestIdeaStatDo
     * @return
     */
    public static Double getIdeaWeight(AdxIdeaStatDo adxIdeaStatDo, AdxIdeaStatDo adxBestIdeaStatDo) {
        Double ret = null;
        // 数据不置信(创意、最优组)
        if (adxIdeaStatDo == null || !adxIdeaStatDo.getConfidence() || adxBestIdeaStatDo == null) {
            return ret;
        }

        Double bestCtr = adxBestIdeaStatDo.getCtr();
        Double bestProfitEcpm = adxBestIdeaStatDo.getProfitEcpm();

        Double ideaCtr = adxIdeaStatDo.getCtr();
        Double ideaProfitEcpm = adxIdeaStatDo.getProfitEcpm();

        boolean ideaConfidence = adxIdeaStatDo.getConfidence();

        Double ctrWeight = getCtrWeight(ideaCtr, bestCtr, ideaConfidence);
        Double profitCpmWeight = getProfitEcpmWeight(ideaProfitEcpm, bestProfitEcpm, ideaConfidence);


        if (ctrWeight != null && profitCpmWeight != null) {
            ret = 0.5 * ctrWeight + 0.5 * profitCpmWeight;
        }
//        System.out.println("ctrWeight="+ctrWeight+"，profitCpmWeight="+profitCpmWeight);
        return ret;
    }


    /**
     * @param ctr
     * @param bestCtr
     * @param isConfidence
     * @return
     */
    private static Double getCtrWeight(Double ctr, Double bestCtr, boolean isConfidence) {
        Double ret = null;
        if (bestCtr != null && isConfidence && ctr != null) {
            Double ratio = DataUtil.division(ctr, bestCtr, 3);
            ret = MathBase.getConfidenceWeight(Math.min(ratio, 1.0), ctrBucket, ctrWeight);
        }
        return ret;
    }

    /**
     * @param pCpm
     * @param bestPCpm
     * @param isConfidence
     * @return
     */
    private static Double getProfitEcpmWeight(Double pCpm, Double bestPCpm, boolean isConfidence) {
        Double ret = null;

        if (pCpm != null && isConfidence && bestPCpm != null) {

            if (bestPCpm >= 0) {
                Double ratio = DataUtil.division(pCpm, bestPCpm + 0.000001, 3);
                ret = MathBase.getConfidenceWeight(Math.min(ratio, 1.0), pCpmBucket, pCpmWeight);
            } else {
                Double ratio = DataUtil.division(pCpm, bestPCpm, 3);
                ret = MathBase.getConfidenceWeight(Math.max(ratio, 1.0), pCpmBucket2, pCpmWeight2);
            }
        }

        return ret;
    }

    /**
     * 计算最大CTR && PECPM
     *
     * @param adxStatDoMap
     * @return
     */
    private static AdxIdeaStatDo getBestCtrAndProfitEcpm(List<Long> ideaList, Map<Long, AdxIdeaStatDo> adxStatDoMap) {
        AdxIdeaStatDo ret = new AdxIdeaStatDo();
        if (AssertUtil.isEmpty(adxStatDoMap)) {
            return null;
        }

        for (int i = 0, size = ideaList.size(); i < size; i++) {
            Long ideaId = ideaList.get(i);
            AdxIdeaStatDo adxIdeaStatDo = adxStatDoMap.get(ideaId);
            if (adxIdeaStatDo != null && adxIdeaStatDo.getConfidence()) {

                if (ret.getCtr() == null || ret.getCtr() < adxIdeaStatDo.getCtr()) {
                    ret.setCtr(adxIdeaStatDo.getCtr());
                }

                if (ret.getProfitEcpm() == null || ret.getProfitEcpm() < adxIdeaStatDo.getProfitEcpm()) {
                    ret.setProfitEcpm(adxIdeaStatDo.getProfitEcpm());
                }

            }
        }


        return ret;
    }


    /**
     * 置信、指标计算
     *
     * @param adxIdeaStatDo
     */
    private static void ideaStatCompute(AdxIdeaStatDo adxIdeaStatDo) {

        if (adxIdeaStatDo != null) {

            boolean isConfidence = false;

            // 判断是否置信
            if (DataUtil.isLarger(adxIdeaStatDo.getBid(), 50L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getBidSuc(), 20L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getExp(), 10L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getClick(), 2L) &&
                    DataUtil.isLarger(adxIdeaStatDo.getAdxConsume(), 50 * ADX_MULTIPLIER) &&
                    DataUtil.isLarger(adxIdeaStatDo.getAdvertConsume(), 50L)) {
                isConfidence = true;
            }


            if (isConfidence) {
                // roi
                Double roi = MathUtil.division(adxIdeaStatDo.getAdvertConsume(), adxIdeaStatDo.getAdxConsume() / ADX_MULTIPLIER, 3);

                // ctr
                Double ctr = MathUtil.division(adxIdeaStatDo.getClick(), adxIdeaStatDo.getExp(), 4);

                // 千次竞价收益
                Double profit = adxIdeaStatDo.getAdvertConsume() / 1.15 - adxIdeaStatDo.getAdxConsume() / ADX_MULTIPLIER;
                Double pEcpm = MathUtil.division(profit * 1000 * roiPenaltyFactor(roi), adxIdeaStatDo.getBid(), 4);


                adxIdeaStatDo.setRoi(roi);
                adxIdeaStatDo.setCtr(ctr);
                adxIdeaStatDo.setProfitEcpm(pEcpm);

            }

            adxIdeaStatDo.setConfidence(isConfidence);


        }
    }


    /**
     * 惩罚因子，当ROI过大时生效
     * 分段函数
     * 0~1.5   1
     * 1.5~2   0.7
     * 2~      0.5
     *
     * @param roi
     * @return
     */
    private static double roiPenaltyFactor(Double roi) {
        double ret = 1;

        if (roi != null) {

            if (roi >= 1.5 && roi < 2.0) {
                ret = (1.5 + (roi - 1.5) * 0.7) / roi;
            }
            if (roi > 2.0) {
                ret = (1.5 + (2.0 - 1.5) * 0.7 + (roi - 2.0) * 0.5) / roi;
            }
        }

        return ret;
    }
}
